<?php
ob_start();
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Add detailed session debugging
error_log("=== VERIFICATION PAGE LOADED ===");
error_log("Session ID: " . session_id());
error_log("Session data: " . print_r($_SESSION, true));

require '../config/conn.php';
require '../vendor/autoload.php';

// Use UTC for consistency across servers
date_default_timezone_set('UTC');

// Ensure MySQL uses same timezone
$conn->query("SET time_zone = '+00:00'");

// Rate limiting
if (!isset($_SESSION['admin_verify_attempts'])) {
    $_SESSION['admin_verify_attempts'] = 0;
    $_SESSION['admin_first_attempt_time'] = time();
}



if (!isset($_SESSION['admin_2fa_id'])) {
    error_log("No admin_2fa_id found, redirecting to login");
    header("Location: admin-login.php");
    exit();
}

$error = '';
$success = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    error_log("=== POST REQUEST RECEIVED ===");
    
    // Verify code submission
    if (isset($_POST['verify'])) {
        $code = trim($_POST['code'] ?? '');
        error_log("Verification attempt with code: $code");
        error_log("Admin ID in session: " . $_SESSION['admin_2fa_id']);
        
        if (empty($code)) {
            $error = "Verification code is required";
            error_log("Error: Empty code");
        } elseif (!preg_match('/^\d{6}$/', $code)) {
            $error = "Invalid code format. Please enter a 6-digit number.";
            error_log("Error: Invalid code format: $code");
        } else {
            // Debug: Check what codes exist for this admin
            $debug_stmt = $conn->prepare("
                SELECT id, code, expires_at, is_used, created_at 
                FROM admin_verification_codes 
                WHERE admin_id = ? 
                ORDER BY created_at DESC 
                LIMIT 5
            ");
            $debug_stmt->bind_param("i", $_SESSION['admin_2fa_id']);
            $debug_stmt->execute();
            $debug_result = $debug_stmt->get_result();
            
            error_log("=== DEBUG: AVAILABLE CODES ===");
            $codes_found = false;
            while($debug_row = $debug_result->fetch_assoc()) {
                $codes_found = true;
                error_log("Code: " . $debug_row['code'] . 
                         " | Expires: " . $debug_row['expires_at'] . 
                         " | Used: " . $debug_row['is_used'] .
                         " | Created: " . $debug_row['created_at']);
            }
            
            if (!$codes_found) {
                error_log("No codes found for admin ID: " . $_SESSION['admin_2fa_id']);
            }
            
            // Main verification query with UTC_TIMESTAMP
            $stmt = $conn->prepare("
                SELECT vc.*, a.* 
                FROM admin_verification_codes vc
                JOIN admins a ON vc.admin_id = a.id
                WHERE vc.admin_id = ? 
                AND vc.code = ? 
                AND vc.expires_at > UTC_TIMESTAMP()
                AND (vc.is_used IS NULL OR vc.is_used = 0)
                ORDER BY vc.created_at DESC 
                LIMIT 1
            ");
            
            if (!$stmt) {
                error_log("Prepare failed: " . $conn->error);
                $error = "Database error. Please try again.";
            } else {
                $stmt->bind_param("is", $_SESSION['admin_2fa_id'], $code);
                
                if (!$stmt->execute()) {
                    error_log("Execute failed: " . $stmt->error);
                    $error = "Verification failed. Please try again.";
                } else {
                    $result = $stmt->get_result();
                    error_log("Query returned " . $result->num_rows . " rows");
                    
                    if ($result->num_rows === 1) {
                        $row = $result->fetch_assoc();
                        error_log("Code verified successfully for admin: " . $row['email']);
                        error_log("Code expires at: " . $row['expires_at']);
                        
                        // Mark code as used
                        $update_stmt = $conn->prepare("
                            UPDATE admin_verification_codes 
                            SET is_used = 1 
                            WHERE id = ?
                        ");
                        
                        if ($update_stmt) {
                            $update_stmt->bind_param("i", $row['id']);
                            
                            if ($update_stmt->execute()) {
                                // Regenerate session ID for security
                                session_regenerate_id(true);
                                
                                // Set admin session variables
                                $_SESSION['admin_id'] = $row['id'];
                                $_SESSION['admin_username'] = $row['full_name'] ?? $row['username'];
                                $_SESSION['admin_email'] = $row['email'];
                                $_SESSION['admin_role'] = $row['role'] ?? 'admin';
                                $_SESSION['admin_logged_in'] = true;
                                $_SESSION['admin_last_activity'] = time();
                                
                                error_log("Session variables set for admin: " . $_SESSION['admin_email']);
                                
                                // Remember me functionality
                                if (isset($_SESSION['admin_2fa_remember_me']) && $_SESSION['admin_2fa_remember_me']) {
                                    error_log("Remember me enabled");
                                    $token = bin2hex(random_bytes(32));
                                    $expiry = date('Y-m-d H:i:s', strtotime('+30 days'));
                                    
                                    $token_stmt = $conn->prepare("
                                        INSERT INTO admin_remember_tokens 
                                        (admin_id, token, expires_at) 
                                        VALUES (?, ?, ?)
                                    ");
                                    
                                    if ($token_stmt) {
                                        $token_stmt->bind_param("iss", $row['id'], $token, $expiry);
                                        
                                        if ($token_stmt->execute()) {
                                            setcookie('admin_remember_token', $token, [
                                                'expires' => time() + (30 * 24 * 60 * 60),
                                                'path' => '/admin',
                                                'secure' => isset($_SERVER['HTTPS']),
                                                'httponly' => true,
                                                'samesite' => 'Strict'
                                            ]);
                                            error_log("Remember token set");
                                        }
                                    }
                                }
                                
                                // Record login history
                                $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
                                $device_info = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
                                
                                $login_stmt = $conn->prepare("
                                    INSERT INTO admin_login_history 
                                    (admin_id, ip_address, device_info) 
                                    VALUES (?, ?, ?)
                                ");
                                
                                if ($login_stmt) {
                                    $login_stmt->bind_param("iss", $row['id'], $ip_address, $device_info);
                                    $login_stmt->execute();
                                }
                                
                                // Update last login
                                $last_login_stmt = $conn->prepare("
                                    UPDATE admins 
                                    SET last_login = UTC_TIMESTAMP() 
                                    WHERE id = ?
                                ");
                                
                                if ($last_login_stmt) {
                                    $last_login_stmt->bind_param("i", $row['id']);
                                    $last_login_stmt->execute();
                                }
                                
                                // Clear 2FA session
                                unset($_SESSION['admin_2fa_id']);
                                unset($_SESSION['admin_2fa_remember_me']);
                                unset($_SESSION['admin_verify_attempts']);
                                
                                // Clear output buffer and redirect
                                ob_end_clean();
                                error_log("Redirecting to index.php");
                                header("Location: index.php");
                                exit();
                            } else {
                                error_log("Failed to mark code as used: " . $update_stmt->error);
                                $error = "System error during verification. Please try again.";
                            }
                        } else {
                            error_log("Failed to prepare update statement: " . $conn->error);
                            $error = "System error. Please try again.";
                        }
                    } else {
                        $_SESSION['admin_verify_attempts']++;
                        $attempts_left = 5 - $_SESSION['admin_verify_attempts'];
                        error_log("Invalid code. Attempts: " . $_SESSION['admin_verify_attempts']);
                        
                        if ($attempts_left > 0) {
                            $error = "Invalid verification code or code has expired. $attempts_left attempts remaining.";
                        } else {
                            $error = "Too many failed attempts. Please try again later.";
                        }
                    }
                }
            }
        }
    }
    
    // Resend code functionality
    if (isset($_POST['resend'])) {
        error_log("Resend code requested");
        
        // Generate a new 6-digit verification code
        $verification_code = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);
        $expires_at = gmdate('Y-m-d H:i:s', strtotime('+10 minutes'));
        
        error_log("New code generated: $verification_code, expires: $expires_at");
        
        // Store the new code in database
        $stmt = $conn->prepare("
            INSERT INTO admin_verification_codes 
            (admin_id, code, expires_at, ip_address) 
            VALUES (?, ?, ?, ?)
        ");
        
        if ($stmt) {
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
            $stmt->bind_param("isss", $_SESSION['admin_2fa_id'], $verification_code, $expires_at, $ip_address);
            
            if ($stmt->execute()) {
                error_log("New verification code stored in database");
                
                // Get admin details
                $user_stmt = $conn->prepare("
                    SELECT email, username, full_name 
                    FROM admins 
                    WHERE id = ?
                ");
                
                if ($user_stmt) {
                    $user_stmt->bind_param("i", $_SESSION['admin_2fa_id']);
                    $user_stmt->execute();
                    $user_result = $user_stmt->get_result();
                    
                    if ($user_result->num_rows === 1) {
                        $admin = $user_result->fetch_assoc();
                        
                        // Send the new verification code via email
                        try {
                            $mail = new PHPMailer\PHPMailer\PHPMailer(true);
                            
                            // Enable verbose debug output on live server
                            // $mail->SMTPDebug = 2;
                            // $mail->Debugoutput = function($str, $level) {
                            //     error_log("MAIL DEBUG: $str");
                            // };
                            
                            // Server settings - UPDATE THESE FOR LIVE SERVER
                            $mail->isSMTP();
                            $mail->Host       = 'smtp.gmail.com'; // Change as needed
                            $mail->SMTPAuth   = true;
                            $mail->Username   = 'your-email@gmail.com'; // UPDATE
                            $mail->Password   = 'your-app-password';    // UPDATE
                            $mail->SMTPSecure = PHPMailer\PHPMailer::ENCRYPTION_SMTPS;
                            $mail->Port       = 465;
                            
                            // Recipients
                            $mail->setFrom('noreply@yourdomain.com', 'Your System Name');
                            $mail->addAddress($admin['email'], $admin['full_name'] ?? $admin['username']);
                            
                            // Content
                            $mail->isHTML(true);
                            $mail->Subject = 'New Admin Login Verification Code';
                            
                            $mail->Body = "
                            <div style='max-width: 600px; margin: 0 auto; border: 1px solid #e0e0e0; border-radius: 8px; overflow: hidden;'>
                                <div style='background-color: #FBBB01; padding: 25px; text-align: center;'>
                                    <h1 style='color: white; margin: 0; font-size: 24px;'>New Admin Verification Code</h1>
                                </div>
                                
                                <div style='padding: 25px;'>
                                    <h2 style='color: #FBBB01; margin-top: 0;'>Hello Admin {$admin['username']},</h2>
                                    
                                    <p style='font-size: 16px;'>Your new admin login verification code is:</p>
                                    
                                    <div style='text-align: center; margin: 25px 0;'>
                                        <div style='display: inline-block; padding: 15px 30px; background-color: #f5f5f5; border: 2px dashed #FBBB01; border-radius: 5px; font-size: 24px; font-weight: bold; letter-spacing: 5px;'>
                                            $verification_code
                                        </div>
                                    </div>
                                    
                                    <p style='font-size: 14px; color: #666;'>This code will expire in 10 minutes. Please do not share this code with anyone.</p>
                                    
                                    <p style='font-size: 12px; color: #999; margin-top: 20px;'>
                                        <strong>Note:</strong> If you didn't request this code, please ignore this email and contact support immediately.
                                    </p>
                                </div>
                                
                                <div style='background-color: #f5f5f5; padding: 15px; text-align: center; font-size: 12px; color: #666;'>
                                    <p>© " . date('Y') . " Your System Name. All rights reserved.</p>
                                </div>
                            </div>
                            ";
                            
                            $mail->AltBody = "Hello Admin {$admin['username']},\n\n" .
                                "Your new admin login verification code is: $verification_code\n\n" .
                                "This code will expire in 10 minutes. Please do not share this code with anyone.\n\n" .
                                "If you didn't request this code, please ignore this email and contact support immediately.\n\n" .
                                "Best regards,\n" .
                                "Your System Name";
                            
                            if ($mail->send()) {
                                error_log("Verification email sent to: " . $admin['email']);
                                $_SESSION['admin_last_code_sent'] = time();
                                $_SESSION['flash_success'] = "A new verification code has been sent to your admin email.";
                                
                                // Redirect with success message
                                header("Location: admin-verify-code.php?resend=success");
                                exit();
                            }
                        } catch (Exception $e) {
                            error_log("Email sending failed: " . $mail->ErrorInfo);
                            // Don't show email error to user, just log it
                            $_SESSION['flash_success'] = "A new verification code has been generated.";
                        }
                    }
                }
                
                // If email fails, still show success message
                $_SESSION['flash_success'] = "A new verification code has been generated.";
                
            } else {
                error_log("Failed to insert new code: " . $stmt->error);
                $_SESSION['flash_error'] = "Failed to generate new code. Please try again.";
            }
        }
        
        // Redirect to avoid form resubmission
        header("Location: admin-verify-code.php");
        exit();
    }
}

// Check for flash messages
if (isset($_SESSION['flash_success'])) {
    $success = $_SESSION['flash_success'];
    unset($_SESSION['flash_success']);
}

if (isset($_SESSION['flash_error'])) {
    $error = $_SESSION['flash_error'];
    unset($_SESSION['flash_error']);
}

// Check for resend parameter
if (isset($_GET['resend']) && $_GET['resend'] === 'success') {
    $success = "A new verification code has been sent to your email.";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Verification | Your System Name</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body {
            font-family: 'Poppins', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .verify-container {
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            width: 100%;
            max-width: 440px;
            overflow: hidden;
        }
        .verify-header {
            background: linear-gradient(135deg, #FBBB01 0%, #FF9900 100%);
            padding: 40px 30px;
            text-align: center;
            color: white;
        }
        .verify-icon {
            width: 80px;
            height: 80px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            font-size: 36px;
        }
        .verify-body {
            padding: 40px;
        }
        .code-input {
            letter-spacing: 8px;
            font-size: 32px;
            font-weight: 600;
            text-align: center;
        }
        .timer {
            font-size: 14px;
            color: #666;
            text-align: center;
            margin-top: 10px;
        }
        .attempts-warning {
            font-size: 13px;
            color: #dc2626;
            text-align: center;
            margin-top: 5px;
        }
        .btn-verify {
            background: linear-gradient(135deg, #FBBB01 0%, #FF9900 100%);
            color: white;
            padding: 14px;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s ease;
            border: none;
            width: 100%;
        }
        .btn-verify:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(251, 187, 1, 0.3);
        }
        .btn-verify:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        .resend-btn {
            color: #FBBB01;
            background: none;
            border: none;
            cursor: pointer;
            font-weight: 500;
        }
        .resend-btn:hover {
            text-decoration: underline;
        }
        .resend-btn:disabled {
            color: #999;
            cursor: not-allowed;
        }
    </style>
</head>
<body>
    <div class="verify-container">
        <div class="verify-header">
            <div class="verify-icon">
                <i class="fas fa-shield-alt"></i>
            </div>
            <h1 class="text-2xl font-bold mb-2">Admin Verification</h1>
            <p class="opacity-90">Enter the 6-digit code sent to your admin email</p>
        </div>
        
        <div class="verify-body">
            <?php if ($error): ?>
                <div class="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6 flex items-start">
                    <i class="fas fa-exclamation-circle mt-1 mr-3"></i>
                    <div>
                        <p class="font-medium">Verification Failed</p>
                        <p class="text-sm mt-1"><?php echo htmlspecialchars($error); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-6 flex items-start">
                    <i class="fas fa-check-circle mt-1 mr-3"></i>
                    <div>
                        <p class="font-medium">Success</p>
                        <p class="text-sm mt-1"><?php echo htmlspecialchars($success); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php 
            $attempts_left = 5 - ($_SESSION['admin_verify_attempts'] ?? 0);
            if ($attempts_left <= 2 && $attempts_left > 0): ?>
                <div class="bg-yellow-50 border border-yellow-200 text-yellow-700 px-4 py-3 rounded-lg mb-6 text-sm">
                    <i class="fas fa-exclamation-triangle mr-2"></i>
                    <?php echo $attempts_left; ?> attempt<?php echo $attempts_left == 1 ? '' : 's'; ?> remaining before lockout
                </div>
            <?php endif; ?>
            
            <form method="POST" action="" id="verifyForm">
                <div class="mb-6">
                    <label for="code" class="block text-gray-700 font-medium mb-3">Verification Code</label>
                    <input type="text" 
                           id="code" 
                           name="code" 
                           class="w-full px-4 py-4 border-2 border-gray-200 rounded-xl focus:outline-none focus:border-[#FBBB01] focus:ring-2 focus:ring-[#FBBB01]/20 code-input"
                           maxlength="6" 
                           pattern="\d{6}" 
                           placeholder="000000"
                           required 
                           autofocus
                           autocomplete="off">
                    <div class="timer" id="timer">
                        Code expires in: <span id="expiryCountdown">10:00</span>
                    </div>
                    <?php if ($attempts_left > 0): ?>
                        <div class="attempts-warning">
                            <?php echo $attempts_left; ?> attempt<?php echo $attempts_left == 1 ? '' : 's'; ?> remaining
                        </div>
                    <?php endif; ?>
                </div>
                
                <button type="submit" 
                        name="verify" 
                        class="btn-verify mb-4"
                        id="verifyBtn">
                    <i class="fas fa-sign-in-alt mr-2"></i> Verify & Continue
                </button>
                
                <div class="text-center pt-6 border-t border-gray-100">
                    <p class="text-gray-600 text-sm mb-4">
                        Didn't receive a code? 
                        <?php 
                        $last_sent = $_SESSION['admin_last_code_sent'] ?? 0;
                        $can_resend = (time() - $last_sent) > 60; // 60 seconds cooldown
                        ?>
                        <button type="submit" 
                                name="resend" 
                                class="resend-btn ml-1" 
                                id="resendBtn"
                                <?php echo !$can_resend ? 'disabled' : ''; ?>>
                            <i class="fas fa-redo mr-1"></i> 
                            <span id="resendText">
                                <?php echo $can_resend ? 'Resend Code' : 'Wait 60 seconds'; ?>
                            </span>
                        </button>
                    </p>
                    
                    <p class="text-gray-500 text-sm">
                        <a href="admin-login.php" class="text-[#FBBB01] hover:text-[#FF9900] hover:underline">
                            <i class="fas fa-arrow-left mr-1"></i> Back to Login
                        </a>
                    </p>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const codeInput = document.getElementById('code');
        const resendBtn = document.getElementById('resendBtn');
        const resendText = document.getElementById('resendText');
        const expiryCountdown = document.getElementById('expiryCountdown');
        
        // Auto-format code input
        codeInput.addEventListener('input', function(e) {
            this.value = this.value.replace(/[^0-9]/g, '').substring(0, 6);
            
            // Auto-submit when 6 digits entered
            if (this.value.length === 6) {
                document.getElementById('verifyBtn').focus();
            }
        });
        
        // Countdown for resend button
        let lastSent = <?php echo $last_sent; ?>;
        let canResend = <?php echo $can_resend ? 'true' : 'false'; ?>;
        
        if (!canResend) {
            let cooldown = 60 - Math.floor((Date.now()/1000) - lastSent);
            updateResendTimer(cooldown);
            
            let timer = setInterval(function() {
                cooldown--;
                if (cooldown <= 0) {
                    clearInterval(timer);
                    resendBtn.disabled = false;
                    resendText.textContent = 'Resend Code';
                } else {
                    updateResendTimer(cooldown);
                }
            }, 1000);
        }
        
        function updateResendTimer(seconds) {
            resendText.textContent = 'Wait ' + seconds + 's';
        }
        
        // Countdown for code expiry (10 minutes)
        let expiryTime = 10 * 60; // 10 minutes in seconds
        updateExpiryTimer(expiryTime);
        
        let expiryTimer = setInterval(function() {
            expiryTime--;
            if (expiryTime <= 0) {
                clearInterval(expiryTimer);
                expiryCountdown.textContent = 'Expired';
                expiryCountdown.style.color = '#dc2626';
            } else {
                updateExpiryTimer(expiryTime);
            }
        }, 1000);
        
        function updateExpiryTimer(seconds) {
            let minutes = Math.floor(seconds / 60);
            let secs = seconds % 60;
            expiryCountdown.textContent = minutes.toString().padStart(2, '0') + ':' + 
                                          secs.toString().padStart(2, '0');
            
            if (seconds < 300) { // Less than 5 minutes
                expiryCountdown.style.color = '#dc2626';
            } else if (seconds < 600) { // Less than 10 minutes
                expiryCountdown.style.color = '#f59e0b';
            }
        }
        
        // Prevent form resubmission on refresh
        if (window.history.replaceState) {
            window.history.replaceState(null, null, window.location.href);
        }
        
        // Focus on code input
        codeInput.focus();
        
        // Auto-select all text when clicking input
        codeInput.addEventListener('click', function() {
            this.select();
        });
    });
    </script>
</body>
</html>