<?php
// bulk-approval.php
session_start();
require '../config/conn.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: admin-login.php");
    exit();
}

$date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');
$admin_id = $_SESSION['admin_id'] ?? 1;

// Check if daily table exists
$table_check = mysqli_query($conn, "SHOW TABLES LIKE 'vendor_daily_transactions'");
$daily_table_exists = mysqli_num_rows($table_check) > 0;

if (!$daily_table_exists) {
    $_SESSION['error'] = "Daily transactions table not found. Please run the update script first.";
    header("Location: update_daily_balances.php");
    exit();
}

// Process bulk approval
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $selected_vendors = $_POST['vendors'] ?? [];
    $notes = mysqli_real_escape_string($conn, $_POST['notes'] ?? '');
    
    if (empty($selected_vendors)) {
        $_SESSION['error'] = "Please select at least one vendor to approve.";
    } else {
        mysqli_begin_transaction($conn);
        
        try {
            $total_approved = 0;
            $approved_count = 0;
            
            foreach ($selected_vendors as $vendor_id) {
                $vendor_id = intval($vendor_id);
                
                // Get vendor daily transaction
                $daily_query = "SELECT * FROM vendor_daily_transactions 
                               WHERE vendor_id = $vendor_id AND transaction_date = '$date'";
                $daily_result = mysqli_query($conn, $daily_query);
                $daily_transaction = mysqli_fetch_assoc($daily_result);
                
                if (!$daily_transaction || $daily_transaction['pending_amount'] <= 0) {
                    continue;
                }
                
                $approve_amount = $daily_transaction['pending_amount'];
                $current_approved = $daily_transaction['approved_amount'];
                $daily_total = $daily_transaction['total_amount'];
                
                // Update vendor wallet
                $wallet_query = "SELECT * FROM vendor_wallets WHERE vendor_id = $vendor_id";
                $wallet_result = mysqli_query($conn, $wallet_query);
                $wallet = mysqli_fetch_assoc($wallet_result);
                
                if ($wallet) {
                    $new_balance = $wallet['balance'] + $approve_amount;
                    $new_pending = $wallet['pending_balance'] - $approve_amount;
                    $new_total_collected = $wallet['total_collected'] + $approve_amount;
                    
                    $update_wallet = "UPDATE vendor_wallets 
                                     SET balance = $new_balance,
                                         pending_balance = $new_pending,
                                         total_collected = $new_total_collected,
                                         updated_at = NOW()
                                     WHERE vendor_id = $vendor_id";
                    mysqli_query($conn, $update_wallet);
                    
                    // Create vendor wallet transaction
                    $vendor_transaction_ref = 'BULK-APPROVAL-' . date('YmdHis') . '-' . $vendor_id;
                    $vendor_wallet_transaction = "
                        INSERT INTO vendor_wallet_transactions 
                        (vendor_id, amount, transaction_type, reference, status, 
                         balance_before, balance_after, description, pending_before, pending_after, notes)
                        VALUES (
                            $vendor_id, 
                            $approve_amount, 
                            'credit', 
                            '$vendor_transaction_ref', 
                            'completed',
                            {$wallet['balance']},
                            $new_balance,
                            'Bulk approval for $date',
                            {$wallet['pending_balance']},
                            $new_pending,
                            '$notes'
                        )
                    ";
                    mysqli_query($conn, $vendor_wallet_transaction);
                }
                
                // Update daily transaction
                $new_approved = $current_approved + $approve_amount;
                $remaining_amount = $daily_total - $new_approved;
                $status = 'fully_approved';
                
                $update_daily = "
                    UPDATE vendor_daily_transactions 
                    SET approved_amount = $new_approved,
                        pending_amount = 0,
                        remaining_amount = $remaining_amount,
                        status = '$status',
                        updated_at = NOW()
                    WHERE vendor_id = $vendor_id AND transaction_date = '$date'
                ";
                mysqli_query($conn, $update_daily);
                
                // Create approval log
                $approval_ref = 'BULK-APPR-' . date('YmdHis') . '-' . $vendor_id;
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $user_agent = mysqli_real_escape_string($conn, $_SERVER['HTTP_USER_AGENT']);
                
                $approval_log = "
                    INSERT INTO vendor_approval_logs 
                    (vendor_id, admin_id, approval_date, approved_amount, daily_total, 
                     previous_approved, new_approved, remaining_amount, notes, ip_address, user_agent)
                    VALUES (
                        $vendor_id,
                        $admin_id,
                        '$date',
                        $approve_amount,
                        $daily_total,
                        $current_approved,
                        $new_approved,
                        $remaining_amount,
                        '$notes',
                        '$ip_address',
                        '$user_agent'
                    )
                ";
                mysqli_query($conn, $approval_log);
                
                $total_approved += $approve_amount;
                $approved_count++;
            }
            
            // Update admin wallet
            $admin_wallet_query = "SELECT * FROM admin_wallets LIMIT 1";
            $admin_wallet_result = mysqli_query($conn, $admin_wallet_query);
            
            if ($admin_wallet_result && mysqli_num_rows($admin_wallet_result) > 0) {
                $admin_wallet = mysqli_fetch_assoc($admin_wallet_result);
                $admin_wallet_id = $admin_wallet['id'];
                $new_admin_balance = $admin_wallet['balance'] + $total_approved;
                
                $update_admin_wallet = "UPDATE admin_wallets 
                                       SET balance = $new_admin_balance, 
                                           updated_at = NOW() 
                                       WHERE id = $admin_wallet_id";
                mysqli_query($conn, $update_admin_wallet);
                
                // Create admin wallet transaction
                $admin_transaction_ref = 'BULK-VENDOR-APPROVAL-' . date('YmdHis');
                $admin_wallet_transaction = "
                    INSERT INTO admin_wallet_transactions 
                    (admin_wallet_id, amount, transaction_type, reference, description, 
                     balance_before, balance_after)
                    VALUES (
                        $admin_wallet_id,
                        $total_approved,
                        'credit',
                        '$admin_transaction_ref',
                        'Bulk approval for $approved_count vendors',
                        {$admin_wallet['balance']},
                        $new_admin_balance
                    )
                ";
                mysqli_query($conn, $admin_wallet_transaction);
            }
            
            mysqli_commit($conn);
            
            $_SESSION['success'] = "Successfully approved ₦" . number_format($total_approved, 2) . 
                                  " for $approved_count vendors.";
            header("Location: vendor-approval.php?date=$date");
            exit();
            
        } catch (Exception $e) {
            mysqli_rollback($conn);
            $_SESSION['error'] = "Bulk approval failed: " . $e->getMessage();
            header("Location: bulk-approval.php?date=$date");
            exit();
        }
    }
}

// Get vendors with pending amounts for the selected date
$vendors_query = "
    SELECT 
        vdt.*,
        v.business_name,
        v.email,
        vw.balance as wallet_balance,
        vw.pending_balance
    FROM vendor_daily_transactions vdt
    LEFT JOIN vendors v ON vdt.vendor_id = v.id
    LEFT JOIN vendor_wallets vw ON vdt.vendor_id = vw.vendor_id
    WHERE vdt.transaction_date = '$date' 
        AND vdt.pending_amount > 0
        AND vdt.status != 'fully_approved'
    ORDER BY vdt.pending_amount DESC
";

$vendors_result = mysqli_query($conn, $vendors_query);
$total_pending = 0;
$vendor_count = mysqli_num_rows($vendors_result);

while ($vendor = mysqli_fetch_assoc($vendors_result)) {
    $total_pending += $vendor['pending_amount'];
}

include 'include/header.php';
?>

<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto">
    <div class="max-w-7xl mx-auto">
        <!-- Header -->
        <div class="mb-6">
            <a href="vendor-approval.php?date=<?= $date ?>" 
               class="inline-flex items-center text-primary-500 hover:text-primary-600 mb-4">
                <i class="fas fa-arrow-left mr-2"></i> Back to Approval Page
            </a>
            
            <div class="flex flex-col md:flex-row md:items-center justify-between">
                <div>
                    <h1 class="text-2xl font-bold text-gray-900 dark:text-white">Bulk Approval</h1>
                    <p class="text-gray-600 dark:text-gray-400">
                        Approve multiple vendors at once for <?= date('F j, Y', strtotime($date)) ?>
                    </p>
                </div>
                
                <div class="mt-4 md:mt-0">
                    <span class="px-3 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300 rounded-full text-sm">
                        Total Pending: ₦<?= number_format($total_pending, 2) ?>
                    </span>
                </div>
            </div>
        </div>

        <!-- Statistics -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-blue-50 dark:bg-blue-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-store text-blue-500 dark:text-blue-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Eligible Vendors</p>
                        <p class="text-2xl font-bold text-gray-900 dark:text-white">
                            <?= $vendor_count ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-yellow-50 dark:bg-yellow-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-clock text-yellow-500 dark:text-yellow-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Total Pending</p>
                        <p class="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                            ₦<?= number_format($total_pending, 2) ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-green-50 dark:bg-green-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-check-circle text-green-500 dark:text-green-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Average per Vendor</p>
                        <p class="text-2xl font-bold text-green-600 dark:text-green-400">
                            ₦<?= number_format($vendor_count > 0 ? $total_pending / $vendor_count : 0, 2) ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Bulk Approval Form -->
        <form method="POST" action="bulk-approval.php">
            <input type="hidden" name="date" value="<?= $date ?>">
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 overflow-hidden mb-6">
                <div class="px-6 py-4 border-b border-gray-100 dark:border-gray-700 bg-gray-50 dark:bg-gray-900">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-4">
                            <div class="flex items-center">
                                <input type="checkbox" id="selectAll" 
                                       class="w-4 h-4 text-primary-500 rounded focus:ring-primary-400">
                                <label for="selectAll" class="ml-2 text-sm font-medium text-gray-700 dark:text-gray-300">
                                    Select All
                                </label>
                            </div>
                            <span class="text-sm text-gray-500 dark:text-gray-400">
                                <span id="selectedCount">0</span> of <?= $vendor_count ?> selected
                            </span>
                        </div>
                        <div class="text-sm font-medium text-yellow-600 dark:text-yellow-400">
                            Total: <span id="selectedTotal">₦0.00</span>
                        </div>
                    </div>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                        <thead class="bg-gray-50 dark:bg-gray-800">
                            <tr>
                                <th class="w-12 px-4 py-3"></th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Vendor</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Daily Total</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Approved</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Pending</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Wallet Balance</th>
                                <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400">Status</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                            <?php 
                            mysqli_data_seek($vendors_result, 0); // Reset result pointer
                            if (mysqli_num_rows($vendors_result) > 0): 
                                while ($vendor = mysqli_fetch_assoc($vendors_result)): 
                            ?>
                                <tr class="hover:bg-gray-50 dark:hover:bg-gray-700">
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <input type="checkbox" name="vendors[]" value="<?= $vendor['vendor_id'] ?>" 
                                               class="vendor-checkbox w-4 h-4 text-primary-500 rounded focus:ring-primary-400"
                                               data-amount="<?= $vendor['pending_amount'] ?>">
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <div>
                                            <div class="text-sm font-medium text-gray-900 dark:text-white">
                                                <?= htmlspecialchars($vendor['business_name']) ?>
                                            </div>
                                            <div class="text-sm text-gray-500 dark:text-gray-400">
                                                <?= htmlspecialchars($vendor['email']) ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900 dark:text-white">
                                            ₦<?= number_format($vendor['total_amount'], 2) ?>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <div class="text-sm text-green-600 dark:text-green-400">
                                            ₦<?= number_format($vendor['approved_amount'], 2) ?>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-yellow-600 dark:text-yellow-400">
                                            ₦<?= number_format($vendor['pending_amount'], 2) ?>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <div class="text-sm">
                                            <div class="text-gray-900 dark:text-white">
                                                ₦<?= number_format($vendor['wallet_balance'] ?? 0, 2) ?>
                                            </div>
                                            <div class="text-xs text-yellow-600 dark:text-yellow-400">
                                                Pending: ₦<?= number_format($vendor['pending_balance'] ?? 0, 2) ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-4 py-4 whitespace-nowrap">
                                        <?php
                                        $status_class = $vendor['status'] == 'pending' ? 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' :
                                                      'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
                                        ?>
                                        <span class="px-2 py-1 rounded-full text-xs font-medium <?= $status_class ?>">
                                            <?= ucfirst(str_replace('_', ' ', $vendor['status'])) ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php 
                                endwhile;
                            else: 
                            ?>
                                <tr>
                                    <td colspan="7" class="px-6 py-8 text-center text-gray-500 dark:text-gray-400">
                                        No vendors with pending amounts for <?= date('F j, Y', strtotime($date)) ?>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Notes and Submit Section -->
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <div>
                        <label for="notes" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Notes for All Approvals (Optional)
                        </label>
                        <textarea id="notes" name="notes" rows="4" 
                                  class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 dark:bg-gray-700 dark:text-white"
                                  placeholder="Add notes that will apply to all selected approvals"></textarea>
                    </div>
                    
                    <div>
                        <div class="bg-blue-50 dark:bg-blue-900/30 border border-blue-200 dark:border-blue-800 rounded-lg p-4 h-full">
                            <h4 class="font-medium text-blue-800 dark:text-blue-300 mb-3">Approval Summary</h4>
                            <div class="space-y-2">
                                <div class="flex justify-between">
                                    <span class="text-blue-600 dark:text-blue-400">Selected Vendors:</span>
                                    <span class="font-medium text-gray-900 dark:text-white" id="summaryCount">0</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-blue-600 dark:text-blue-400">Total Amount:</span>
                                    <span class="font-medium text-green-600 dark:text-green-400" id="summaryTotal">₦0.00</span>
                                </div>
                                <div class="flex justify-between">
                                    <span class="text-blue-600 dark:text-blue-400">Average per Vendor:</span>
                                    <span class="font-medium text-gray-900 dark:text-white" id="summaryAverage">₦0.00</span>
                                </div>
                            </div>
                            
                            <div class="mt-6">
                                <div class="flex items-center justify-between">
                                    <button type="submit" 
                                            class="px-6 py-3 bg-green-500 hover:bg-green-600 text-white rounded-md font-medium disabled:opacity-50 disabled:cursor-not-allowed"
                                            id="approveButton" disabled>
                                        <i class="fas fa-check-double mr-2"></i> Approve Selected
                                    </button>
                                    <a href="vendor-approval.php?date=<?= $date ?>" 
                                       class="px-6 py-3 bg-gray-200 dark:bg-gray-700 hover:bg-gray-300 dark:hover:bg-gray-600 text-gray-800 dark:text-white rounded-md font-medium">
                                        Cancel
                                    </a>
                                </div>
                                <p class="text-xs text-gray-500 dark:text-gray-400 mt-2">
                                    This will approve the full pending amount for each selected vendor.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</main>

<script>
// Select All functionality
const selectAllCheckbox = document.getElementById('selectAll');
const vendorCheckboxes = document.querySelectorAll('.vendor-checkbox');
const selectedCount = document.getElementById('selectedCount');
const selectedTotal = document.getElementById('selectedTotal');
const summaryCount = document.getElementById('summaryCount');
const summaryTotal = document.getElementById('summaryTotal');
const summaryAverage = document.getElementById('summaryAverage');
const approveButton = document.getElementById('approveButton');

function updateSelectionSummary() {
    let count = 0;
    let total = 0;
    
    vendorCheckboxes.forEach(checkbox => {
        if (checkbox.checked) {
            count++;
            total += parseFloat(checkbox.dataset.amount);
        }
    });
    
    // Update counters
    selectedCount.textContent = count;
    summaryCount.textContent = count;
    
    // Update totals
    const formattedTotal = '₦' + total.toFixed(2);
    selectedTotal.textContent = formattedTotal;
    summaryTotal.textContent = formattedTotal;
    
    // Update average
    const average = count > 0 ? total / count : 0;
    summaryAverage.textContent = '₦' + average.toFixed(2);
    
    // Update select all checkbox state
    selectAllCheckbox.checked = count > 0 && count === vendorCheckboxes.length;
    selectAllCheckbox.indeterminate = count > 0 && count < vendorCheckboxes.length;
    
    // Enable/disable approve button
    approveButton.disabled = count === 0;
}

// Select All event
selectAllCheckbox.addEventListener('change', function() {
    const isChecked = this.checked;
    vendorCheckboxes.forEach(checkbox => {
        checkbox.checked = isChecked;
    });
    updateSelectionSummary();
});

// Individual checkbox events
vendorCheckboxes.forEach(checkbox => {
    checkbox.addEventListener('change', updateSelectionSummary);
});

// Initialize summary
updateSelectionSummary();
</script>

<?php include 'include/footer.php'; ?>