<?php
// process-approval.php
session_start();
require '../config/conn.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: admin-login.php");
    exit();
}

$admin_id = $_SESSION['admin_id'] ?? 1; // Default to admin ID 1 if not set

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $vendor_id = intval($_POST['vendor_id']);
    $approve_amount = floatval($_POST['approve_amount']);
    $transaction_date = $_POST['transaction_date'];
    $notes = mysqli_real_escape_string($conn, $_POST['notes'] ?? '');
    
    // Validate amount
    if ($approve_amount <= 0) {
        $_SESSION['error'] = "Please enter a valid amount to approve.";
        header("Location: vendor-approval.php?date=$transaction_date");
        exit();
    }
    
    // Start transaction
    mysqli_begin_transaction($conn);
    
    try {
        // 1. Get daily transaction details
        $daily_query = "SELECT * FROM vendor_daily_transactions 
                       WHERE vendor_id = $vendor_id AND transaction_date = '$transaction_date'";
        $daily_result = mysqli_query($conn, $daily_query);
        $daily_transaction = mysqli_fetch_assoc($daily_result);
        
        if (!$daily_transaction) {
            throw new Exception("Daily transaction not found for this vendor and date.");
        }
        
        $current_approved = $daily_transaction['approved_amount'];
        $daily_total = $daily_transaction['total_amount'];
        $current_pending = $daily_transaction['pending_amount'];
        
        // Validate approval amount doesn't exceed pending
        if ($approve_amount > $current_pending) {
            throw new Exception("Approval amount exceeds pending amount.");
        }
        
        // 2. Update vendor wallet (move from pending to approved balance)
        // Get current wallet balances
        $wallet_query = "SELECT * FROM vendor_wallets WHERE vendor_id = $vendor_id";
        $wallet_result = mysqli_query($conn, $wallet_query);
        $wallet = mysqli_fetch_assoc($wallet_result);
        
        if (!$wallet) {
            // Create wallet if doesn't exist
            $create_wallet = "INSERT INTO vendor_wallets (vendor_id, balance, pending_balance, total_collected) 
                             VALUES ($vendor_id, 0, 0, 0)";
            mysqli_query($conn, $create_wallet);
            
            $wallet = [
                'balance' => 0,
                'pending_balance' => 0,
                'total_collected' => 0
            ];
        }
        
        $new_balance = $wallet['balance'] + $approve_amount;
        $new_pending = $wallet['pending_balance'] - $approve_amount;
        $new_total_collected = $wallet['total_collected'] + $approve_amount;
        
        // Update vendor wallet
        $update_wallet = "UPDATE vendor_wallets 
                         SET balance = $new_balance,
                             pending_balance = $new_pending,
                             total_collected = $new_total_collected,
                             updated_at = NOW()
                         WHERE vendor_id = $vendor_id";
        mysqli_query($conn, $update_wallet);
        
        // 3. Create vendor wallet transaction record
        $vendor_transaction_ref = 'APPROVAL-' . date('YmdHis') . '-' . $vendor_id;
        $vendor_wallet_transaction = "
            INSERT INTO vendor_wallet_transactions 
            (vendor_id, amount, transaction_type, reference, status, 
             balance_before, balance_after, description, pending_before, pending_after, notes)
            VALUES (
                $vendor_id, 
                $approve_amount, 
                'credit', 
                '$vendor_transaction_ref', 
                'completed',
                {$wallet['balance']},
                $new_balance,
                'Daily collection approval for $transaction_date',
                {$wallet['pending_balance']},
                $new_pending,
                '$notes'
            )
        ";
        mysqli_query($conn, $vendor_wallet_transaction);
        
        // 4. Update daily transaction record
        $new_approved = $current_approved + $approve_amount;
        $new_pending_amount = $current_pending - $approve_amount;
        $remaining_amount = $daily_total - $new_approved;
        
        // Determine new status
        if ($new_pending_amount == 0) {
            $status = 'fully_approved';
        } elseif ($new_approved > 0) {
            $status = 'partially_approved';
        } else {
            $status = 'pending';
        }
        
        $update_daily = "
            UPDATE vendor_daily_transactions 
            SET approved_amount = $new_approved,
                pending_amount = $new_pending_amount,
                remaining_amount = $remaining_amount,
                status = '$status',
                notes = CONCAT(IFNULL(notes, ''), '\nApproved: ₦$approve_amount on " . date('Y-m-d H:i:s') . "'),
                updated_at = NOW()
            WHERE vendor_id = $vendor_id AND transaction_date = '$transaction_date'
        ";
        mysqli_query($conn, $update_daily);
        
        // 5. Create approval log
        $approval_ref = 'APPR-' . date('YmdHis') . '-' . $vendor_id;
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $user_agent = mysqli_real_escape_string($conn, $_SERVER['HTTP_USER_AGENT']);
        
        $approval_log = "
            INSERT INTO vendor_approval_logs 
            (vendor_id, admin_id, approval_date, approved_amount, daily_total, 
             previous_approved, new_approved, remaining_amount, notes, ip_address, user_agent)
            VALUES (
                $vendor_id,
                $admin_id,
                '$transaction_date',
                $approve_amount,
                $daily_total,
                $current_approved,
                $new_approved,
                $remaining_amount,
                '$notes',
                '$ip_address',
                '$user_agent'
            )
        ";
        mysqli_query($conn, $approval_log);
        
        // 6. Update admin wallet (if you want to track admin wallet)
        // Get admin wallet
        $admin_wallet_query = "SELECT * FROM admin_wallets LIMIT 1";
        $admin_wallet_result = mysqli_query($conn, $admin_wallet_query);
        
        if ($admin_wallet_result && mysqli_num_rows($admin_wallet_result) > 0) {
            $admin_wallet = mysqli_fetch_assoc($admin_wallet_result);
            $admin_wallet_id = $admin_wallet['id'];
            $new_admin_balance = $admin_wallet['balance'] + $approve_amount;
            
            // Update admin wallet
            $update_admin_wallet = "UPDATE admin_wallets 
                                   SET balance = $new_admin_balance, 
                                       updated_at = NOW() 
                                   WHERE id = $admin_wallet_id";
            mysqli_query($conn, $update_admin_wallet);
            
            // Create admin wallet transaction
            $admin_transaction_ref = 'VENDOR-APPROVAL-' . date('YmdHis') . '-' . $vendor_id;
            $admin_wallet_transaction = "
                INSERT INTO admin_wallet_transactions 
                (admin_wallet_id, amount, transaction_type, reference, description, 
                 balance_before, balance_after, vendor_id)
                VALUES (
                    $admin_wallet_id,
                    $approve_amount,
                    'credit',
                    '$admin_transaction_ref',
                    'Approval from vendor ' . (SELECT business_name FROM vendors WHERE id = $vendor_id),
                    {$admin_wallet['balance']},
                    $new_admin_balance,
                    $vendor_id
                )
            ";
            mysqli_query($conn, $admin_wallet_transaction);
        }
        
        // Commit transaction
        mysqli_commit($conn);
        
        $_SESSION['success'] = "Successfully approved ₦" . number_format($approve_amount, 2) . 
                              " for vendor ID: $vendor_id";
        
    } catch (Exception $e) {
        // Rollback transaction on error
        mysqli_rollback($conn);
        $_SESSION['error'] = "Approval failed: " . $e->getMessage();
    }
    
    header("Location: vendor-approval.php?date=$transaction_date");
    exit();
    
} else {
    // GET request - show approval form
    $vendor_id = isset($_GET['vendor_id']) ? intval($_GET['vendor_id']) : 0;
    $date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');
    
    if (!$vendor_id) {
        header("Location: vendor-approval.php");
        exit();
    }
    
    // Get vendor and daily transaction details
    $vendor_query = "SELECT v.* FROM vendors v WHERE v.id = $vendor_id";
    $vendor_result = mysqli_query($conn, $vendor_query);
    $vendor = mysqli_fetch_assoc($vendor_result);
    
    $daily_query = "SELECT * FROM vendor_daily_transactions 
                   WHERE vendor_id = $vendor_id AND transaction_date = '$date'";
    $daily_result = mysqli_query($conn, $daily_query);
    $daily_transaction = mysqli_fetch_assoc($daily_result);
    
    if (!$vendor || !$daily_transaction) {
        $_SESSION['error'] = "Vendor or daily transaction not found.";
        header("Location: vendor-approval.php?date=$date");
        exit();
    }
    
    // Get vendor name field
    $vendor_columns_query = "SHOW COLUMNS FROM vendors";
    $vendor_columns_result = mysqli_query($conn, $vendor_columns_query);
    $vendor_columns = [];
    while ($row = mysqli_fetch_assoc($vendor_columns_result)) {
        $vendor_columns[] = $row['Field'];
    }
    
    $name_field = 'name';
    if (in_array('business_name', $vendor_columns)) {
        $name_field = 'business_name';
    } elseif (in_array('vendor_name', $vendor_columns)) {
        $name_field = 'vendor_name';
    } elseif (in_array('full_name', $vendor_columns)) {
        $name_field = 'full_name';
    }
    
    include 'include/header.php';
    ?>

<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto">
    <div class="max-w-4xl mx-auto">
        <!-- Back Button & Header -->
        <div class="mb-6">
            <a href="vendor-transaction-details.php?vendor_id=<?= $vendor_id ?>&date=<?= $date ?>" 
               class="inline-flex items-center text-primary-500 hover:text-primary-600 mb-4">
                <i class="fas fa-arrow-left mr-2"></i> Back to Details
            </a>
            
            <h1 class="text-2xl font-bold text-gray-900 dark:text-white">Approve Vendor Money</h1>
            <p class="text-gray-600 dark:text-gray-400">
                <?= htmlspecialchars($vendor[$name_field] ?? 'Vendor') ?> - <?= date('F j, Y', strtotime($date)) ?>
            </p>
        </div>

        <!-- Summary Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <p class="text-sm text-gray-500 dark:text-gray-400 mb-1">Daily Total</p>
                <p class="text-2xl font-bold text-gray-900 dark:text-white">
                    ₦<?= number_format($daily_transaction['total_amount'], 2) ?>
                </p>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <p class="text-sm text-gray-500 dark:text-gray-400 mb-1">Already Approved</p>
                <p class="text-2xl font-bold text-green-600 dark:text-green-400">
                    ₦<?= number_format($daily_transaction['approved_amount'], 2) ?>
                </p>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <p class="text-sm text-gray-500 dark:text-gray-400 mb-1">Available to Approve</p>
                <p class="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                    ₦<?= number_format($daily_transaction['pending_amount'], 2) ?>
                </p>
            </div>
        </div>

        <!-- Approval Form -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
            <form method="POST" action="process-approval.php">
                <input type="hidden" name="vendor_id" value="<?= $vendor_id ?>">
                <input type="hidden" name="transaction_date" value="<?= $date ?>">
                
                <div class="space-y-6">
                    <!-- Vendor Info -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Vendor Information
                        </label>
                        <div class="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">Name</p>
                                    <p class="font-medium text-gray-900 dark:text-white">
                                        <?= htmlspecialchars($vendor[$name_field] ?? 'N/A') ?>
                                    </p>
                                </div>
                                <div>
                                    <p class="text-sm text-gray-500 dark:text-gray-400">Email</p>
                                    <p class="font-medium text-gray-900 dark:text-white">
                                        <?= htmlspecialchars($vendor['email'] ?? 'N/A') ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Amount to Approve -->
                    <div>
                        <label for="approve_amount" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Amount to Approve *
                        </label>
                        <div class="relative">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <span class="text-gray-500 dark:text-gray-400">₦</span>
                            </div>
                            <input type="number" id="approve_amount" name="approve_amount" 
                                   step="0.01" min="0.01" max="<?= $daily_transaction['pending_amount'] ?>"
                                   value="<?= $daily_transaction['pending_amount'] ?>"
                                   class="pl-8 w-full px-3 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 dark:bg-gray-700 dark:text-white"
                                   required>
                        </div>
                        <div class="flex justify-between mt-2">
                            <p class="text-xs text-gray-500 dark:text-gray-400">
                                Max: ₦<?= number_format($daily_transaction['pending_amount'], 2) ?>
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400">
                                Remaining after: <span id="remainingAfter">₦0.00</span>
                            </p>
                        </div>
                    </div>
                    
                    <!-- Quick Amount Buttons -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Quick Amount
                        </label>
                        <div class="flex flex-wrap gap-2">
                            <button type="button" onclick="setAmount(1000)" 
                                    class="px-3 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-800 dark:text-white rounded-md text-sm">
                                ₦1,000
                            </button>
                            <button type="button" onclick="setAmount(5000)" 
                                    class="px-3 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-800 dark:text-white rounded-md text-sm">
                                ₦5,000
                            </button>
                            <button type="button" onclick="setAmount(10000)" 
                                    class="px-3 py-2 bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-800 dark:text-white rounded-md text-sm">
                                ₦10,000
                            </button>
                            <button type="button" onclick="setAmount(<?= $daily_transaction['pending_amount'] ?>)" 
                                    class="px-3 py-2 bg-yellow-100 dark:bg-yellow-900/30 hover:bg-yellow-200 dark:hover:bg-yellow-800/30 text-yellow-800 dark:text-yellow-400 rounded-md text-sm">
                                Full Amount
                            </button>
                        </div>
                    </div>
                    
                    <!-- Notes -->
                    <div>
                        <label for="notes" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Notes (Optional)
                        </label>
                        <textarea id="notes" name="notes" rows="3" 
                                  class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-primary-500 dark:bg-gray-700 dark:text-white"
                                  placeholder="Add any notes about this approval"></textarea>
                    </div>
                    
                    <!-- Preview -->
                    <div class="bg-blue-50 dark:bg-blue-900/30 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                        <h4 class="font-medium text-blue-800 dark:text-blue-300 mb-2">Approval Preview</h4>
                        <div class="grid grid-cols-2 gap-4 text-sm">
                            <div>
                                <p class="text-blue-600 dark:text-blue-400">Current Approved</p>
                                <p class="font-medium text-gray-900 dark:text-white">
                                    ₦<?= number_format($daily_transaction['approved_amount'], 2) ?>
                                </p>
                            </div>
                            <div>
                                <p class="text-blue-600 dark:text-blue-400">After Approval</p>
                                <p class="font-medium text-green-600 dark:text-green-400" id="previewNewApproved">
                                    ₦<?= number_format($daily_transaction['approved_amount'] + $daily_transaction['pending_amount'], 2) ?>
                                </p>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Submit Buttons -->
                    <div class="flex justify-end space-x-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                        <a href="vendor-transaction-details.php?vendor_id=<?= $vendor_id ?>&date=<?= $date ?>" 
                           class="px-6 py-3 bg-gray-200 dark:bg-gray-700 hover:bg-gray-300 dark:hover:bg-gray-600 text-gray-800 dark:text-white rounded-md font-medium">
                            Cancel
                        </a>
                        <button type="submit" 
                                class="px-6 py-3 bg-green-500 hover:bg-green-600 text-white rounded-md font-medium">
                            <i class="fas fa-check-circle mr-2"></i> Confirm Approval
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</main>

<script>
const dailyTotal = <?= $daily_transaction['total_amount'] ?>;
const currentApproved = <?= $daily_transaction['approved_amount'] ?>;
const maxAmount = <?= $daily_transaction['pending_amount'] ?>;

function setAmount(amount) {
    if (amount > maxAmount) {
        amount = maxAmount;
    }
    document.getElementById('approve_amount').value = amount.toFixed(2);
    updatePreview();
}

function updatePreview() {
    const amountInput = document.getElementById('approve_amount');
    const amount = parseFloat(amountInput.value) || 0;
    
    // Validate max amount
    if (amount > maxAmount) {
        amountInput.value = maxAmount.toFixed(2);
        amount = maxAmount;
    }
    
    // Update remaining amount
    const remaining = maxAmount - amount;
    document.getElementById('remainingAfter').textContent = '₦' + remaining.toFixed(2);
    
    // Update preview
    const newApproved = currentApproved + amount;
    document.getElementById('previewNewApproved').textContent = '₦' + newApproved.toFixed(2);
}

// Add event listeners
document.getElementById('approve_amount').addEventListener('input', updatePreview);

// Initialize on page load
updatePreview();
</script>

<?php
    include 'include/footer.php';
}
?>