<?php
session_start();
// transaction-details.php - Shows money approved to vendor and admin
require '../config/conn.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: admin-login.php");
    exit();
}

// Get parameters
$vendor_id = isset($_GET['vendor_id']) ? intval($_GET['vendor_id']) : 0;
$date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');

// Validate vendor_id
if ($vendor_id <= 0) {
    header("Location: admin-approve-vendor.php");
    exit();
}

// Get vendor details - with COALESCE to handle NULL values
$vendor_query = "SELECT v.*, 
                COALESCE(vw.balance, 0) as balance, 
                COALESCE(vw.pending_balance, 0) as pending_balance 
                FROM vendors v 
                LEFT JOIN vendor_wallets vw ON v.id = vw.vendor_id 
                WHERE v.id = '$vendor_id'";
$vendor_result = mysqli_query($conn, $vendor_query);
$vendor = mysqli_fetch_assoc($vendor_result);

if (!$vendor) {
    header("Location: admin-approve-vendor.php?error=Vendor not found");
    exit();
}

// Ensure numeric values
$vendor_balance = floatval($vendor['balance']);
$vendor_pending = floatval($vendor['pending_balance']);

// Get NEW transactions that occurred on the selected date (only pending VENDOR-ENTERED transactions)
// This should show transactions waiting for approval
$transactions_query = "SELECT * FROM vendor_wallet_transactions 
                      WHERE vendor_id = '$vendor_id' 
                      AND DATE(created_at) = '$date'
                      AND status = 'pending'
                      AND transaction_type NOT IN ('admin_approval', 'commission', 'withdrawal')
                      ORDER BY created_at DESC";
$transactions_result = mysqli_query($conn, $transactions_query);
$pending_count = mysqli_num_rows($transactions_result);

// Calculate pending total for today (vendor-entered only)
$daily_pending_total = 0;
$transactions_data = [];
while ($row = mysqli_fetch_assoc($transactions_result)) {
    $transactions_data[] = $row;
    $daily_pending_total += floatval($row['amount']);
}

// Get ALL transactions for today (including admin transactions) for comparison
$all_today_summary_query = "SELECT 
                           COUNT(*) as all_total_count,
                           COALESCE(SUM(amount), 0) as all_grand_total,
                           SUM(CASE WHEN transaction_type IN ('admin_approval', 'commission', 'withdrawal') THEN 1 ELSE 0 END) as admin_transaction_count
                           FROM vendor_wallet_transactions 
                           WHERE vendor_id = '$vendor_id' 
                           AND DATE(created_at) = '$date'";
$all_today_summary_result = mysqli_query($conn, $all_today_summary_query);
$all_today_summary = mysqli_fetch_assoc($all_today_summary_result);
$admin_transaction_count = intval($all_today_summary['admin_transaction_count'] ?: 0);
$all_total_count = intval($all_today_summary['all_total_count'] ?: 0);

// Get APPROVAL HISTORY for this vendor on the selected date
$approvals_query = "SELECT aa.*, a.full_name as admin_name 
                   FROM admin_approvals aa 
                   LEFT JOIN admins a ON aa.admin_id = a.admin_id 
                   WHERE aa.vendor_id = '$vendor_id' 
                   AND aa.approval_date = '$date'
                   ORDER BY aa.created_at DESC";
$approvals_result = mysqli_query($conn, $approvals_query);
$approvals_count = mysqli_num_rows($approvals_result);

// Calculate totals from approvals
$total_gross_approved = 0;
$total_commission = 0;
$total_net_to_vendor = 0;
$approvals_data = [];

if ($approvals_count > 0) {
    while ($row = mysqli_fetch_assoc($approvals_result)) {
        $approvals_data[] = $row;
        $total_gross_approved += floatval($row['approval_amount']);
        $total_commission += floatval($row['commission_amount']);
        $total_net_to_vendor += floatval($row['net_amount']);
    }
}
?>

<?php include 'include/header.php'; ?>

<!-- Content Area -->
<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto">
    <div class="max-w-7xl mx-auto">
        <!-- Header with Back Button -->
        <div class="mb-6 flex items-center justify-between">
            <div>
                <a href="admin-approve-vendor.php<?php echo isset($_GET['date']) ? '?date=' . htmlspecialchars($_GET['date']) : ''; ?>" 
                   class="inline-flex items-center text-yellow-600 hover:text-yellow-800 dark:text-yellow-400 dark:hover:text-yellow-300 mb-2">
                    <i class="fas fa-arrow-left mr-2"></i> Back to Vendors
                </a>
                <h1 class="text-2xl font-bold text-gray-900 dark:text-white">Daily Transactions & Approvals</h1>
                <p class="text-gray-600 dark:text-gray-400">
                    <?php echo htmlspecialchars($vendor['business_name']); ?> - <?php echo date('F j, Y', strtotime($date)); ?>
                </p>
            </div>
            <div class="text-right">
                <!-- Date Filter Form -->
                <form method="GET" action="" class="flex items-center space-x-2">
                    <input type="hidden" name="vendor_id" value="<?php echo $vendor_id; ?>">
                    <input type="date" name="date" value="<?php echo htmlspecialchars($date); ?>" 
                           class="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-yellow-500 dark:bg-gray-700 dark:text-white">
                    <button type="submit" class="bg-yellow-500 hover:bg-yellow-600 text-white px-4 py-2 rounded-md font-medium">
                        <i class="fas fa-filter mr-2"></i> View Day
                    </button>
                </form>
            </div>
        </div>

        <!-- Vendor Summary - SIMPLIFIED -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-blue-50 dark:bg-blue-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-store text-blue-500 dark:text-blue-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Vendor</p>
                        <p class="text-lg font-bold text-gray-900 dark:text-white truncate" title="<?php echo htmlspecialchars($vendor['business_name']); ?>">
                            <?php echo htmlspecialchars($vendor['business_name']); ?>
                        </p>
                        <p class="text-xs text-gray-500 dark:text-gray-400 mt-1">ID: <?php echo $vendor_id; ?></p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-green-50 dark:bg-green-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-wallet text-green-500 dark:text-green-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Available Balance</p>
                        <p class="text-2xl font-bold text-gray-900 dark:text-white">₦<?php echo number_format($vendor_balance, 2); ?></p>
                        <p class="text-xs text-gray-500 dark:text-gray-400">For withdrawal</p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700">
                <div class="flex items-center">
                    <div class="w-12 h-12 rounded-full bg-yellow-50 dark:bg-yellow-900/30 flex items-center justify-center mr-4">
                        <i class="fas fa-clock text-yellow-500 dark:text-yellow-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-500 dark:text-gray-400">Total Pending</p>
                        <p class="text-2xl font-bold text-gray-900 dark:text-white">₦<?php echo number_format($vendor_pending, 2); ?></p>
                        <p class="text-xs text-gray-500 dark:text-gray-400">Awaiting approval</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Day Summary Card - SIMPLIFIED (Only Pending Today) -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6 border border-gray-100 dark:border-gray-700 mb-6">
            <h3 class="text-lg font-semibold dark:text-white mb-4"><?php echo date('l, F j, Y', strtotime($date)); ?> Summary</h3>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div class="bg-gray-50 dark:bg-gray-800 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                    <div class="flex items-center">
                        <div class="w-10 h-10 rounded-full bg-gray-100 dark:bg-gray-700 flex items-center justify-center mr-3">
                            <i class="fas fa-layer-group text-gray-500 dark:text-gray-400"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600 dark:text-gray-300">Vendor Transactions Today</p>
                            <p class="text-xl font-bold text-gray-900 dark:text-white">
                                <?php echo $pending_count; ?> transactions
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400">
                                ₦<?php echo number_format($daily_pending_total, 2); ?> total value
                            </p>
                        </div>
                    </div>
                </div>
                
                <div class="bg-yellow-50 dark:bg-yellow-900/20 p-4 rounded-lg">
                    <div class="flex items-center">
                        <div class="w-10 h-10 rounded-full bg-yellow-100 dark:bg-yellow-900 flex items-center justify-center mr-3">
                            <i class="fas fa-clock text-yellow-500 dark:text-yellow-400"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600 dark:text-gray-300">Pending for Approval</p>
                            <p class="text-xl font-bold text-yellow-600 dark:text-yellow-400">
                                <?php echo $pending_count; ?> transactions
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400">
                                ₦<?php echo number_format($daily_pending_total, 2); ?> pending
                            </p>
                        </div>
                    </div>
                </div>
            </div>
            
            <?php if ($admin_transaction_count > 0): ?>
            <div class="mt-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-info-circle text-blue-500 dark:text-blue-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-blue-700 dark:text-blue-300">
                            Note: There are <?php echo $admin_transaction_count; ?> admin-related transaction<?php echo $admin_transaction_count == 1 ? '' : 's'; ?> 
                            (approvals, commissions, withdrawals) not included in the counts above.
                        </p>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>

        <!-- Two Column Layout: Pending Transactions and Approvals History -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
            <!-- Left Column: Pending Transactions -->
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 overflow-hidden">
                <div class="px-6 py-4 border-b border-gray-100 dark:border-gray-700">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold dark:text-white">
                            Pending Transactions
                            <?php if ($pending_count > 0): ?>
                                <span class="text-sm font-normal text-gray-500 dark:text-gray-400 ml-2">
                                    (<?php echo $pending_count; ?> pending)
                                </span>
                            <?php endif; ?>
                        </h3>
                        <div class="text-sm">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300">
                                <i class="fas fa-clock mr-1"></i>
                                Awaiting Approval
                            </span>
                        </div>
                    </div>
                </div>
                
                <div class="overflow-x-auto">
                    <?php if ($pending_count > 0): ?>
                        <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                            <thead class="bg-gray-50 dark:bg-gray-800">
                                <tr>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Time</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Amount</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Description</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Type</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                <?php foreach ($transactions_data as $transaction): ?>
                                <tr class="hover:bg-gray-50 dark:hover:bg-gray-700/50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        <?php echo date('h:i A', strtotime($transaction['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-lg font-bold text-gray-900 dark:text-white">
                                            ₦<?php echo number_format(floatval($transaction['amount']), 2); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="text-sm text-gray-900 dark:text-white truncate max-w-xs" title="<?php echo htmlspecialchars($transaction['description']); ?>">
                                            <?php echo htmlspecialchars($transaction['description']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                            <?php echo $transaction['transaction_type'] === 'payment' ? 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-300' : 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300'; ?>">
                                            <?php echo htmlspecialchars($transaction['transaction_type']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <div class="text-center py-8">
                            <div class="mx-auto w-16 h-16 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center mb-4">
                                <i class="fas fa-check text-green-500 dark:text-green-400 text-xl"></i>
                            </div>
                            <h4 class="text-lg font-medium text-gray-700 dark:text-gray-300 mb-2">All Caught Up!</h4>
                            <p class="text-gray-500 dark:text-gray-400">
                                No pending vendor transactions for today.
                            </p>
                            <?php if ($admin_transaction_count > 0): ?>
                            <p class="text-sm text-orange-600 dark:text-orange-400 mt-2">
                                <i class="fas fa-info-circle mr-1"></i>
                                There are <?php echo $admin_transaction_count; ?> admin transaction<?php echo $admin_transaction_count == 1 ? '' : 's'; ?> 
                                not shown here.
                            </p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if ($pending_count > 0): ?>
                <div class="px-6 py-4 border-t border-gray-100 dark:border-gray-700 bg-gray-50 dark:bg-gray-800">
                    <div class="flex justify-between items-center">
                        <div>
                            <p class="text-sm text-gray-600 dark:text-gray-300">Total Pending for Approval:</p>
                            <p class="text-xl font-bold text-gray-900 dark:text-white">
                                ₦<?php echo number_format($daily_pending_total, 2); ?>
                            </p>
                            <p class="text-xs text-gray-500 dark:text-gray-400">
                                Vendor-entered transactions only
                            </p>
                        </div>
                        <a href="admin-approve-vendor.php?date=<?php echo urlencode($date); ?>" 
                           class="bg-yellow-500 hover:bg-yellow-600 text-white px-4 py-2 rounded-md text-sm font-medium">
                            <i class="fas fa-check-circle mr-2"></i> Go to Approve
                        </a>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Right Column: Approval History -->
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700 overflow-hidden">
                <div class="px-6 py-4 border-b border-gray-100 dark:border-gray-700">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold dark:text-white">
                            Approval History
                            <?php if ($approvals_count > 0): ?>
                                <span class="text-sm font-normal text-gray-500 dark:text-gray-400 ml-2">
                                    (<?php echo $approvals_count; ?> approval<?php echo $approvals_count == 1 ? '' : 's'; ?>)
                                </span>
                            <?php endif; ?>
                        </h3>
                        <div class="text-sm">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300">
                                <i class="fas fa-history mr-1"></i>
                                Past Approvals
                            </span>
                        </div>
                    </div>
                </div>
                
                <div class="overflow-x-auto max-h-96">
                    <?php if ($approvals_count > 0): ?>
                        <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                            <thead class="bg-gray-50 dark:bg-gray-800 sticky top-0">
                                <tr>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Time</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Approved By</th>
                                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Amounts</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                <?php foreach ($approvals_data as $approval): 
                                    $approval_amount = floatval($approval['approval_amount']);
                                    $commission_amount = floatval($approval['commission_amount']);
                                    $net_amount = floatval($approval['net_amount']);
                                    $commission_percentage = $approval_amount > 0 ? 
                                        ($commission_amount / $approval_amount * 100) : 0;
                                ?>
                                <tr class="hover:bg-gray-50 dark:hover:bg-gray-700/50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900 dark:text-white">
                                        <?php echo date('h:i A', strtotime($approval['created_at'])); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900 dark:text-white">
                                            <?php echo htmlspecialchars($approval['admin_name'] ?: 'Admin'); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4">
                                        <div class="space-y-1">
                                            <!-- Gross Amount -->
                                            <div class="flex justify-between text-sm">
                                                <span class="text-gray-600 dark:text-gray-300">Gross:</span>
                                                <span class="font-medium text-gray-900 dark:text-white">
                                                    ₦<?php echo number_format($approval_amount, 2); ?>
                                                </span>
                                            </div>
                                            <!-- Commission -->
                                            <div class="flex justify-between text-sm">
                                                <span class="text-red-600 dark:text-red-400">Commission:</span>
                                                <span class="font-medium text-red-600 dark:text-red-400">
                                                    -₦<?php echo number_format($commission_amount, 2); ?>
                                                    <span class="text-xs">(<?php echo number_format($commission_percentage, 1); ?>%)</span>
                                                </span>
                                            </div>
                                            <!-- Net to Vendor -->
                                            <div class="flex justify-between text-sm pt-1 border-t border-gray-100 dark:border-gray-700">
                                                <span class="text-green-600 dark:text-green-400 font-medium">Net to Vendor:</span>
                                                <span class="font-bold text-green-600 dark:text-green-400">
                                                    ₦<?php echo number_format($net_amount, 2); ?>
                                                </span>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <div class="text-center py-8">
                            <div class="mx-auto w-16 h-16 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center mb-4">
                                <i class="fas fa-file-invoice-dollar text-blue-500 dark:text-blue-400 text-xl"></i>
                            </div>
                            <h4 class="text-lg font-medium text-gray-700 dark:text-gray-300 mb-2">No Approvals Yet</h4>
                            <p class="text-gray-500 dark:text-gray-400 mb-4">
                                No approvals have been processed for this date.
                            </p>
                            <?php if ($pending_count > 0): ?>
                            <p class="text-sm text-yellow-600 dark:text-yellow-400">
                                <i class="fas fa-exclamation-circle mr-1"></i>
                                There are <?php echo $pending_count; ?> pending transaction<?php echo $pending_count == 1 ? '' : 's'; ?> waiting for approval.
                            </p>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if ($approvals_count > 0): ?>
                <div class="px-6 py-4 border-t border-gray-100 dark:border-gray-700 bg-gray-50 dark:bg-gray-800">
                    <div class="grid grid-cols-3 gap-4 text-sm">
                        <div>
                            <p class="text-gray-600 dark:text-gray-300">Total Gross:</p>
                            <p class="font-bold text-blue-600 dark:text-blue-400">
                                ₦<?php echo number_format($total_gross_approved, 2); ?>
                            </p>
                        </div>
                        <div>
                            <p class="text-gray-600 dark:text-gray-300">Total Commission:</p>
                            <p class="font-bold text-red-600 dark:text-red-400">
                                ₦<?php echo number_format($total_commission, 2); ?>
                            </p>
                        </div>
                        <div>
                            <p class="text-gray-600 dark:text-gray-300">Total Net:</p>
                            <p class="font-bold text-green-600 dark:text-green-400">
                                ₦<?php echo number_format($total_net_to_vendor, 2); ?>
                            </p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Simplified Information Box -->
        <div class="mt-6 bg-blue-50 dark:bg-blue-900/20 border border-blue-100 dark:border-blue-800 rounded-lg p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-info-circle text-blue-500 dark:text-blue-400 text-lg"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800 dark:text-blue-300">Transaction Summary</h3>
                    <div class="mt-2 text-sm text-blue-700 dark:text-blue-400">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <p><strong>For Vendor (<?php echo htmlspecialchars($vendor['business_name']); ?>):</strong></p>
                                <ul class="list-disc pl-5 space-y-1 mt-1">
                                    <li>Available Balance: ₦<?php echo number_format($vendor_balance, 2); ?></li>
                                    <li>Total Pending Balance: ₦<?php echo number_format($vendor_pending, 2); ?></li>
                                    <li>Pending Today: <?php echo $pending_count; ?> transactions (₦<?php echo number_format($daily_pending_total, 2); ?>)</li>
                                </ul>
                            </div>
                            <div>
                                <p><strong>Today's Summary:</strong></p>
                                <ul class="list-disc pl-5 space-y-1 mt-1">
                                    <li>Vendor Transactions: <?php echo $pending_count; ?> (₦<?php echo number_format($daily_pending_total, 2); ?>)</li>
                                    <?php if ($approvals_count > 0): ?>
                                    <li>Approval Actions: <?php echo $approvals_count; ?></li>
                                    <li>Total Commission: ₦<?php echo number_format($total_commission, 2); ?></li>
                                    <?php endif; ?>
                                    <?php if ($admin_transaction_count > 0): ?>
                                    <li>Admin Transactions: <?php echo $admin_transaction_count; ?> (not included in counts)</li>
                                    <?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<script>
// Function to go to today's date
function goToToday() {
    const today = new Date().toISOString().split('T')[0];
    window.location.href = `?vendor_id=<?php echo $vendor_id; ?>&date=${today}`;
}

// Function to go to previous day
function goToPreviousDay() {
    const currentDate = new Date('<?php echo $date; ?>');
    currentDate.setDate(currentDate.getDate() - 1);
    const prevDate = currentDate.toISOString().split('T')[0];
    window.location.href = `?vendor_id=<?php echo $vendor_id; ?>&date=${prevDate}`;
}

// Function to go to next day
function goToNextDay() {
    const currentDate = new Date('<?php echo $date; ?>');
    currentDate.setDate(currentDate.getDate() + 1);
    const nextDate = currentDate.toISOString().split('T')[0];
    window.location.href = `?vendor_id=<?php echo $vendor_id; ?>&date=${nextDate}`;
}
</script>

<?php include 'include/footer.php'; ?>