<?php
// Epay-management/cron/check-pending-transfers.php

// Get absolute path
require_once __DIR__ . '/config/conn.php';

// Check if vendor autoload exists
$vendorAutoload = __DIR__ . '/vendor/autoload.php';
if (file_exists($vendorAutoload)) {
    require_once $vendorAutoload;
    
    use Yabacon\Paystack;
    
    // Initialize Paystack
    if (!defined('PAYSTACK_SECRET_KEY')) {
        error_log("Cron: PAYSTACK_SECRET_KEY not defined");
        exit();
    }
    
    $paystack = new Paystack(PAYSTACK_SECRET_KEY);
    
    // Log cron execution
    error_log("Cron job started: " . date('Y-m-d H:i:s'));
    
    // Get pending transfers older than 5 minutes
    $stmt = $conn->prepare("
        SELECT * FROM withdrawal_transactions 
        WHERE status IN ('pending', 'processing') 
        AND created_at < DATE_SUB(NOW(), INTERVAL 5 MINUTE)
        LIMIT 10
    ");
    
    if ($stmt) {
        $stmt->execute();
        $result = $stmt->get_result();
        $count = 0;
        
        while ($transaction = $result->fetch_assoc()) {
            if ($transaction['paystack_reference']) {
                try {
                    $transfer = $paystack->transfer->verify([
                        'reference' => $transaction['paystack_reference']
                    ]);
                    
                    if ($transfer->status) {
                        $status = $transfer->data->status;
                        
                        if ($status === 'success') {
                            // Update as completed
                            $update = $conn->prepare("UPDATE withdrawal_transactions SET status = 'completed' WHERE id = ?");
                            $update->bind_param("i", $transaction['id']);
                            $update->execute();
                            $update->close();
                            
                            error_log("Cron: Transfer {$transaction['reference']} marked as completed");
                            $count++;
                            
                        } elseif ($status === 'failed') {
                            // Update as failed
                            $update = $conn->prepare("UPDATE withdrawal_transactions SET status = 'failed' WHERE id = ?");
                            $update->bind_param("i", $transaction['id']);
                            $update->execute();
                            $update->close();
                            
                            // Refund wallet balance
                            refundFailedTransfer($conn, $transaction);
                            
                            error_log("Cron: Transfer {$transaction['reference']} marked as failed");
                            $count++;
                        }
                    }
                } catch (Exception $e) {
                    error_log("Cron: Error checking transfer {$transaction['reference']}: " . $e->getMessage());
                }
            }
        }
        
        $stmt->close();
        error_log("Cron job completed. Processed: {$count} transactions");
    }
    
} else {
    error_log("Cron: Vendor autoload not found");
}

// Function to refund failed transfers
function refundFailedTransfer($conn, $transaction) {
    // Refund to wallet
    $update_wallet = $conn->prepare("UPDATE student_wallets SET balance = balance + ? WHERE id = ?");
    $update_wallet->bind_param("di", $transaction['total_amount'], $transaction['wallet_id']);
    $update_wallet->execute();
    $update_wallet->close();
    
    // Record refund transaction
    $refund_ref = 'REF' . time() . mt_rand(1000, 9999);
    $refund_desc = "Refund for failed withdrawal";
    
    $refund_trans = $conn->prepare("
        INSERT INTO wallet_transactions (
            student_id,
            wallet_id,
            amount, 
            transaction_type, 
            description, 
            status, 
            reference,
            created_at
        ) VALUES (?, ?, ?, 'credit', ?, 'completed', ?, NOW())
    ");
    $refund_trans->bind_param("iidss", 
        $transaction['student_id'], 
        $transaction['wallet_id'], 
        $transaction['total_amount'], 
        $refund_desc, 
        $refund_ref
    );
    $refund_trans->execute();
    $refund_trans->close();
    
    return true;
}
?>