<?php
// Epay-management/cron/other-cron-jobs.php
// Multiple cron jobs combined into one file

// Get absolute path
require_once __DIR__ . '/config/conn.php';

// Log execution
error_log("=== CRON JOBS STARTED: " . date('Y-m-d H:i:s') . " ===");

// Determine which job to run based on time or parameter
$job_to_run = isset($_GET['job']) ? $_GET['job'] : 'all';

// Run selected jobs
switch ($job_to_run) {
    case 'cleanup':
        cleanupOldData();
        break;
    case 'notifications':
        sendScheduledNotifications();
        break;
    case 'vendor_payouts':
        processVendorPayouts();
        break;
    case 'reports':
        generateDailyReports();
        break;
    case 'all':
    default:
        // Run all jobs in sequence
        cleanupOldData();
        sendScheduledNotifications();
        processVendorPayouts();
        generateDailyReports();
        checkFailedTransactions();
        updateExchangeRates();
        backupDatabase();
        break;
}

error_log("=== CRON JOBS COMPLETED: " . date('Y-m-d H:i:s') . " ===");

// -----------------------------------------------------------------
// JOB 1: Cleanup Old Data
// -----------------------------------------------------------------
function cleanupOldData() {
    global $conn;
    
    error_log("Cron: Starting data cleanup...");
    
    // 1. Delete failed login attempts older than 30 days
    $query = "DELETE FROM login_attempts WHERE created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)";
    mysqli_query($conn, $query);
    $count1 = mysqli_affected_rows($conn);
    
    // 2. Clean old session data (if you store sessions in DB)
    $query = "DELETE FROM user_sessions WHERE last_activity < DATE_SUB(NOW(), INTERVAL 7 DAY)";
    mysqli_query($conn, $query);
    $count2 = mysqli_affected_rows($conn);
    
    // 3. Archive completed transactions older than 90 days
    archiveOldTransactions();
    
    // 4. Clean temporary files
    cleanTempFiles();
    
    error_log("Cron: Cleanup completed. Removed $count1 login attempts, $count2 old sessions");
    
    return ['login_attempts' => $count1, 'sessions' => $count2];
}

// -----------------------------------------------------------------
// JOB 2: Send Scheduled Notifications
// -----------------------------------------------------------------
function sendScheduledNotifications() {
    global $conn;
    
    error_log("Cron: Starting notification sending...");
    
    // 1. Send low balance notifications
    $query = "
        SELECT s.id, s.email, s.full_name, sw.balance 
        FROM students s 
        JOIN student_wallets sw ON s.id = sw.student_id 
        WHERE sw.balance < 500 
        AND sw.balance > 0 
        AND s.receive_notifications = 1
        AND s.id NOT IN (
            SELECT student_id FROM notifications 
            WHERE type = 'low_balance' 
            AND DATE(created_at) = CURDATE()
        )
    ";
    
    $result = mysqli_query($conn, $query);
    $low_balance_count = 0;
    
    while ($student = mysqli_fetch_assoc($result)) {
        // Send email notification
        $subject = "Low Wallet Balance Alert";
        $message = "Dear {$student['full_name']}, your wallet balance is low: ₦{$student['balance']}. Please top up to continue using our services.";
        
        // Send email (you would use PHPMailer or similar)
        // mail($student['email'], $subject, $message);
        
        // Record notification
        $insert = "INSERT INTO notifications (student_id, type, title, message, created_at) 
                   VALUES ({$student['id']}, 'low_balance', 'Low Balance', '$message', NOW())";
        mysqli_query($conn, $insert);
        
        $low_balance_count++;
    }
    
    // 2. Send vendor payout notifications
    $query = "
        SELECT v.id, v.email, v.business_name, vw.balance 
        FROM vendors v 
        JOIN vendor_wallets vw ON v.id = vw.vendor_id 
        WHERE vw.balance >= 1000 
        AND v.receive_notifications = 1
        AND v.id NOT IN (
            SELECT vendor_id FROM vendor_notifications 
            WHERE type = 'payout_available' 
            AND DATE(created_at) = CURDATE()
        )
    ";
    
    $result = mysqli_query($conn, $query);
    $payout_count = 0;
    
    while ($vendor = mysqli_fetch_assoc($result)) {
        $subject = "Payout Available";
        $message = "Dear {$vendor['business_name']}, you have ₦{$vendor['balance']} available for withdrawal.";
        
        // Record notification
        $insert = "INSERT INTO vendor_notifications (vendor_id, type, title, message, created_at) 
                   VALUES ({$vendor['id']}, 'payout_available', 'Payout Available', '$message', NOW())";
        mysqli_query($conn, $insert);
        
        $payout_count++;
    }
    
    error_log("Cron: Notifications sent. Low balance: $low_balance_count, Payout alerts: $payout_count");
    
    return ['low_balance' => $low_balance_count, 'payout_alerts' => $payout_count];
}

// -----------------------------------------------------------------
// JOB 3: Process Vendor Payouts (Auto-approve after X days)
// -----------------------------------------------------------------
function processVendorPayouts() {
    global $conn;
    
    error_log("Cron: Starting vendor payout processing...");
    
    // 1. Auto-approve pending vendor balances older than 7 days
    $query = "
        UPDATE vendor_wallets 
        SET balance = balance + pending_balance,
            pending_balance = 0,
            updated_at = NOW()
        WHERE pending_balance > 0 
        AND updated_at < DATE_SUB(NOW(), INTERVAL 7 DAY)
    ";
    
    mysqli_query($conn, $query);
    $approved_count = mysqli_affected_rows($conn);
    
    // 2. Record the approval transactions
    if ($approved_count > 0) {
        $query = "
            SELECT vw.id, vw.vendor_id, vw.pending_balance, v.business_name
            FROM vendor_wallets vw
            JOIN vendors v ON vw.vendor_id = v.id
            WHERE vw.pending_balance > 0 
            AND vw.updated_at < DATE_SUB(NOW(), INTERVAL 7 DAY)
        ";
        
        $result = mysqli_query($conn, $query);
        while ($vendor = mysqli_fetch_assoc($result)) {
            $reference = 'APR' . time() . mt_rand(1000, 9999);
            $description = "Auto-approved pending balance after 7 days";
            
            $insert = "INSERT INTO vendor_wallet_transactions 
                (vendor_id, amount, transaction_type, reference, description, status, created_at) 
                VALUES ({$vendor['vendor_id']}, {$vendor['pending_balance']}, 'credit', 
                '$reference', '$description', 'completed', NOW())";
            mysqli_query($conn, $insert);
        }
    }
    
    // 3. Process scheduled payouts (if vendors set up auto-payout)
    processScheduledVendorPayouts();
    
    error_log("Cron: Vendor payouts processed. Auto-approved: $approved_count");
    
    return ['auto_approved' => $approved_count];
}

// -----------------------------------------------------------------
// JOB 4: Generate Daily Reports
// -----------------------------------------------------------------
function generateDailyReports() {
    global $conn;
    
    error_log("Cron: Generating daily reports...");
    
    $date = date('Y-m-d', strtotime('-1 day')); // Yesterday's report
    
    // Check if report already exists for yesterday
    $check = "SELECT id FROM daily_reports WHERE report_date = '$date'";
    $result = mysqli_query($conn, $check);
    
    if (mysqli_num_rows($result) > 0) {
        error_log("Cron: Report already exists for $date");
        return ['status' => 'exists'];
    }
    
    // 1. Get daily transactions summary
    $query = "
        SELECT 
            COUNT(*) as total_transactions,
            SUM(CASE WHEN transaction_type = 'debit' THEN amount ELSE 0 END) as total_debits,
            SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE 0 END) as total_credits,
            SUM(CASE WHEN transaction_type = 'debit' THEN fee ELSE 0 END) as total_fees
        FROM wallet_transactions 
        WHERE DATE(created_at) = '$date'
    ";
    
    $result = mysqli_query($conn, $query);
    $transactions = mysqli_fetch_assoc($result);
    
    // 2. Get daily withdrawals summary
    $query = "
        SELECT 
            COUNT(*) as total_withdrawals,
            SUM(amount) as withdrawal_amount,
            SUM(fee) as withdrawal_fees
        FROM withdrawal_transactions 
        WHERE DATE(created_at) = '$date' AND status = 'completed'
    ";
    
    $result = mysqli_query($conn, $query);
    $withdrawals = mysqli_fetch_assoc($result);
    
    // 3. Get new users
    $query = "
        SELECT COUNT(*) as new_students 
        FROM students 
        WHERE DATE(created_at) = '$date'
    ";
    $result = mysqli_query($conn, $query);
    $new_students = mysqli_fetch_assoc($result)['new_students'];
    
    // 4. Get new vendors
    $query = "
        SELECT COUNT(*) as new_vendors 
        FROM vendors 
        WHERE DATE(created_at) = '$date'
    ";
    $result = mysqli_query($conn, $query);
    $new_vendors = mysqli_fetch_assoc($result)['new_vendors'];
    
    // 5. Calculate revenue
    $revenue = ($transactions['total_fees'] ?? 0) + ($withdrawals['withdrawal_fees'] ?? 0);
    
    // Insert report
    $insert = "INSERT INTO daily_reports 
        (report_date, total_transactions, total_debits, total_credits, 
         total_withdrawals, withdrawal_amount, new_students, new_vendors, 
         revenue, created_at) 
        VALUES ('$date', 
                {$transactions['total_transactions']}, 
                {$transactions['total_debits']}, 
                {$transactions['total_credits']},
                {$withdrawals['total_withdrawals']},
                {$withdrawals['withdrawal_amount']},
                $new_students,
                $new_vendors,
                $revenue,
                NOW())";
    
    mysqli_query($conn, $insert);
    
    error_log("Cron: Daily report generated for $date. Revenue: ₦$revenue");
    
    return [
        'date' => $date,
        'transactions' => $transactions['total_transactions'],
        'revenue' => $revenue
    ];
}

// -----------------------------------------------------------------
// JOB 5: Check Failed Transactions & Retry
// -----------------------------------------------------------------
function checkFailedTransactions() {
    global $conn;
    
    error_log("Cron: Checking failed transactions...");
    
    // 1. Check failed Paystack transfers (retry once)
    $query = "
        SELECT * FROM withdrawal_transactions 
        WHERE status = 'failed' 
        AND retry_count < 3 
        AND created_at > DATE_SUB(NOW(), INTERVAL 3 DAY)
        ORDER BY created_at DESC 
        LIMIT 5
    ";
    
    $result = mysqli_query($conn, $query);
    $retry_count = 0;
    
    while ($transaction = mysqli_fetch_assoc($result)) {
        // Check if failure was temporary (e.g., network issue)
        if (shouldRetryTransaction($transaction)) {
            retryFailedTransaction($transaction);
            $retry_count++;
        }
    }
    
    // 2. Mark old pending transactions as expired
    $query = "
        UPDATE withdrawal_transactions 
        SET status = 'expired'
        WHERE status = 'pending' 
        AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
    ";
    
    mysqli_query($conn, $query);
    $expired_count = mysqli_affected_rows($conn);
    
    error_log("Cron: Failed transactions checked. Retried: $retry_count, Expired: $expired_count");
    
    return ['retried' => $retry_count, 'expired' => $expired_count];
}

// -----------------------------------------------------------------
// JOB 6: Update Exchange Rates
// -----------------------------------------------------------------
function updateExchangeRates() {
    global $conn;
    
    error_log("Cron: Updating exchange rates...");
    
    // Fetch from external API (example using free API)
    $api_url = 'https://api.exchangerate-api.com/v4/latest/USD';
    
    try {
        $data = @file_get_contents($api_url);
        if ($data) {
            $rates = json_decode($data, true);
            
            if (isset($rates['rates']['NGN'])) {
                $ngn_rate = $rates['rates']['NGN'];
                $timestamp = date('Y-m-d H:i:s');
                
                // Insert into exchange_rates table
                $query = "INSERT INTO exchange_rates (currency_from, currency_to, rate, updated_at) 
                          VALUES ('USD', 'NGN', $ngn_rate, '$timestamp')
                          ON DUPLICATE KEY UPDATE rate = VALUES(rate), updated_at = VALUES(updated_at)";
                
                mysqli_query($conn, $query);
                
                error_log("Cron: Exchange rate updated. USD to NGN: $ngn_rate");
                return ['rate' => $ngn_rate, 'status' => 'updated'];
            }
        }
    } catch (Exception $e) {
        error_log("Cron: Failed to update exchange rates: " . $e->getMessage());
    }
    
    return ['status' => 'failed'];
}

// -----------------------------------------------------------------
// JOB 7: Database Backup
// -----------------------------------------------------------------
function backupDatabase() {
    global $conn;
    
    error_log("Cron: Starting database backup...");
    
    $backup_dir = __DIR__ . '/../backups/';
    
    // Create backups directory if it doesn't exist
    if (!file_exists($backup_dir)) {
        mkdir($backup_dir, 0755, true);
    }
    
    // Configuration
    $db_host = 'localhost';
    $db_user = 'root';
    $db_pass = '';
    $db_name = 'epay_management';
    
    $backup_file = $backup_dir . $db_name . '_' . date('Y-m-d_H-i-s') . '.sql';
    
    // Command to backup using mysqldump
    $command = "mysqldump --opt -h$db_host -u$db_user -p$db_pass $db_name > \"$backup_file\"";
    
    // Execute command
    system($command, $output);
    
    if ($output === 0 && file_exists($backup_file)) {
        // Compress the backup
        $compressed_file = $backup_file . '.gz';
        $gz = gzopen($compressed_file, 'w9');
        gzwrite($gz, file_get_contents($backup_file));
        gzclose($gz);
        
        // Remove uncompressed file
        unlink($backup_file);
        
        // Delete backups older than 7 days
        $files = glob($backup_dir . '*.sql.gz');
        foreach ($files as $file) {
            if (filemtime($file) < time() - (7 * 24 * 60 * 60)) {
                unlink($file);
            }
        }
        
        error_log("Cron: Database backup completed: " . basename($compressed_file));
        return ['file' => basename($compressed_file), 'size' => filesize($compressed_file)];
    } else {
        error_log("Cron: Database backup failed");
        return ['status' => 'failed'];
    }
}

// -----------------------------------------------------------------
// HELPER FUNCTIONS
// -----------------------------------------------------------------
function archiveOldTransactions() {
    global $conn;
    
    // Archive transactions older than 90 days
    $query = "
        INSERT INTO wallet_transactions_archive 
        SELECT * FROM wallet_transactions 
        WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)
    ";
    
    mysqli_query($conn, $query);
    $archived = mysqli_affected_rows($conn);
    
    if ($archived > 0) {
        // Delete from main table
        $query = "DELETE FROM wallet_transactions WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)";
        mysqli_query($conn, $query);
    }
    
    return $archived;
}

function cleanTempFiles() {
    $temp_dir = __DIR__ . '/../temp/';
    
    if (file_exists($temp_dir)) {
        $files = glob($temp_dir . '*');
        $deleted = 0;
        
        foreach ($files as $file) {
            // Delete files older than 24 hours
            if (filemtime($file) < time() - (24 * 60 * 60)) {
                unlink($file);
                $deleted++;
            }
        }
        
        return $deleted;
    }
    
    return 0;
}

function processScheduledVendorPayouts() {
    global $conn;
    
    // Get vendors with auto-payout enabled
    $query = "
        SELECT v.id, v.business_name, vw.balance, vba.*
        FROM vendors v
        JOIN vendor_wallets vw ON v.id = vw.vendor_id
        JOIN vendor_bank_accounts vba ON v.id = vba.vendor_id AND vba.is_primary = 1
        WHERE v.auto_payout_enabled = 1 
        AND vw.balance >= v.payout_threshold
        AND (v.last_payout_date IS NULL OR v.last_payout_date < DATE_SUB(NOW(), INTERVAL v.payout_frequency DAYS))
    ";
    
    $result = mysqli_query($conn, $query);
    $processed = 0;
    
    while ($vendor = mysqli_fetch_assoc($result)) {
        // Process automatic payout
        // This would integrate with Paystack API
        $processed++;
    }
    
    return $processed;
}

function shouldRetryTransaction($transaction) {
    // Check if transaction should be retried
    $non_retryable_errors = [
        'insufficient_funds',
        'invalid_account',
        'account_closed'
    ];
    
    // If failure reason contains non-retryable error
    foreach ($non_retryable_errors as $error) {
        if (stripos($transaction['failure_reason'], $error) !== false) {
            return false;
        }
    }
    
    // Check retry count
    if ($transaction['retry_count'] >= 3) {
        return false;
    }
    
    return true;
}

function retryFailedTransaction($transaction) {
    global $conn;
    
    // Increment retry count
    $query = "UPDATE withdrawal_transactions SET retry_count = retry_count + 1 WHERE id = {$transaction['id']}";
    mysqli_query($conn, $query);
    
    // Here you would re-initiate the Paystack transfer
    // For now, just log it
    error_log("Cron: Retrying transaction {$transaction['reference']}");
    
    return true;
}

// -----------------------------------------------------------------
// WEB INTERFACE FOR MANUAL EXECUTION
// -----------------------------------------------------------------
if (php_sapi_name() !== 'cli') {
    // Running via web request
    
    // Security check
    $valid_token = 'your-cron-secret-token';
    if (!isset($_GET['token']) || $_GET['token'] !== $valid_token) {
        http_response_code(403);
        die('Access denied');
    }
    
    // Set headers for web output
    header('Content-Type: application/json');
    
    $results = [];
    
    // Run specified job
    if (isset($_GET['job'])) {
        switch ($_GET['job']) {
            case 'cleanup':
                $results = cleanupOldData();
                break;
            case 'notifications':
                $results = sendScheduledNotifications();
                break;
            case 'vendor_payouts':
                $results = processVendorPayouts();
                break;
            case 'reports':
                $results = generateDailyReports();
                break;
            case 'all':
                $results = [
                    'cleanup' => cleanupOldData(),
                    'notifications' => sendScheduledNotifications(),
                    'vendor_payouts' => processVendorPayouts(),
                    'reports' => generateDailyReports(),
                    'failed_transactions' => checkFailedTransactions(),
                    'exchange_rates' => updateExchangeRates(),
                    'backup' => backupDatabase()
                ];
                break;
        }
    }
    
    echo json_encode([
        'success' => true,
        'timestamp' => date('Y-m-d H:i:s'),
        'results' => $results
    ]);
}
?>