<?php
session_start();
require_once '../config/conn.php';
header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Session expired. Please login again.']);
    exit();
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

// Validate required fields
if (empty($data['transaction_id']) || empty($data['reference']) || empty($data['payment_reference'])) {
    echo json_encode(['success' => false, 'message' => 'Missing required data']);
    exit();
}

$student_id = $_SESSION['user_id'];
$transaction_id = intval($data['transaction_id']);
$reference = $conn->real_escape_string($data['reference']);
$payment_reference = $conn->real_escape_string($data['payment_reference']);

// Start transaction
$conn->begin_transaction();

try {
    // Verify transaction belongs to student and is pending
    $verify_stmt = $conn->prepare("
        SELECT wt.* 
        FROM withdrawal_transactions wt
        WHERE wt.id = ? AND wt.student_id = ? AND wt.status = 'pending'
        FOR UPDATE
    ");
    $verify_stmt->bind_param("ii", $transaction_id, $student_id);
    $verify_stmt->execute();
    $transaction = $verify_stmt->get_result()->fetch_assoc();
    
    if (!$transaction) {
        throw new Exception("Transaction not found or already processed");
    }
    
    // TODO: Integrate with actual payment gateway (Paystack, Flutterwave, etc.)
    // This is a simulation - replace with actual API call
    
    // Simulate successful payment processing
    $payment_successful = simulatePaymentProcessing($transaction);
    
    if ($payment_successful) {
        // Update transaction status
        $update_stmt = $conn->prepare("
            UPDATE withdrawal_transactions 
            SET status = 'processing', 
                payment_reference = ?,
                description = 'Withdrawal processing initiated',
                processed_at = NOW()
            WHERE id = ?
        ");
        $update_stmt->bind_param("si", $payment_reference, $transaction_id);
        $update_stmt->execute();
        
        // Log payment response
        $payment_response = json_encode([
            'status' => 'success',
            'message' => 'Payment queued for processing',
            'reference' => $payment_reference,
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        
        $log_stmt = $conn->prepare("
            UPDATE withdrawal_transactions 
            SET payment_response = ?
            WHERE id = ?
        ");
        $log_stmt->bind_param("si", $payment_response, $transaction_id);
        $log_stmt->execute();
        
        $conn->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Withdrawal processing initiated successfully',
            'transaction' => [
                'id' => $transaction_id,
                'reference' => $reference,
                'amount' => $transaction['amount'],
                'status' => 'processing',
                'processed_at' => date('Y-m-d H:i:s')
            ]
        ]);
        
    } else {
        // Payment failed - refund wallet
        $refund_amount = $transaction['total_deduction'];
        $refund_stmt = $conn->prepare("
            UPDATE student_wallets 
            SET balance = balance + ? 
            WHERE id = ? AND student_id = ?
        ");
        $refund_stmt->bind_param("dii", $refund_amount, $transaction['wallet_id'], $student_id);
        $refund_stmt->execute();
        
        // Update transaction status
        $fail_stmt = $conn->prepare("
            UPDATE withdrawal_transactions 
            SET status = 'failed', 
                description = 'Payment processing failed'
            WHERE id = ?
        ");
        $fail_stmt->bind_param("i", $transaction_id);
        $fail_stmt->execute();
        
        // Create refund transaction record
        $refund_audit_stmt = $conn->prepare("
            INSERT INTO wallet_transactions 
            (student_id, wallet_id, transaction_type, amount, balance_before, balance_after, reference, description)
            VALUES (?, ?, 'deposit', ?, ?, ?, ?, ?)
        ");
        
        // Get current balance for audit
        $balance_stmt = $conn->prepare("SELECT balance FROM student_wallets WHERE id = ?");
        $balance_stmt->bind_param("i", $transaction['wallet_id']);
        $balance_stmt->execute();
        $current_balance = $balance_stmt->get_result()->fetch_assoc()['balance'];
        
        $new_balance = $current_balance + $refund_amount;
        $refund_reference = 'REF-' . time() . '-' . strtoupper(uniqid());
        $description = "Refund for failed withdrawal: " . $reference;
        
        $refund_audit_stmt->bind_param(
            "iidddss", 
            $student_id, 
            $transaction['wallet_id'], 
            $refund_amount, 
            $current_balance, 
            $new_balance, 
            $refund_reference, 
            $description
        );
        $refund_audit_stmt->execute();
        
        $conn->commit();
        
        throw new Exception("Payment processing failed. Amount has been refunded to your wallet.");
    }
    
} catch (Exception $e) {
    $conn->rollback();
    error_log("Withdrawal completion failed: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => $e->getMessage()
    ]);
}

function simulatePaymentProcessing($transaction) {
    // Simulate random failures (10% failure rate for demo)
    $should_fail = rand(1, 10) === 1;
    
    if ($should_fail) {
        // Simulate payment gateway failure
        return false;
    }
    
    // Simulate API call delay
    usleep(500000); // 0.5 seconds
    
    // In real implementation, this would call Paystack/Flutterwave API
    // Example Paystack transfer recipient creation and transfer initiation
    
    return true;
}
?>