<?php
session_start();
require_once '../config/conn.php';
header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Session expired. Please login again.']);
    exit();
}

// Get POST data
$data = json_decode(file_get_contents('php://input'), true);

// Debug logging (remove in production)
error_log("Withdrawal initiation data: " . print_r($data, true));

// Validate required fields
$required_fields = ['amount', 'wallet_id', 'student_id', 'bank_account_id', 'transaction_pin'];
foreach ($required_fields as $field) {
    if (empty($data[$field])) {
        echo json_encode(['success' => false, 'message' => 'Missing required field: ' . $field]);
        exit();
    }
}

$student_id = $_SESSION['user_id'];
$amount = floatval($data['amount']);
$wallet_id = intval($data['wallet_id']);
$bank_account_id = intval($data['bank_account_id']);
$pin = $data['transaction_pin'];

// Verify that the request is for the logged-in user
if ($student_id != $data['student_id']) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit();
}

// Verify transaction PIN
$pin_stmt = $conn->prepare("SELECT pin_hash FROM wallet_pins WHERE student_id = ?");
$pin_stmt->bind_param("i", $student_id);
$pin_stmt->execute();
$pin_result = $pin_stmt->get_result();

if ($pin_result->num_rows === 0) {
    echo json_encode(['success' => false, 'message' => 'Transaction PIN not set']);
    exit();
}

$pin_data = $pin_result->fetch_assoc();
if (!password_verify($pin, $pin_data['pin_hash'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid transaction PIN']);
    exit();
}

// Get wallet balance
$wallet_stmt = $conn->prepare("SELECT balance FROM student_wallets WHERE id = ? AND student_id = ?");
$wallet_stmt->bind_param("ii", $wallet_id, $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();

if ($wallet_result->num_rows === 0) {
    echo json_encode(['success' => false, 'message' => 'Wallet not found']);
    exit();
}

$wallet = $wallet_result->fetch_assoc();
$balance = floatval($wallet['balance']);

// REMOVED FEE: No fee for withdrawals
$fee = 0.00; // No fee
$total_deduction = $amount; // Only deduct the withdrawal amount

// Check if balance is sufficient
if ($total_deduction > $balance) {
    echo json_encode(['success' => false, 'message' => 'Insufficient balance']);
    exit();
}

// Validate minimum withdrawal
if ($amount < 100) {
    echo json_encode(['success' => false, 'message' => 'Minimum withdrawal is ₦100']);
    exit();
}

// Verify bank account belongs to student
$bank_stmt = $conn->prepare("SELECT * FROM student_bank_accounts WHERE id = ? AND student_id = ?");
$bank_stmt->bind_param("ii", $bank_account_id, $student_id);
$bank_stmt->execute();
$bank_result = $bank_stmt->get_result();

if ($bank_result->num_rows === 0) {
    echo json_encode(['success' => false, 'message' => 'Bank account not found']);
    exit();
}

$bank_account = $bank_result->fetch_assoc();

// Generate unique references
$reference = 'WDL-' . time() . '-' . strtoupper(uniqid());
$payment_reference = 'PAY-' . time() . '-' . strtoupper(uniqid());

// Start transaction
$conn->begin_transaction();

try {
    // Create withdrawal transaction record
    $withdraw_stmt = $conn->prepare("
        INSERT INTO withdrawal_transactions 
        (student_id, wallet_id, bank_account_id, reference, payment_reference, amount, fee, total_deduction, status, attempted_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
    ");
    $withdraw_stmt->bind_param(
        "iiissddd", 
        $student_id, 
        $wallet_id, 
        $bank_account_id, 
        $reference, 
        $payment_reference, 
        $amount, 
        $fee, 
        $total_deduction
    );
    $withdraw_stmt->execute();
    $transaction_id = $conn->insert_id;
    
    // Create audit trail in wallet transactions
    $audit_stmt = $conn->prepare("
        INSERT INTO wallet_transactions 
        (student_id, wallet_id, transaction_type, amount, balance_before, balance_after, reference, description)
        VALUES (?, ?, 'withdrawal', ?, ?, ?, ?, ?)
    ");
    
    $balance_after = $balance - $total_deduction;
    $description = "Withdrawal to {$bank_account['bank_name']} - ****" . substr($bank_account['account_number'], -4);
    $audit_stmt->bind_param(
        "iidddss", 
        $student_id, 
        $wallet_id, 
        $total_deduction, 
        $balance, 
        $balance_after, 
        $reference, 
        $description
    );
    $audit_stmt->execute();
    
    // Deduct from wallet
    $update_wallet_stmt = $conn->prepare("
        UPDATE student_wallets 
        SET balance = balance - ? 
        WHERE id = ? AND student_id = ?
    ");
    $update_wallet_stmt->bind_param("dii", $total_deduction, $wallet_id, $student_id);
    $update_wallet_stmt->execute();
    
    // Check if rows were affected
    if ($update_wallet_stmt->affected_rows === 0) {
        throw new Exception("Failed to update wallet balance");
    }
    
    $conn->commit();
    
    // Get updated transaction details
    $transaction_stmt = $conn->prepare("
        SELECT wt.*, 
               ba.bank_name, 
               ba.account_number, 
               ba.account_name,
               ba.bank_code
        FROM withdrawal_transactions wt
        JOIN student_bank_accounts ba ON wt.bank_account_id = ba.id
        WHERE wt.id = ?
    ");
    $transaction_stmt->bind_param("i", $transaction_id);
    $transaction_stmt->execute();
    $transaction = $transaction_stmt->get_result()->fetch_assoc();
    
    echo json_encode([
        'success' => true,
        'message' => 'Withdrawal initiated successfully',
        'transaction_id' => $transaction_id,
        'reference' => $reference,
        'payment_reference' => $payment_reference,
        'transaction' => $transaction
    ]);
    
} catch (Exception $e) {
    $conn->rollback();
    error_log("Withdrawal initiation failed: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Failed to initiate withdrawal. Please try again.'
    ]);
}

// Ensure no extra output
exit();
?>