<?php
require_once __DIR__ . '/../../config/conn.php';
require_once __DIR__ . '/../../vendor/autoload.php'; // For Paystack SDK

use Yabacon\Paystack;

header('Content-Type: application/json');

// Get JSON input
$data = json_decode(file_get_contents('php://input'), true);

// Initialize Paystack
$paystack = new Paystack(PAYSTACK_SECRET_KEY);

try {
    // Step 1: Create Transfer Recipient
    $recipient = $paystack->transferrecipient->create([
        'type' => 'nuban',
        'name' => $data['account_name'],
        'account_number' => $data['account_number'],
        'bank_code' => $data['bank_code'],
        'currency' => 'NGN'
    ]);
    
    if (!$recipient->status) {
        throw new Exception('Failed to create recipient: ' . $recipient->message);
    }
    
    $recipient_code = $recipient->data->recipient_code;
    
    // Step 2: Initiate Transfer
    $transfer = $paystack->transfer->initiate([
        'source' => 'balance',
        'amount' => $data['amount'], // In kobo
        'recipient' => $recipient_code,
        'reason' => 'Wallet Withdrawal - Ref: ' . $data['reference']
    ]);
    
    if (!$transfer->status) {
        throw new Exception('Transfer failed: ' . $transfer->message);
    }
    
    // Update transaction with Paystack reference
    $update_stmt = $conn->prepare("
        UPDATE withdrawal_transactions 
        SET paystack_reference = ?, 
            paystack_recipient = ?,
            status = 'processing'
        WHERE id = ?
    ");
    $update_stmt->bind_param("ssi", $transfer->data->reference, $recipient_code, $data['transaction_id']);
    $update_stmt->execute();
    
    echo json_encode([
        'success' => true,
        'message' => 'Transfer initiated successfully',
        'reference' => $transfer->data->reference,
        'transfer_code' => $transfer->data->transfer_code,
        'recipient_code' => $recipient_code
    ]);
    
} catch (Exception $e) {
    // Update transaction status to failed
    if (isset($data['transaction_id'])) {
        $update_stmt = $conn->prepare("UPDATE withdrawal_transactions SET status = 'failed', failure_reason = ? WHERE id = ?");
        $error_message = substr($e->getMessage(), 0, 255); // Limit to column size
        $update_stmt->bind_param("si", $error_message, $data['transaction_id']);
        $update_stmt->execute();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Paystack transfer failed: ' . $e->getMessage()
    ]);
}
?>