<?php
// student/ajax/paystack-webhook.php

// Set the correct path to config
require_once __DIR__ . '/../../config/conn.php';

// Check if vendor autoload exists
$vendorAutoload = __DIR__ . '/../../vendor/autoload.php';
if (file_exists($vendorAutoload)) {
    require_once $vendorAutoload;
} else {
    // Fallback: Try to include Paystack SDK manually
    $paystackPath = __DIR__ . '/../../vendor/yabacon/paystack-php/src/';
    if (file_exists($paystackPath . 'Paystack.php')) {
        require_once $paystackPath . 'Paystack.php';
        require_once $paystackPath . 'Exception/PaystackException.php';
        // Add other required files as needed
    } else {
        error_log("Paystack SDK not found. Please run: composer require yabacon/paystack-php");
        http_response_code(500);
        echo json_encode(['error' => 'Payment processor not configured']);
        exit();
    }
}

// Log webhook access for debugging
error_log("Paystack webhook accessed: " . date('Y-m-d H:i:s'));

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error_log("Webhook: Method not allowed - " . $_SERVER['REQUEST_METHOD']);
    http_response_code(405);
    exit();
}

// Get input
$input = file_get_contents('php://input');
error_log("Webhook raw input: " . $input);

// Verify signature if in production
$signature = $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] ?? '';
if (empty($signature)) {
    error_log("Webhook: No signature provided");
    // In production, require signature
    // http_response_code(401);
    // exit();
}

// Get Paystack secret key from config
// Make sure this is defined in your config/conn.php
if (!defined('PAYSTACK_SECRET_KEY')) {
    error_log("Webhook: PAYSTACK_SECRET_KEY not defined");
    http_response_code(500);
    exit();
}

$secret = PAYSTACK_SECRET_KEY;

// Verify signature (for production)
if (!empty($signature) && !empty($secret)) {
    $computedSignature = hash_hmac('sha512', $input, $secret);
    
    if (!hash_equals($signature, $computedSignature)) {
        error_log("Webhook: Invalid signature");
        http_response_code(401);
        exit();
    }
}

// Parse event
$event = json_decode($input);
if (!$event) {
    error_log("Webhook: Invalid JSON");
    http_response_code(400);
    exit();
}

// Log event type
error_log("Webhook event: " . ($event->event ?? 'unknown'));

// Initialize database connection
if (!$conn) {
    error_log("Webhook: Database connection failed");
    http_response_code(500);
    exit();
}

// Handle transfer.success event
if ($event->event === 'transfer.success') {
    $transfer = $event->data;
    
    error_log("Transfer success: " . ($transfer->reference ?? 'no reference'));
    
    // Update transaction status
    $stmt = $conn->prepare("
        UPDATE withdrawal_transactions 
        SET status = 'completed',
            paystack_status = 'success',
            completed_at = NOW()
        WHERE paystack_reference = ?
    ");
    
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        exit();
    }
    
    $stmt->bind_param("s", $transfer->reference);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
    }
    $stmt->close();
    
    // Also update wallet_transactions if needed
    $stmt2 = $conn->prepare("
        UPDATE wallet_transactions 
        SET status = 'completed'
        WHERE reference = (
            SELECT reference FROM withdrawal_transactions 
            WHERE paystack_reference = ? LIMIT 1
        )
    ");
    if ($stmt2) {
        $stmt2->bind_param("s", $transfer->reference);
        $stmt2->execute();
        $stmt2->close();
    }
    
    error_log("Transfer marked as completed: " . $transfer->reference);
    
} elseif ($event->event === 'transfer.failed') {
    $transfer = $event->data;
    
    error_log("Transfer failed: " . ($transfer->reference ?? 'no reference') . " - " . ($transfer->failure_reason ?? 'no reason'));
    
    // Update transaction status
    $stmt = $conn->prepare("
        UPDATE withdrawal_transactions 
        SET status = 'failed',
            paystack_status = 'failed',
            failure_reason = ?,
            completed_at = NOW()
        WHERE paystack_reference = ?
    ");
    
    if (!$stmt) {
        error_log("Prepare failed: " . $conn->error);
        http_response_code(500);
        exit();
    }
    
    $failure_reason = substr($transfer->failure_reason ?? 'Transfer failed', 0, 255);
    $stmt->bind_param("ss", $failure_reason, $transfer->reference);
    if (!$stmt->execute()) {
        error_log("Execute failed: " . $stmt->error);
    }
    $stmt->close();
    
    // Refund wallet
    $refund_stmt = $conn->prepare("
        SELECT wt.student_id, wt.wallet_id, wt.total_amount 
        FROM withdrawal_transactions wt
        WHERE wt.paystack_reference = ?
    ");
    
    if ($refund_stmt) {
        $refund_stmt->bind_param("s", $transfer->reference);
        $refund_stmt->execute();
        $result = $refund_stmt->get_result();
        
        if ($transaction = $result->fetch_assoc()) {
            // Refund to wallet
            $update_wallet = $conn->prepare("UPDATE student_wallets SET balance = balance + ? WHERE id = ?");
            if ($update_wallet) {
                $update_wallet->bind_param("di", $transaction['total_amount'], $transaction['wallet_id']);
                $update_wallet->execute();
                $update_wallet->close();
            }
            
            // Record refund transaction
            $refund_ref = 'REF' . time() . mt_rand(1000, 9999);
            $refund_desc = "Refund for failed withdrawal";
            
            $refund_trans = $conn->prepare("
                INSERT INTO wallet_transactions (
                    student_id,
                    wallet_id,
                    amount, 
                    transaction_type, 
                    description, 
                    status, 
                    reference,
                    created_at
                ) VALUES (?, ?, ?, 'credit', ?, 'completed', ?, NOW())
            ");
            
            if ($refund_trans) {
                $refund_trans->bind_param("iidss", 
                    $transaction['student_id'], 
                    $transaction['wallet_id'], 
                    $transaction['total_amount'], 
                    $refund_desc, 
                    $refund_ref
                );
                $refund_trans->execute();
                $refund_trans->close();
            }
            
            error_log("Refund processed for failed transfer: " . $transfer->reference);
        }
        $refund_stmt->close();
    }
    
    error_log("Transfer marked as failed: " . $transfer->reference);
    
} else {
    // Log other events for debugging
    error_log("Unhandled webhook event: " . $event->event);
}

http_response_code(200);
echo json_encode(['status' => 'success', 'message' => 'Webhook processed']);
?>