<?php
// ajax/purchase-product.php
session_start();
require_once __DIR__ . '/../../config/conn.php';

// Check if request is AJAX
if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) != 'xmlhttprequest') {
    http_response_code(403);
    die(json_encode(['success' => false, 'message' => 'Direct access not allowed']));
}

// Check session
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    die(json_encode(['success' => false, 'message' => 'Unauthorized']));
}

$student_id = $_SESSION['user_id'];

// Get POST data
$vendor_id = isset($_POST['vendor_id']) ? intval($_POST['vendor_id']) : 0;
$product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
$quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 0;
$special_instructions = isset($_POST['special_instructions']) ? trim($_POST['special_instructions']) : '';
$pin = isset($_POST['pin']) ? trim($_POST['pin']) : '';

// Validate inputs
if ($vendor_id <= 0 || $product_id <= 0) {
    die(json_encode(['success' => false, 'message' => 'Invalid vendor or product ID']));
}

if ($quantity <= 0) {
    die(json_encode(['success' => false, 'message' => 'Quantity must be greater than 0']));
}

if (empty($pin)) {
    die(json_encode(['success' => false, 'message' => 'PIN is required']));
}

// Get vendor details
$vendor_query = "SELECT * FROM vendors WHERE id = '$vendor_id'";
$vendor_result = mysqli_query($conn, $vendor_query);
$vendor = mysqli_fetch_assoc($vendor_result);

if (!$vendor) {
    die(json_encode(['success' => false, 'message' => 'Vendor not found']));
}

// Get product details
$product_query = "SELECT * FROM products WHERE id = '$product_id' AND vendor_id = '$vendor_id'";
$product_result = mysqli_query($conn, $product_query);
$product = mysqli_fetch_assoc($product_result);

if (!$product) {
    die(json_encode(['success' => false, 'message' => 'Product not found']));
}

// Check if product is in stock
if ($product['stock_status'] != 'in_stock') {
    die(json_encode(['success' => false, 'message' => 'This product is out of stock']));
}

// Get student wallet
$wallet_query = "SELECT * FROM student_wallets WHERE student_id = '$student_id'";
$wallet_result = mysqli_query($conn, $wallet_query);
$student_wallet = mysqli_fetch_assoc($wallet_result);

if (!$student_wallet) {
    die(json_encode(['success' => false, 'message' => 'Student wallet not found']));
}

// Get student PIN
$pin_query = "SELECT * FROM wallet_pins WHERE student_id = '$student_id'";
$pin_result = mysqli_query($conn, $pin_query);
$wallet_pin = mysqli_fetch_assoc($pin_result);

// Verify PIN
if (!$wallet_pin) {
    die(json_encode(['success' => false, 'message' => 'PIN not set. Please set your PIN first.']));
}

if (!password_verify($pin, $wallet_pin['pin_hash'])) {
    die(json_encode(['success' => false, 'message' => 'Invalid PIN']));
}

// Get vendor wallet
$vendor_wallet_query = "SELECT * FROM vendor_wallets WHERE vendor_id = '$vendor_id'";
$vendor_wallet_result = mysqli_query($conn, $vendor_wallet_query);
$vendor_wallet = mysqli_fetch_assoc($vendor_wallet_result);

// If vendor wallet doesn't exist, create one
if (!$vendor_wallet) {
    $create_vendor_wallet = "INSERT INTO vendor_wallets (vendor_id, balance, pending_balance, created_at, updated_at) VALUES ('$vendor_id', 0, 0, NOW(), NOW())";
    if (mysqli_query($conn, $create_vendor_wallet)) {
        $vendor_wallet_result = mysqli_query($conn, $vendor_wallet_query);
        $vendor_wallet = mysqli_fetch_assoc($vendor_wallet_result);
    } else {
        die(json_encode(['success' => false, 'message' => 'Failed to create vendor wallet']));
    }
}

// Calculate total amount
$total_amount = $product['price'] * $quantity;

// Check student balance
if ($total_amount > $student_wallet['balance']) {
    die(json_encode(['success' => false, 'message' => 'Insufficient wallet balance']));
}

// Start transaction
mysqli_begin_transaction($conn);

try {
    // STUDENT TRANSACTION
    $student_balance_before = $student_wallet['balance'];
    $student_balance_after = $student_balance_before - $total_amount;
    
    // VENDOR TRANSACTION - Money goes to pending/unapproved
    $vendor_balance_before = $vendor_wallet['balance'];
    $vendor_pending_before = $vendor_wallet['pending_balance'] ?? 0;
    $vendor_pending_after = $vendor_pending_before + $total_amount;
    
    $student_reference = 'PUR-' . time() . '-' . $student_id;
    $vendor_reference = 'SALE-' . time() . '-' . $vendor_id;
    
    // Update student wallet balance
    $update_student = "UPDATE student_wallets SET balance = '$student_balance_after', updated_at = NOW() WHERE student_id = '$student_id'";
    if (!mysqli_query($conn, $update_student)) {
        throw new Exception("Failed to update student wallet");
    }
    
    // Update vendor wallet - add to pending balance only
    $update_vendor = "UPDATE vendor_wallets SET pending_balance = '$vendor_pending_after', updated_at = NOW() WHERE vendor_id = '$vendor_id'";
    if (!mysqli_query($conn, $update_vendor)) {
        throw new Exception("Failed to update vendor wallet");
    }
    
    // Record student wallet transaction
    $student_notes = "Purchase from " . mysqli_real_escape_string($conn, $vendor['business_name']) . ": " . mysqli_real_escape_string($conn, $product['name']) . " (Qty: {$quantity})";
    
    // Check if balance_before column exists
    $check_balance_columns = mysqli_query($conn, "SHOW COLUMNS FROM wallet_transactions LIKE 'balance_before'");
    $has_balance_columns = mysqli_num_rows($check_balance_columns) > 0;
    
    if ($has_balance_columns) {
        $student_transaction = "INSERT INTO wallet_transactions 
            (student_id, wallet_id, amount, transaction_type, reference, status, description, balance_before, balance_after, created_at) 
            VALUES ('$student_id', '{$student_wallet['id']}', '$total_amount', 'debit', '$student_reference', 'completed', '$student_notes', '$student_balance_before', '$student_balance_after', NOW())";
    } else {
        $student_transaction = "INSERT INTO wallet_transactions 
            (student_id, wallet_id, amount, transaction_type, reference, status, description, created_at) 
            VALUES ('$student_id', '{$student_wallet['id']}', '$total_amount', 'debit', '$student_reference', 'completed', '$student_notes', NOW())";
    }
    
    if (!mysqli_query($conn, $student_transaction)) {
        throw new Exception("Failed to record student transaction");
    }
    
    // Record vendor wallet transaction - status is 'pending' until admin approves
    $vendor_description = "Sale to student ID {$student_id}: " . mysqli_real_escape_string($conn, $product['name']) . " (Qty: {$quantity})";
    $vendor_notes = "";
    if (!empty($special_instructions)) {
        $vendor_notes = "Instructions: " . mysqli_real_escape_string($conn, $special_instructions);
    }
    
    // Record purchase order if table exists
    $check_purchase_orders = mysqli_query($conn, "SHOW TABLES LIKE 'purchase_orders'");
    if (mysqli_num_rows($check_purchase_orders) > 0) {
        $product_name_escaped = mysqli_real_escape_string($conn, $product['name']);
        $special_instructions_escaped = mysqli_real_escape_string($conn, $special_instructions);
        
        $purchase_order = "INSERT INTO purchase_orders 
            (student_id, vendor_id, product_name, unit_price, quantity, total_amount, special_instructions, status, reference, created_at) 
            VALUES ('$student_id', '$vendor_id', '$product_name_escaped', '{$product['price']}', '$quantity', '$total_amount', '$special_instructions_escaped', 'completed', '$student_reference', NOW())";
        
        if (!mysqli_query($conn, $purchase_order)) {
            error_log("Failed to record purchase order (non-critical)");
        }
    }
    
    // Commit transaction
    mysqli_commit($conn);
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Purchase completed successfully!',
        'reference' => $student_reference,
        'new_balance' => $student_balance_after
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    mysqli_rollback($conn);
    
    // Log error for debugging
    error_log('Purchase error: ' . $e->getMessage());
    
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}