<?php
// ajax/search-products.php
session_start();

// Adjust the path based on your directory structure
// Since this file is in /student/ajax/, we need to go up two levels to reach /config/
require_once __DIR__ . '/../../config/conn.php';

// Check if request is AJAX (optional but recommended)
if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
    // This is an AJAX request
} else {
    // Optionally redirect or show error for non-AJAX requests
    // http_response_code(403);
    // die('Direct access not allowed');
}

// Check session
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    die(json_encode(['error' => 'Unauthorized']));
}

// Get search parameters
$search = isset($_POST['search']) ? trim(mysqli_real_escape_string($conn, $_POST['search'])) : '';
$vendor_id = isset($_POST['vendor_id']) ? intval($_POST['vendor_id']) : 0;

if ($vendor_id <= 0) {
    die(json_encode(['error' => 'Invalid vendor ID']));
}

// Get vendor details
$vendor_query = "SELECT * FROM vendors WHERE id = '$vendor_id'";
$vendor_result = mysqli_query($conn, $vendor_query);
$vendor = mysqli_fetch_assoc($vendor_result);

if (!$vendor) {
    die(json_encode(['error' => 'Vendor not found']));
}

// Build query
$where_clause = "vendor_id = '$vendor_id' AND stock_status = 'in_stock'";
if (!empty($search)) {
    $where_clause .= " AND (name LIKE '%$search%' OR description LIKE '%$search%')";
}

$products_query = "SELECT * FROM products WHERE $where_clause ORDER BY created_at DESC";
$products_result = mysqli_query($conn, $products_query);

// Check for errors
if (!$products_result) {
    die(json_encode(['error' => 'Error fetching products: ' . mysqli_error($conn)]));
}

$product_count = mysqli_num_rows($products_result);
?>

<?php if ($product_count > 0): ?>
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6" id="productsGrid">
        <?php while ($product = mysqli_fetch_assoc($products_result)): ?>
        <div class="bg-white dark:bg-gray-800 overflow-hidden hover:shadow-lg transition-duration-300">
            <!-- Product Image -->
            <div class="h-48 bg-gray-200 dark:bg-gray-700 relative">
                <?php if (!empty($product['image_path'])): ?>
                    <img src="../<?= htmlspecialchars($product['image_path']) ?>" 
                         alt="<?= htmlspecialchars($product['name']) ?>"
                         class="w-full h-full object-cover">
                <?php else: ?>
                    <div class="w-full h-full flex items-center justify-center">
                        <i class="fas fa-box text-4xl text-gray-400 dark:text-gray-500"></i>
                    </div>
                <?php endif; ?>
                
                <!-- Stock Status Badge -->
                <span class="absolute top-2 right-2 px-2 py-1 text-xs font-semibold rounded-full 
                    <?= $product['stock_status'] == 'in_stock' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200' ?>">
                    <?= $product['stock_status'] == 'in_stock' ? 'In Stock' : 'Out of Stock' ?>
                </span>
            </div>
            
            <!-- Product Details -->
            <div class="p-4">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2 truncate">
                    <?= htmlspecialchars($product['name']) ?>
                </h3>
                
                <p class="text-gray-600 dark:text-gray-400 text-sm mb-3 line-clamp-2">
                    <?= htmlspecialchars($product['description']) ?>
                </p>
                
                <div class="flex items-center justify-between mb-4">
                    <span class="text-2xl font-bold text-yellow-600 dark:text-yellow-400">
                        ₦<?= number_format($product['price'], 2) ?>
                    </span>
                    
                    <!-- Buy Button -->
                    <?php if ($product['stock_status'] == 'in_stock'): ?>
                        <a href="buy-product.php?vendor_id=<?= $vendor_id ?>&product_id=<?= $product['id'] ?>" 
                           class="bg-yellow-500 text-white py-2 px-4 rounded-md hover:bg-yellow-600 focus:outline-none focus:ring-2 focus:ring-yellow-500 transition duration-200 font-medium">
                            <i class="fas fa-shopping-cart mr-2"></i> Buy Now
                        </a>
                    <?php else: ?>
                        <button disabled class="bg-gray-300 dark:bg-gray-700 text-gray-500 dark:text-gray-400 py-2 px-4 rounded-md cursor-not-allowed">
                            Out of Stock
                        </button>
                    <?php endif; ?>
                </div>
                
                <!-- Quick Info -->
                <div class="flex items-center justify-between text-sm text-gray-500 dark:text-gray-400 border-t dark:border-gray-700 pt-3">
                    <span class="flex items-center">
                        <i class="fas fa-store mr-1"></i>
                        <?= htmlspecialchars($vendor['business_name']) ?>
                    </span>
                    <?php if (!empty($product['created_at'])): ?>
                    <span>
                        Added: <?= date('M d, Y', strtotime($product['created_at'])) ?>
                    </span>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endwhile; ?>
    </div>

    <!-- Product Count -->
    <div class="mt-6 text-center text-gray-600 dark:text-gray-400">
        <p id="productCount">Showing <?= $product_count ?> product<?= $product_count > 1 ? 's' : '' ?></p>
    </div>
<?php else: ?>
    <!-- No Products Message -->
    <div class="bg-white dark:bg-gray-800 p-12 text-center">
        <i class="fas fa-box-open text-4xl text-gray-400 dark:text-gray-500 mb-4"></i>
        <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">No products found</h3>
        <p class="text-gray-600 dark:text-gray-400 mb-4">
            <?= !empty($search) ? 'Try a different search term.' : 'This vendor has no products available at the moment.' ?>
        </p>
        <?php if (!empty($search)): ?>
            <button onclick="clearSearch()" class="text-yellow-600 hover:text-yellow-800 dark:text-yellow-400 dark:hover:text-yellow-300">
                <i class="fas fa-times mr-1"></i> Clear search and show all products
            </button>
        <?php endif; ?>
    </div>
<?php endif; ?>

<style>
.line-clamp-2 {
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    overflow: hidden;
}
</style>