<?php
session_start();
require_once '../config/conn.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Session expired. Please login again.']);
    exit();
}

$student_id = $_SESSION['user_id'];

// Get action
$action = $_POST['action'] ?? '';

switch ($action) {
    case 'get_products':
        getProducts($conn, $student_id);
        break;
    
    case 'get_product_details':
        getProductDetails($conn);
        break;
    
    case 'purchase_product':
        purchaseProduct($conn, $student_id);
        break;
    
    case 'get_wallet_balance':
        getWalletBalance($conn, $student_id);
        break;
    
    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action.']);
        break;
}

function getProducts($conn, $student_id) {
    $vendor_id = intval($_POST['vendor_id'] ?? 0);
    $search = trim($_POST['search'] ?? '');
    
    if (!$vendor_id) {
        echo json_encode(['success' => false, 'message' => 'Vendor ID is required.']);
        return;
    }
    
    // Build query
    $query = "SELECT p.* FROM products p WHERE p.vendor_id = ? AND p.stock_status = 'in_stock'";
    
    if (!empty($search)) {
        $search_term = "%$search%";
        $query .= " AND (p.name LIKE ? OR p.description LIKE ?)";
    }
    
    $query .= " ORDER BY p.created_at DESC";
    
    $stmt = mysqli_prepare($conn, $query);
    
    if (!empty($search)) {
        mysqli_stmt_bind_param($stmt, "iss", $vendor_id, $search_term, $search_term);
    } else {
        mysqli_stmt_bind_param($stmt, "i", $vendor_id);
    }
    
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    $products = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $products[] = $row;
    }
    
    echo json_encode([
        'success' => true,
        'products' => $products,
        'count' => count($products)
    ]);
}

function getProductDetails($conn) {
    $product_id = intval($_POST['product_id'] ?? 0);
    
    if (!$product_id) {
        echo json_encode(['success' => false, 'message' => 'Product ID is required.']);
        return;
    }
    
    $query = "SELECT * FROM products WHERE id = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "i", $product_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($product = mysqli_fetch_assoc($result)) {
        echo json_encode([
            'success' => true,
            'product' => $product
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Product not found.']);
    }
}

function purchaseProduct($conn, $student_id) {
    // Validate inputs
    $product_id = intval($_POST['product_id'] ?? 0);
    $vendor_id = intval($_POST['vendor_id'] ?? 0);
    $quantity = intval($_POST['quantity'] ?? 1);
    $instructions = trim($_POST['instructions'] ?? '');
    
    if (!$product_id || !$vendor_id || $quantity < 1) {
        echo json_encode(['success' => false, 'message' => 'Invalid input data.']);
        return;
    }
    
    // Start transaction
    mysqli_begin_transaction($conn);
    
    try {
        // Get product details
        $product_query = "SELECT * FROM products WHERE id = ? FOR UPDATE";
        $stmt = mysqli_prepare($conn, $product_query);
        mysqli_stmt_bind_param($stmt, "i", $product_id);
        mysqli_stmt_execute($stmt);
        $product_result = mysqli_stmt_get_result($stmt);
        $product = mysqli_fetch_assoc($product_result);
        
        if (!$product) {
            throw new Exception('Product not found.');
        }
        
        if ($product['stock_status'] != 'in_stock') {
            throw new Exception('Product is out of stock.');
        }
        
        // Check wallet balance
        $wallet_query = "SELECT balance FROM student_wallets WHERE student_id = ? FOR UPDATE";
        $stmt = mysqli_prepare($conn, $wallet_query);
        mysqli_stmt_bind_param($stmt, "i", $student_id);
        mysqli_stmt_execute($stmt);
        $wallet_result = mysqli_stmt_get_result($stmt);
        $wallet = mysqli_fetch_assoc($wallet_result);
        
        if (!$wallet) {
            throw new Exception('Wallet not found.');
        }
        
        $total_amount = $product['price'] * $quantity;
        
        if ($wallet['balance'] < $total_amount) {
            throw new Exception('Insufficient balance.');
        }
        
        // Generate reference
        $reference = 'PUR' . date('YmdHis') . rand(1000, 9999);
        
        // Update wallet balance
        $new_balance = $wallet['balance'] - $total_amount;
        $update_wallet = "UPDATE student_wallets SET balance = ? WHERE student_id = ?";
        $stmt = mysqli_prepare($conn, $update_wallet);
        mysqli_stmt_bind_param($stmt, "di", $new_balance, $student_id);
        mysqli_stmt_execute($stmt);
        
        // Record wallet transaction
        $transaction_query = "INSERT INTO wallet_transactions 
                             (student_id, amount, transaction_type, reference, status, description) 
                             VALUES (?, ?, 'debit', ?, 'completed', ?)";
        $description = "Purchase: {$product['name']} x{$quantity}";
        $stmt = mysqli_prepare($conn, $transaction_query);
        mysqli_stmt_bind_param($stmt, "idss", $student_id, $total_amount, $reference, $description);
        mysqli_stmt_execute($stmt);
        
        // Create purchase order
        $order_query = "INSERT INTO purchase_orders 
                       (student_id, vendor_id, product_id, product_name, quantity, unit_price, total_amount, special_instructions, reference) 
                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = mysqli_prepare($conn, $order_query);
        mysqli_stmt_bind_param($stmt, "iiisiddss", 
            $student_id, $vendor_id, $product_id, $product['name'], $quantity, 
            $product['price'], $total_amount, $instructions, $reference);
        mysqli_stmt_execute($stmt);
        
        // Update product stock if needed (optional)
        // $update_stock = "UPDATE products SET stock_quantity = stock_quantity - ? WHERE id = ?";
        // $stmt = mysqli_prepare($conn, $update_stock);
        // mysqli_stmt_bind_param($stmt, "ii", $quantity, $product_id);
        // mysqli_stmt_execute($stmt);
        
        // Commit transaction
        mysqli_commit($conn);
        
        echo json_encode([
            'success' => true,
            'message' => 'Purchase completed successfully!',
            'reference' => $reference,
            'amount' => $total_amount
        ]);
        
    } catch (Exception $e) {
        // Rollback transaction on error
        mysqli_rollback($conn);
        
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}

function getWalletBalance($conn, $student_id) {
    $query = "SELECT balance FROM student_wallets WHERE student_id = ?";
    $stmt = mysqli_prepare($conn, $query);
    mysqli_stmt_bind_param($stmt, "i", $student_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    
    if ($wallet = mysqli_fetch_assoc($result)) {
        echo json_encode([
            'success' => true,
            'balance' => $wallet['balance']
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'balance' => 0
        ]);
    }
}
?>