<?php
session_start();
require_once '../config/conn.php';

header('Content-Type: application/json');

$response = ['success' => false, 'message' => '', 'errors' => []];

try {
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('Session expired. Please login again.');
    }

    $student_id = $_SESSION['user_id'];
    $wallet_id = intval($_POST['wallet_id'] ?? 0);
    $bank_account_id = intval($_POST['bank_account_id'] ?? 0);
    $amount = floatval($_POST['amount'] ?? 0);
    $transaction_pin = $_POST['transaction_pin'] ?? '';

    // Validation
    if ($amount < 100) {
        $response['errors']['amount'] = 'Minimum withdrawal amount is ₦100';
    }

    if (empty($bank_account_id)) {
        $response['errors']['bank_account_id'] = 'Please select a bank account';
    }

    if (empty($transaction_pin) || strlen($transaction_pin) !== 4 || !ctype_digit($transaction_pin)) {
        $response['errors']['transaction_pin'] = 'PIN must be exactly 4 digits';
    }

    if (!empty($response['errors'])) {
        $response['message'] = 'Please fix the errors below.';
        echo json_encode($response);
        exit;
    }

    // Verify wallet ownership
    $wallet_stmt = $conn->prepare("SELECT balance FROM student_wallets WHERE id = ? AND student_id = ?");
    $wallet_stmt->bind_param("ii", $wallet_id, $student_id);
    $wallet_stmt->execute();
    $wallet_result = $wallet_stmt->get_result();
    $wallet = $wallet_result->fetch_assoc();

    if (!$wallet) {
        throw new Exception('Wallet not found.');
    }

    // Check balance
    if ($wallet['balance'] < $amount) {
        throw new Exception('Insufficient balance.');
    }

    // Verify bank account ownership
    $bank_stmt = $conn->prepare("SELECT * FROM student_bank_accounts WHERE id = ? AND student_id = ?");
    $bank_stmt->bind_param("ii", $bank_account_id, $student_id);
    $bank_stmt->execute();
    $bank_result = $bank_stmt->get_result();
    $bank_account = $bank_result->fetch_assoc();

    if (!$bank_account) {
        throw new Exception('Bank account not found.');
    }

    // Verify transaction PIN
    $pin_stmt = $conn->prepare("SELECT pin_hash FROM wallet_pins WHERE student_id = ?");
    $pin_stmt->bind_param("i", $student_id);
    $pin_stmt->execute();
    $pin_result = $pin_stmt->get_result();
    $pin_data = $pin_result->fetch_assoc();

    if (!$pin_data || !password_verify($transaction_pin, $pin_data['pin_hash'])) {
        throw new Exception('Invalid transaction PIN.');
    }

    $conn->begin_transaction();

    // Generate unique reference
    $reference = 'WTH_' . $student_id . '_' . time();

    // Create withdrawal transaction
    $txn_stmt = $conn->prepare("INSERT INTO wallet_transactions 
        (student_id, wallet_id, bank_account_id, amount, transaction_type, reference, status, description, gateway, created_at) 
        VALUES (?, ?, ?, ?, 'withdrawal', ?, 'pending', 'Withdrawal to bank', 'bank_transfer', NOW())");
    $txn_stmt->bind_param("iiids", $student_id, $wallet_id, $bank_account_id, $amount, $reference);

    if (!$txn_stmt->execute()) {
        throw new Exception('Failed to create withdrawal request.');
    }

    // For immediate processing (if you want to deduct balance immediately)
    // $update_stmt = $conn->prepare("UPDATE student_wallets SET balance = balance - ? WHERE id = ?");
    // $update_stmt->bind_param("di", $amount, $wallet_id);
    // $update_stmt->execute();

    $conn->commit();

    $response['success'] = true;
    $response['message'] = 'Withdrawal request submitted successfully! It will be processed within 1-3 business days.';
    $response['reference'] = $reference;

} catch (Exception $e) {
    if (isset($conn) && $conn) {
        $conn->rollback();
    }
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>