<?php
require_once '../config/conn.php';
session_start();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit();
}

$student_id = $_SESSION['user_id'];
$account_id = $_POST['account_id'] ?? '';

try {
    // Get account details
    $stmt = $conn->prepare("
        SELECT account_number, bank_code, bank_name 
        FROM student_bank_accounts 
        WHERE id = ? AND student_id = ?
    ");
    $stmt->bind_param("ii", $account_id, $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $account = $result->fetch_assoc();
    
    if (!$account) {
        echo json_encode(['success' => false, 'message' => 'Account not found']);
        exit();
    }
    
    // Re-verify with Paystack
    $paystack_secret_key = getenv('PAYSTACK_SECRET_KEY') ?: (function() {
        if (file_exists('../config/paystack.php')) {
            include '../config/paystack.php';
            return PAYSTACK_SECRET_KEY ?? '';
        }
        return '';
    })();
    
    if (empty($paystack_secret_key)) {
        echo json_encode(['success' => false, 'message' => 'Paystack not configured']);
        exit();
    }
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.paystack.co/bank/resolve?account_number=' . urlencode($account['account_number']) . '&bank_code=' . urlencode($account['bank_code']));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $paystack_secret_key
    ]);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    $data = json_decode($response, true);
    
    $is_verified = $data['status'] ? 1 : 2;
    $account_name = $data['status'] ? $data['data']['account_name'] : $account['account_name'];
    
    // Update verification status
    $update_stmt = $conn->prepare("
        UPDATE student_bank_accounts 
        SET is_verified = ?, account_name = ?, updated_at = CURRENT_TIMESTAMP 
        WHERE id = ?
    ");
    $update_stmt->bind_param("isi", $is_verified, $account_name, $account_id);
    
    if ($update_stmt->execute()) {
        echo json_encode([
            'success' => true,
            'message' => $data['status'] ? 'Account verified successfully!' : 'Verification failed. Please check your account details.',
            'verified' => $data['status']
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update verification status']);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>