<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require_once '../config/conn.php';

header('Content-Type: application/json');

// Initialize response array
$response = [
    'success' => false,
    'message' => '',
    'errors' => [],
    'redirect' => ''
];

try {
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('Session expired. Please login again.');
    }

    $student_id = $_SESSION['user_id'];

    // Validate required fields
    $required_fields = ['full_name'];
    foreach ($required_fields as $field) {
        if (empty($_POST[$field])) {
            $response['errors'][$field] = 'This field is required.';
        }
    }

    // Get and sanitize input data
    $full_name = trim($_POST['full_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $gender = $_POST['gender'] ?? '';
    $dob = $_POST['dob'] ?? '';
    $nationality = trim($_POST['nationality'] ?? '');
    $faculty_id = $_POST['faculty_id'] ?? '';
    $department_id = $_POST['department_id'] ?? '';
    $program = trim($_POST['program'] ?? '');
    $year_of_study = $_POST['year_of_study'] ?? '';
    $semester = $_POST['semester'] ?? '';

    // Validate full name
    if (!empty($full_name) && !preg_match('/^[a-zA-Z\s]{2,50}$/', $full_name)) {
        $response['errors']['full_name'] = 'Please enter a valid name (letters and spaces only, 2-50 characters).';
    }

    // Validate phone number
    if (!empty($phone) && !preg_match('/^[\d\s\-\+\(\)]{10,15}$/', $phone)) {
        $response['errors']['phone'] = 'Please enter a valid phone number.';
    }

    // Validate date of birth
    if (!empty($dob)) {
        $dob_date = new DateTime($dob);
        $today = new DateTime();
        if ($dob_date > $today) {
            $response['errors']['dob'] = 'Date of birth cannot be in the future.';
        }
    }

    // If there are validation errors, return them
    if (!empty($response['errors'])) {
        $response['message'] = 'Please fix the errors below.';
        echo json_encode($response);
        exit;
    }

    // Check if student exists
    $check_stmt = $conn->prepare("SELECT id FROM students WHERE id = ?");
    $check_stmt->bind_param("i", $student_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();

    if ($check_result->num_rows === 0) {
        throw new Exception('Student record not found.');
    }

    // Update student profile
    $update_stmt = $conn->prepare("UPDATE students SET full_name=?, phone=?, gender=?, dob=?, nationality=?, faculty_id=?, department_id=?, program=?, year_of_study=?, semester=?, updated_at=NOW() WHERE id=?");
    
    if (!$update_stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }

    // Handle empty values for optional fields
    $faculty_id = empty($faculty_id) ? null : $faculty_id;
    $department_id = empty($department_id) ? null : $department_id;
    $year_of_study = empty($year_of_study) ? null : $year_of_study;
    $semester = empty($semester) ? null : $semester;

    $update_stmt->bind_param("ssssssssssi", 
        $full_name, 
        $phone, 
        $gender, 
        $dob, 
        $nationality, 
        $faculty_id, 
        $department_id, 
        $program, 
        $year_of_study, 
        $semester, 
        $student_id
    );

    if ($update_stmt->execute()) {
        $response['success'] = true;
        $response['message'] = 'Profile updated successfully!';
        
        // Optionally redirect to profile page
        // $response['redirect'] = 'profile.php';
        
    } else {
        throw new Exception('Failed to update profile: ' . $update_stmt->error);
    }

} catch (Exception $e) {
    error_log("Profile update error: " . $e->getMessage());
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>