<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require_once '../config/conn.php';

header('Content-Type: application/json');

// Initialize response array
$response = [
    'success' => false,
    'message' => '',
    'errors' => [],
    'new_image_url' => ''
];

try {
    // Check if user is logged in
    if (!isset($_SESSION['user_id'])) {
        throw new Exception('Session expired. Please login again.');
    }

    $student_id = $_SESSION['user_id'];

    // Check if file was uploaded
    if (!isset($_FILES['profile_picture']) || $_FILES['profile_picture']['error'] === UPLOAD_ERR_NO_FILE) {
        $response['errors']['file'] = 'Please select a file to upload.';
        echo json_encode($response);
        exit;
    }

    $file = $_FILES['profile_picture'];

    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        switch ($file['error']) {
            case UPLOAD_ERR_INI_SIZE:
            case UPLOAD_ERR_FORM_SIZE:
                $response['errors']['file'] = 'File size too large. Maximum 5MB allowed.';
                break;
            case UPLOAD_ERR_PARTIAL:
                $response['errors']['file'] = 'File upload was incomplete.';
                break;
            case UPLOAD_ERR_NO_FILE:
                $response['errors']['file'] = 'No file was selected.';
                break;
            default:
                $response['errors']['file'] = 'File upload failed. Please try again.';
        }
        echo json_encode($response);
        exit;
    }

    // Validate file type
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!in_array($mime_type, $allowed_types)) {
        $response['errors']['file'] = 'Invalid file type. Only JPEG, PNG, and GIF images are allowed.';
        echo json_encode($response);
        exit;
    }

    // Validate file size (5MB)
    if ($file['size'] > 5 * 1024 * 1024) {
        $response['errors']['file'] = 'File size must be less than 5MB.';
        echo json_encode($response);
        exit;
    }

    // Get current student data to delete old image
    $stmt = $conn->prepare("SELECT profile_picture FROM students WHERE id = ?");
    $stmt->bind_param("i", $student_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $student = $result->fetch_assoc();

    if (!$student) {
        throw new Exception('Student record not found.');
    }

    // Prepare upload directory
    $target_dir = "../uploads/";
    if (!file_exists($target_dir)) {
        if (!mkdir($target_dir, 0777, true)) {
            throw new Exception('Failed to create upload directory.');
        }
    }

    // Generate unique filename
    $file_extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $new_filename = "profile_" . $student_id . "_" . time() . "." . $file_extension;
    $target_file = $target_dir . $new_filename;

    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $target_file)) {
        throw new Exception('Failed to move uploaded file.');
    }

    // Delete old profile picture if it's not the default
    $old_picture = $student['profile_picture'];
    if (!empty($old_picture) && $old_picture != 'assets/images/default-avatar.png') {
        if (file_exists($old_picture)) {
            @unlink($old_picture);
        }
    }

    // Update database
    $update_stmt = $conn->prepare("UPDATE students SET profile_picture = ? WHERE id = ?");
    if (!$update_stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }

    $update_stmt->bind_param("si", $target_file, $student_id);

    if ($update_stmt->execute()) {
        $response['success'] = true;
        $response['message'] = 'Profile picture updated successfully!';
        $response['new_image_url'] = $target_file;
    } else {
        // Delete the uploaded file if database update failed
        @unlink($target_file);
        throw new Exception('Failed to update profile picture in database.');
    }

} catch (Exception $e) {
    error_log("Profile picture update error: " . $e->getMessage());
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
?>