<?php
include 'include/header.php';
require_once '../config/conn.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$student_id = $_SESSION['user_id'];

// Get the student's bank account (if any)
$account_stmt = $conn->prepare("
    SELECT * FROM student_bank_accounts
    WHERE student_id = ?
    LIMIT 1
");
$account_stmt->bind_param("i", $student_id);
$account_stmt->execute();
$account_result = $account_stmt->get_result();
$bank_account = $account_result->fetch_assoc();

// Check if Paystack is configured
$paystack_configured = !empty(getenv('PAYSTACK_SECRET_KEY')) || file_exists('../config/paystack.php');
?>

<div class="container mx-auto px-3 sm:px-4 py-4 sm:py-8">
    <div class="w-full">
        <!-- Messages Container -->
        <div id="messageContainer" class="mb-4 sm:mb-6"></div>
        
        <?php if (!$paystack_configured): ?>
            <div class="p-3 sm:p-4 bg-red-50 border-l-4 border-red-400 mb-4 sm:mb-6 rounded-lg text-sm sm:text-base">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-red-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-red-700">
                            <strong>Paystack not configured:</strong> Bank account verification is not available. Please contact administrator.
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>
        
        <div class="bg-white overflow-hidden  ">
            <div class="px-4 sm:px-6 py-3 sm:py-4 border-b border-gray-200">
                <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center space-y-2 sm:space-y-0">
                    <div>
                        <h2 class="text-lg sm:text-xl font-bold text-gray-800">Bank Account Details</h2>
                        <p class="text-xs sm:text-sm text-gray-600 mt-1">Add your bank account to receive refunds or payments</p>
                    </div>
                    <?php if (!$bank_account): ?>
                    <button onclick="openAddAccountModal()"
                            class="bg-primary-500 hover:bg-primary-600 text-white px-4 py-2 rounded-lg text-sm font-medium transition duration-200 w-full sm:w-auto">
                        <i class="fas fa-plus mr-2"></i> Add Bank Account
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="p-4 sm:p-6">
                <?php if (!$bank_account): ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="mx-auto w-16 h-16 sm:w-24 sm:h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4 sm:mb-6">
                            <i class="fas fa-university text-gray-400 text-2xl sm:text-3xl"></i>
                        </div>
                        <h3 class="text-base sm:text-lg font-medium text-gray-900 mb-2">No Bank Account Linked</h3>
                        <p class="text-sm text-gray-500 mb-6 sm:mb-8">Add your bank account to receive payments or refunds</p>
                        <button onclick="openAddAccountModal()"
                                class="bg-primary-500 hover:bg-primary-600 text-white px-4 sm:px-6 py-2 sm:py-3 rounded-lg font-medium text-sm sm:text-base w-full sm:w-auto">
                            <i class="fas fa-plus mr-2"></i> Add Bank Account
                        </button>
                    </div>
                <?php else: ?>
                    <div class="space-y-4 sm:space-y-6">
                        <!-- Account Status Card -->
                        <div class="bg-blue-50 border-l-4 border-blue-500 p-3 sm:p-4 rounded-r-lg text-sm">
                            <div class="flex">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-info-circle text-blue-500"></i>
                                </div>
                                <div class="ml-3">
                                    <p class="text-xs sm:text-sm text-blue-700">
                                        Your bank account is used for receiving payments or refunds. Ensure details are correct.
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Account Details Card -->
                        <div class="bg-gray-50 rounded-lg p-4 sm:p-6">
                            <div class="flex flex-col sm:flex-row sm:items-start">
                                <div class="flex-shrink-0 mb-4 sm:mb-0">
                                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                                        <i class="fas fa-university text-blue-600 text-xl"></i>
                                    </div>
                                </div>
                                <div class="sm:ml-4 flex-1">
                                    <div class="flex flex-col sm:flex-row sm:items-start sm:justify-between">
                                        <div class="flex-1">
                                            <h3 class="text-base sm:text-lg font-semibold text-gray-900 break-words">
                                                <?= htmlspecialchars($bank_account['bank_name']) ?>
                                            </h3>
                                            <div class="mt-2 space-y-1 text-sm sm:text-base">
                                                <p class="text-gray-600">
                                                    <span class="font-medium">Account Name:</span> 
                                                    <span class="break-all"><?= htmlspecialchars($bank_account['account_name']) ?></span>
                                                </p>
                                                <p class="text-gray-600">
                                                    <span class="font-medium">Account Number:</span> 
                                                    <span class="font-mono"><?= htmlspecialchars($bank_account['account_number']) ?></span>
                                                </p>
                                                <p class="text-gray-600">
                                                    <span class="font-medium">Added:</span> 
                                                    <?= date('M j, Y', strtotime($bank_account['created_at'])) ?>
                                                </p>
                                            </div>
                                        </div>
                                        
                                        <div class="mt-3 sm:mt-0 sm:ml-4">
                                            <?php if ($bank_account['is_verified'] == 1): ?>
                                                <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                                    <i class="fas fa-check-circle mr-1"></i>
                                                    Verified
                                                </span>
                                            <?php elseif ($bank_account['is_verified'] == 0): ?>
                                                <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
                                                    <i class="fas fa-clock mr-1"></i>
                                                    Pending
                                                </span>
                                            <?php else: ?>
                                                <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                                    <i class="fas fa-times-circle mr-1"></i>
                                                    Failed
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="mt-4 sm:mt-6 flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-3">
                                        <button onclick="openEditAccountModal()"
                                                class="inline-flex items-center justify-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50">
                                            <i class="fas fa-edit mr-2"></i>
                                            Edit Details
                                        </button>
                                        <button onclick="deleteBankAccount(<?= $bank_account['id'] ?>)"
                                                class="inline-flex items-center justify-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-red-600 hover:bg-red-700">
                                            <i class="fas fa-trash mr-2"></i>
                                            Delete Account
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Information Cards -->
                        <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-3 sm:p-4 rounded-r-lg text-xs sm:text-sm">
                                <div class="flex">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-clock text-yellow-400"></i>
                                    </div>
                                    <div class="ml-3">
                                        <p class="text-yellow-700">
                                            <strong>Processing Time:</strong> Payments are processed within 2-3 business days.
                                        </p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="bg-blue-50 border-l-4 border-blue-400 p-3 sm:p-4 rounded-r-lg text-xs sm:text-sm">
                                <div class="flex">
                                    <div class="flex-shrink-0">
                                        <i class="fas fa-shield-alt text-blue-400"></i>
                                    </div>
                                    <div class="ml-3">
                                        <p class="text-blue-700">
                                            <strong>Secure:</strong> Your bank details are encrypted and protected.
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Bank Account Modal -->
<div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-2 sm:p-4 hidden z-50" id="accountModal">
    <div class="bg-white p-4 sm:p-6 rounded-lg w-full max-w-sm sm:max-w-md mx-auto relative max-h-[90vh] overflow-y-auto">
        <button type="button" onclick="closeAccountModal()"
                class="absolute top-2 right-2 sm:top-4 sm:right-4 text-gray-400 hover:text-gray-500">
            <i class="fas fa-times text-lg sm:text-xl"></i>
        </button>
        
        <h3 class="text-lg sm:text-xl font-bold text-gray-900 mb-3 sm:mb-4" id="modalTitle">Add Bank Account</h3>
        
        <!-- Warning Message -->
        <div class="mb-4 sm:mb-6 p-3 sm:p-4 bg-red-50 border border-red-200 rounded-lg text-sm">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                </div>
                <div class="ml-3">
                    <p class="text-red-700">
                        <strong>Important:</strong> Please ensure the account name matches your bank account exactly.
                    </p>
                </div>
            </div>
        </div>
        
        <!-- Modal Messages -->
        <div id="modalMessageContainer" class="mb-3 sm:mb-4"></div>
        
        <form id="bankAccountForm" method="POST">
            <input type="hidden" name="student_id" value="<?= $student_id ?>">
            <input type="hidden" name="account_id" id="accountId" value="<?= $bank_account['id'] ?? '' ?>">
            
            <div class="space-y-3 sm:space-y-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1 sm:mb-2">Bank Name</label>
                    <select name="bank_code" id="bankCode" 
                            class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 text-sm sm:text-base"
                            required>
                        <option value="">Select Bank</option>
                        <!-- Banks will be loaded via JavaScript -->
                    </select>
                    <small class="text-red-500 text-xs sm:text-sm mt-1 hidden" id="bank_code_error"></small>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1 sm:mb-2">Account Number</label>
                    <input type="text" name="account_number" id="accountNumber"
                           class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500 text-sm sm:text-base"
                           maxlength="10"
                           placeholder="Enter 10-digit account number"
                           oninput="validateAccountNumber(this.value)"
                           required>
                    <small class="text-red-500 text-xs sm:text-sm mt-1 hidden" id="account_number_error"></small>
                    <p class="text-xs text-gray-500 mt-1">10-digit NUBAN account number</p>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-1 sm:mb-2">Account Name</label>
                    <input type="text" name="account_name" id="accountNameInput"
                           class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg bg-gray-50 text-sm sm:text-base"
                           readonly>
                    <small class="text-red-500 text-xs sm:text-sm mt-1 hidden" id="account_name_error"></small>
                    <p class="text-xs text-gray-500 mt-1">Automatically verified via Paystack</p>
                </div>
                
                <div class="pt-3 sm:pt-4 border-t border-gray-200">
                    <div class="flex items-start">
                        <div class="flex items-center h-5">
                            <input type="checkbox" id="confirmName" name="confirm_name"
                                   class="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded"
                                   required>
                        </div>
                        <div class="ml-3">
                            <label for="confirmName" class="text-xs sm:text-sm text-gray-700">
                                I confirm that the displayed account name exactly matches my bank account
                            </label>
                            <small class="text-red-500 text-xs sm:text-sm mt-1 hidden" id="confirm_name_error"></small>
                        </div>
                    </div>
                </div>
                
                <div class="pt-4 sm:pt-6 flex flex-col sm:flex-row sm:justify-end space-y-2 sm:space-y-0 sm:space-x-3">
                    <button type="button" onclick="closeAccountModal()"
                            class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 text-sm font-medium order-2 sm:order-1">
                        Cancel
                    </button>
                    <button type="submit" id="submitBtn"
                            class="px-4 py-2 bg-primary-500 text-white rounded-lg hover:bg-primary-600 text-sm font-medium flex items-center justify-center space-x-2 order-1 sm:order-2">
                        <span id="btnText">Save Account</span>
                        <div id="btnSpinner" class="hidden w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
// Modal functions
function openAddAccountModal() {
    document.getElementById('modalTitle').textContent = 'Add Bank Account';
    document.getElementById('accountModal').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
    document.getElementById('modalMessageContainer').innerHTML = '';
    resetForm();
    fetchBanksFromPaystack();
}

function openEditAccountModal() {
    document.getElementById('modalTitle').textContent = 'Edit Bank Account';
    document.getElementById('accountModal').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
    document.getElementById('modalMessageContainer').innerHTML = '';
    resetForm();
    
    // Pre-fill existing data
    <?php if (isset($bank_account)): ?>
    $('#accountId').val('<?= $bank_account['id'] ?>');
    $('#accountNumber').val('<?= $bank_account['account_number'] ?>');
    $('#accountNameInput').val('<?= addslashes($bank_account['account_name']) ?>');
    $('#accountNameInput').addClass('bg-green-100');
    $('#confirmName').prop('disabled', false);
    <?php endif; ?>
    
    fetchBanksFromPaystack();
}

function closeAccountModal() {
    document.getElementById('accountModal').classList.add('hidden');
    document.body.style.overflow = 'auto';
    resetForm();
}

function resetForm() {
    document.getElementById('bankAccountForm').reset();
    document.getElementById('accountNameInput').value = '';
    document.getElementById('accountNameInput').classList.remove('bg-gray-50', 'bg-green-100', 'bg-red-100');
    document.getElementById('accountNameInput').classList.add('bg-gray-50');
    document.getElementById('confirmName').checked = false;
    document.getElementById('confirmName').disabled = true;
    hideAllErrors();
}

function hideAllErrors() {
    $('.text-red-500').addClass('hidden').text('');
}

// Fetch banks from Paystack
function fetchBanksFromPaystack() {
    $('#bankCode').html('<option value="">Loading banks...</option>');
    $('#bankCode').prop('disabled', true);
    
    $.ajax({
        url: 'ajax/student_fetch_banks.php',
        type: 'GET',
        success: function(response) {
            $('#bankCode').prop('disabled', false);
            
            if (response.success && response.banks.length > 0) {
                let options = '<option value="">Select Bank</option>';
                response.banks.forEach(function(bank) {
                    const isSelected = <?php if (isset($bank_account['bank_code'])): ?>
                        bank.code === '<?= $bank_account['bank_code'] ?>' ? 'selected' : ''
                    <?php else: ?>''<?php endif; ?>;
                    options += `<option value="${bank.code}" ${isSelected}>${bank.name}</option>`;
                });
                $('#bankCode').html(options);
            } else {
                $('#bankCode').html('<option value="">Failed to load banks</option>');
                showModalMessage(response.message || 'Failed to load banks', 'error');
            }
        },
        error: function() {
            $('#bankCode').prop('disabled', false);
            $('#bankCode').html('<option value="">Failed to load banks</option>');
            showModalMessage('Network error. Please try again.', 'error');
        }
    });
}

// Account number validation
let accountValidationTimeout;

function validateAccountNumber(accountNumber) {
    clearTimeout(accountValidationTimeout);
    
    const accountNumberError = document.getElementById('account_number_error');
    const accountNameInput = document.getElementById('accountNameInput');
    const confirmCheckbox = document.getElementById('confirmName');
    
    // Basic validation
    if (!/^\d{10}$/.test(accountNumber)) {
        accountNumberError.textContent = 'Please enter a valid 10-digit account number';
        accountNumberError.classList.remove('hidden');
        accountNameInput.value = '';
        accountNameInput.classList.remove('bg-green-100', 'bg-red-100');
        accountNameInput.classList.add('bg-gray-50');
        confirmCheckbox.checked = false;
        confirmCheckbox.disabled = true;
        return;
    } else {
        accountNumberError.classList.add('hidden');
    }
    
    // Lookup account name
    accountValidationTimeout = setTimeout(() => {
        lookupAccountName(accountNumber);
    }, 800);
}

function lookupAccountName(accountNumber) {
    const bankCode = document.getElementById('bankCode').value;
    const accountNameInput = document.getElementById('accountNameInput');
    const confirmCheckbox = document.getElementById('confirmName');
    
    if (!bankCode) {
        accountNameInput.value = 'Please select a bank first';
        accountNameInput.classList.remove('bg-gray-50', 'bg-green-100', 'bg-red-100');
        accountNameInput.classList.add('bg-red-100');
        confirmCheckbox.checked = false;
        confirmCheckbox.disabled = true;
        return;
    }
    
    accountNameInput.value = 'Verifying account...';
    accountNameInput.classList.remove('bg-gray-50', 'bg-green-100', 'bg-red-100');
    confirmCheckbox.checked = false;
    confirmCheckbox.disabled = true;
    
    $.ajax({
        url: 'ajax/student_lookup_account_name.php',
        type: 'POST',
        data: {
            account_number: accountNumber,
            bank_code: bankCode
        },
        success: function(response) {
            if (response.success) {
                accountNameInput.value = response.account_name;
                accountNameInput.classList.remove('bg-gray-50', 'bg-red-100');
                accountNameInput.classList.add('bg-green-100');
                confirmCheckbox.disabled = false;
            } else {
                accountNameInput.value = 'Account verification failed';
                accountNameInput.classList.remove('bg-gray-50', 'bg-green-100');
                accountNameInput.classList.add('bg-red-100');
                confirmCheckbox.checked = false;
                confirmCheckbox.disabled = true;
                
                if (response.message) {
                    showModalMessage(response.message, 'error');
                }
            }
        },
        error: function() {
            accountNameInput.value = 'Network error';
            accountNameInput.classList.remove('bg-gray-50', 'bg-green-100');
            accountNameInput.classList.add('bg-red-100');
            confirmCheckbox.checked = false;
            confirmCheckbox.disabled = true;
            showModalMessage('Failed to verify account. Please try again.', 'error');
        }
    });
}

// Form submission
$(document).ready(function() {
    $('#bankAccountForm').on('submit', function(e) {
        e.preventDefault();
        
        const formData = $(this).serialize();
        const submitBtn = $('#submitBtn');
        const btnText = $('#btnText');
        const btnSpinner = $('#btnSpinner');
        const modalMessageContainer = $('#modalMessageContainer');
        
        // Clear previous errors
        hideAllErrors();
        modalMessageContainer.html('');
        
        // Validate
        const accountName = $('#accountNameInput').val();
        const isConfirmed = $('#confirmName').is(':checked');
        
        if (!accountName || accountName.includes('Verifying') || accountName.includes('failed') || accountName.includes('error')) {
            showModalMessage('Please verify your account number first', 'error');
            return;
        }
        
        if (!isConfirmed) {
            $('#confirm_name_error').text('Please confirm the account name matches').removeClass('hidden');
            return;
        }
        
        // Show loading
        submitBtn.prop('disabled', true);
        btnText.text('Saving...');
        btnSpinner.removeClass('hidden');
        
        // Send request
        $.ajax({
            url: 'ajax/student_save_bank_account.php',
            type: 'POST',
            data: formData,
            success: function(response) {
                submitBtn.prop('disabled', false);
                btnText.text('Save Account');
                btnSpinner.addClass('hidden');
                
                if (response.success) {
                    showMessage(response.message, 'success');
                    closeAccountModal();
                    
                    // Reload page
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showModalMessage(response.message, 'error');
                    
                    if (response.errors) {
                        for (const field in response.errors) {
                            $(`#${field}_error`).text(response.errors[field]).removeClass('hidden');
                        }
                    }
                }
            },
            error: function() {
                submitBtn.prop('disabled', false);
                btnText.text('Save Account');
                btnSpinner.addClass('hidden');
                showModalMessage('An error occurred. Please try again.', 'error');
            }
        });
    });
});

// Delete bank account
function deleteBankAccount(accountId) {
    if (!confirm('Are you sure you want to delete this bank account? This action cannot be undone.')) {
        return;
    }
    
    $.ajax({
        url: 'ajax/student_delete_bank_account.php',
        type: 'POST',
        data: { account_id: accountId },
        success: function(response) {
            if (response.success) {
                showMessage(response.message, 'success');
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showMessage(response.message, 'error');
            }
        },
        error: function() {
            showMessage('An error occurred while deleting the account.', 'error');
        }
    });
}

// Message functions
function showMessage(message, type) {
    const messageContainer = document.getElementById('messageContainer');
    showMessageInContainer(message, type, messageContainer);
}

function showModalMessage(message, type) {
    const messageContainer = document.getElementById('modalMessageContainer');
    showMessageInContainer(message, type, messageContainer);
}

function showMessageInContainer(message, type, container) {
    const alertClass = type === 'success' 
        ? 'bg-green-100 border-green-400 text-green-700' 
        : 'bg-red-100 border-red-400 text-red-700';
    
    const icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-triangle';
    
    const messageHTML = `
        <div class="rounded-lg ${alertClass} p-4">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <i class="fas ${icon}"></i>
                </div>
                <div class="ml-3">
                    <p class="text-sm">${message}</p>
                </div>
                <div class="ml-auto pl-3">
                    <button type="button" onclick="this.parentElement.parentElement.parentElement.remove()" 
                            class="inline-flex ${type === 'success' ? 'text-green-500 hover:text-green-600' : 'text-red-500 hover:text-red-600'}">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            </div>
        </div>
    `;
    
    container.innerHTML = messageHTML;
}

// Close modal on outside click
window.addEventListener('click', function(event) {
    const modal = document.getElementById('accountModal');
    if (event.target === modal) {
        closeAccountModal();
    }
});

// Auto-validate when bank is changed
$('#bankCode').on('change', function() {
    const accountNumber = $('#accountNumber').val();
    if (/^\d{10}$/.test(accountNumber)) {
        lookupAccountName(accountNumber);
    }
});

// Prevent body scroll when modal is open
document.getElementById('accountModal').addEventListener('wheel', function(e) {
    if (this.scrollHeight > this.clientHeight) {
        e.stopPropagation();
    }
});
</script>

<?php include 'include/footer.php'; ?>