<?php
ob_start();
include 'include/header.php';
require_once '../config/conn.php';

// Authentication check
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$student_id = $_SESSION['user_id'];

// Get wallet info
$wallet_stmt = $conn->prepare("SELECT id, wallet_id, balance FROM student_wallets WHERE student_id = ?");
$wallet_stmt->bind_param("i", $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();
$wallet = $wallet_result->fetch_assoc();

// Get student info for email
$student_stmt = $conn->prepare("SELECT email, full_name, phone FROM students WHERE id = ?");
$student_stmt->bind_param("i", $student_id);
$student_stmt->execute();
$student_result = $student_stmt->get_result();
$student = $student_result->fetch_assoc();
?>

<div class="max-w-sm mx-auto p-4 ">
    <!-- Messages Container -->
    <div id="messageContainer" class="mb-4"></div>

    <!-- Header -->
    <div class="text-center mb-2">
        <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-3">
            <i class="fas fa-plus text-green-500 text-lg"></i>
        </div>
        <h2 class="text-lg font-bold text-gray-800">Deposit Funds</h2>
        <p class="text-gray-500 text-sm mt-1">Add money to your wallet</p>
    </div>

    <!-- Deposit Form -->
    <div class=" p-4">
        <form id="depositForm">
            <input type="hidden" name="student_id" value="<?= $student_id ?>">
            <input type="hidden" name="wallet_id" value="<?= $wallet['id'] ?>">
            <input type="hidden" name="wallet_id_display" value="<?= $wallet['wallet_id'] ?>">
            <input type="hidden" name="email" value="<?= $student['email'] ?>">
            <input type="hidden" name="fullname" value="<?= $student['full_name'] ?>">
            <input type="hidden" name="phone" value="<?= $student['phone'] ?>">
            
            <!-- Amount Input -->
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-1">Amount (₦)</label>
                <div class="relative">
                    <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500">₦</span>
                    <input type="text" name="amount" id="amountInput"
                           class="w-full pl-8 pr-3 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]" 
                           placeholder="0.00" 
                           required
                           oninput="formatAmount(this)">
                </div>
                <p class="text-xs text-gray-500 mt-1">Minimum deposit: ₦100</p>
                <small class="text-red-500 text-xs mt-1 block hidden" id="amount_error"></small>
            </div>
            
            <!-- Payment Method -->
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2">Payment Method</label>
                <div class="space-y-2">
                    <button type="button" onclick="selectPaymentMethod('paystack')" 
                            class="payment-method w-full p-3 border border-gray-300 rounded-lg flex items-center hover:border-[#fbbb01] hover:bg-yellow-50 transition">
                        <div class="w-8 h-8 mr-3 flex items-center justify-center">
                            <img src="../img/pay.png" alt="Paystack" class="h-6">
                        </div>
                        <div class="text-left">
                            <span class="text-sm font-medium">Paystack</span>
                            <p class="text-xs text-gray-500">Card, Bank Transfer, USSD</p>
                        </div>
                    </button>

                </div>
                <input type="hidden" name="payment_method" id="paymentMethod" required>
                <small class="text-red-500 text-xs mt-1 block hidden" id="payment_method_error"></small>
            </div>
            
            <!-- Bank Transfer Details (Hidden by default) -->
            <div id="bankTransferDetails" class="hidden mb-4">
                <div class="bg-blue-50 border border-blue-200 p-3 rounded-lg">
                    <p class="text-xs font-medium text-blue-700 mb-2">Transfer to:</p>
                    <p class="text-sm text-blue-700 mb-1"><strong>Bank:</strong> Your Bank Name</p>
                    <p class="text-sm text-blue-700 mb-1"><strong>Account:</strong> 0123456789</p>
                    <p class="text-sm text-blue-700"><strong>Name:</strong> Campus Services</p>
                    
                    <div class="mt-3">
                        <label class="block text-gray-700 text-sm font-medium mb-1">Upload Payment Proof</label>
                        <input type="file" name="payment_proof" id="paymentProof" 
                               accept=".jpg,.jpeg,.png,.pdf"
                               class="w-full text-sm border border-gray-300 rounded-lg p-2">
                        <small class="text-red-500 text-xs mt-1 block hidden" id="payment_proof_error"></small>
                        <p class="text-xs text-gray-500 mt-1">Upload screenshot of transaction</p>
                    </div>
                </div>
            </div>
            
            <!-- Submit Button -->
            <button type="submit" id="depositSubmit" 
                    class="w-full bg-[#fbbb01] text-white text-sm font-medium py-2.5 rounded-lg hover:bg-[#e0a800] transition disabled:opacity-70 disabled:cursor-not-allowed">
                <span id="btnText">Proceed to Payment</span>
                <span id="loadingSpinner" class="hidden ml-2">
                    <i class="fas fa-spinner fa-spin"></i>
                </span>
            </button>
            
            <!-- Back Link -->
            <div class="text-center mt-4">
                <a href="wallet.php" class="text-gray-500 hover:text-[#fbbb01] text-xs">
                    ← Back to Wallet
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Processing Modal -->
<div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden z-50 p-4" id="processingModal">
    <div class="bg-white p-6 rounded-lg w-full max-w-sm text-center">
        <div class="w-12 h-12 border-2 border-[#fbbb01] border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
        <h3 class="text-lg font-bold mb-2">Processing Payment</h3>
        <p class="text-gray-600 text-sm">Please wait...</p>
    </div>
</div>

<script src="https://js.paystack.co/v1/inline.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const depositForm = document.getElementById('depositForm');
    let paymentData = null;
    
    // Amount formatting
    window.formatAmount = function(input) {
        let value = input.value.replace(/[^\d.]/g, '');
        
        // Remove extra decimal points
        let decimalCount = (value.match(/\./g) || []).length;
        if (decimalCount > 1) {
            value = value.substring(0, value.lastIndexOf('.'));
        }
        
        // Format with commas
        if (value) {
            let parts = value.split('.');
            parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
            input.value = parts.join('.');
        }
        
        validateAmount(value.replace(/,/g, ''));
    };
    
    function validateAmount(rawAmount) {
        const amountError = document.getElementById('amount_error');
        const amount = parseFloat(rawAmount);
        
        if (isNaN(amount) || amount < 100) {
            amountError.textContent = 'Minimum deposit amount is ₦100';
            amountError.classList.remove('hidden');
            return false;
        } else {
            amountError.classList.add('hidden');
            return true;
        }
    }
    
    function getRawAmount() {
        const formattedAmount = document.getElementById('amountInput').value;
        return parseFloat(formattedAmount.replace(/,/g, '')) || 0;
    }
    
    window.selectPaymentMethod = function(method) {
        document.getElementById('paymentMethod').value = method;
        
        // Highlight selected method
        document.querySelectorAll('.payment-method').forEach(btn => {
            btn.classList.remove('border-[#fbbb01]', 'bg-yellow-50', 'ring-1', 'ring-[#fbbb01]');
            btn.classList.add('border-gray-300');
        });
        
        event.currentTarget.classList.add('border-[#fbbb01]', 'bg-yellow-50', 'ring-1', 'ring-[#fbbb01]');
        event.currentTarget.classList.remove('border-gray-300');
        
        // Show/hide Bank Transfer details
        const bankDetails = document.getElementById('bankTransferDetails');
        if (method === 'bank_transfer') {
            bankDetails.classList.remove('hidden');
        } else {
            bankDetails.classList.add('hidden');
        }
        
        // Clear payment method error
        document.getElementById('payment_method_error').classList.add('hidden');
    };
    
    // Form submission
    depositForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const amount = getRawAmount();
        const paymentMethod = document.getElementById('paymentMethod').value;
        
        // Clear previous errors
        document.getElementById('messageContainer').innerHTML = '';
        document.querySelectorAll('[id$="_error"]').forEach(el => {
            el.textContent = '';
            el.classList.add('hidden');
        });
        
        // Validation
        if (!validateAmount(amount)) {
            return;
        }
        
        if (!paymentMethod) {
            document.getElementById('payment_method_error').textContent = 'Please select a payment method';
            document.getElementById('payment_method_error').classList.remove('hidden');
            return;
        }
        
        // Show loading state
        const submitBtn = document.getElementById('depositSubmit');
        const btnText = document.getElementById('btnText');
        const loadingSpinner = document.getElementById('loadingSpinner');
        
        submitBtn.disabled = true;
        btnText.textContent = 'Processing...';
        loadingSpinner.classList.remove('hidden');
        
        // Prepare payment data
        paymentData = {
            amount: amount * 100, // Convert to kobo for Paystack
            email: document.querySelector('input[name="email"]').value,
            fullname: document.querySelector('input[name="fullname"]').value,
            phone: document.querySelector('input[name="phone"]').value,
            wallet_id: document.querySelector('input[name="wallet_id_display"]').value,
            student_id: document.querySelector('input[name="student_id"]').value,
            wallet_db_id: document.querySelector('input[name="wallet_id"]').value,
            purpose: 'wallet_funding'
        };
        
        // Initiate payment based on selected method
        if (paymentMethod === 'paystack') {
            initiatePaystackPayment(paymentData);
        } else if (paymentMethod === 'bank_transfer') {
            handleBankTransfer(paymentData);
        }
    });
    
    // Handle Bank Transfer
    function handleBankTransfer(paymentData) {
        const paymentProof = document.getElementById('paymentProof').files[0];
        
        if (!paymentProof) {
            document.getElementById('payment_proof_error').textContent = 'Please upload payment proof';
            document.getElementById('payment_proof_error').classList.remove('hidden');
            resetSubmitButton();
            return;
        }
        
        // Show processing modal
        document.getElementById('processingModal').classList.remove('hidden');
        
        // Create FormData for file upload
        const formData = new FormData();
        formData.append('student_id', paymentData.student_id);
        formData.append('amount', paymentData.amount / 100); // Convert back to Naira
        formData.append('wallet_id', paymentData.wallet_db_id);
        formData.append('payment_proof', paymentProof);
        formData.append('reference', 'BT_' + Date.now());
        
        fetch('ajax_process_bank_transfer.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            document.getElementById('processingModal').classList.add('hidden');
            resetSubmitButton();
            
            if (data.success) {
                showMessage(data.message, 'success');
                setTimeout(() => {
                    window.location.href = 'wallet.php';
                }, 1500);
            } else {
                showMessage(data.message, 'error');
                if (data.errors) {
                    for (const field in data.errors) {
                        const errorEl = document.getElementById(field + '_error');
                        if (errorEl) {
                            errorEl.textContent = data.errors[field];
                            errorEl.classList.remove('hidden');
                        }
                    }
                }
            }
        })
        .catch(error => {
            document.getElementById('processingModal').classList.add('hidden');
            resetSubmitButton();
            console.error('Error:', error);
            showMessage('Network error. Please try again.', 'error');
        });
    }
    
    // Paystack payment integration
    function initiatePaystackPayment(paymentData) {
        // Generate unique reference
        const reference = 'PSK_' + paymentData.student_id + '_' + Date.now();
        
        const handler = PaystackPop.setup({
            key: 'pk_test_b47847d06c11275c802c304d0e9568b02a8c7391', // Replace with your actual Paystack public key
            email: paymentData.email,
            amount: paymentData.amount,
            currency: 'NGN',
            ref: reference,
            metadata: {
                student_id: paymentData.student_id,
                wallet_id: paymentData.wallet_db_id,
                purpose: paymentData.purpose,
                custom_fields: [
                    {
                        display_name: "Wallet ID",
                        variable_name: "wallet_id",
                        value: paymentData.wallet_id
                    }
                ]
            },
            callback: function(response) {
                // Show processing modal
                document.getElementById('processingModal').classList.remove('hidden');
                
                // Verify payment
                verifyPayment(response.reference, 'paystack');
            },
            onClose: function() {
                resetSubmitButton();
                showMessage('Payment cancelled', 'warning');
            }
        });
        
        handler.openIframe();
    }
    
    // Verify payment
    function verifyPayment(reference, gateway) {
        fetch('verify_payment.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                reference: reference,
                gateway: gateway
            })
        })
        .then(response => response.json())
        .then(data => {
            document.getElementById('processingModal').classList.add('hidden');
            resetSubmitButton();
            
            if (data.success) {
                showMessage(`Payment successful! ₦${data.amount.toLocaleString()} added to wallet`, 'success');
                setTimeout(() => {
                    window.location.href = 'wallet.php';
                }, 1500);
            } else {
                showMessage('Payment failed: ' + data.message, 'error');
            }
        })
        .catch(error => {
            document.getElementById('processingModal').classList.add('hidden');
            resetSubmitButton();
            console.error('Error:', error);
            showMessage('Payment verification failed. Contact support if funds were deducted.', 'error');
        });
    }
    
    function resetSubmitButton() {
        const submitBtn = document.getElementById('depositSubmit');
        const btnText = document.getElementById('btnText');
        const loadingSpinner = document.getElementById('loadingSpinner');
        
        submitBtn.disabled = false;
        btnText.textContent = 'Proceed to Payment';
        loadingSpinner.classList.add('hidden');
    }
    
    function showMessage(message, type) {
        const container = document.getElementById('messageContainer');
        const colors = {
            success: 'bg-green-50 text-green-700 border border-green-200',
            error: 'bg-red-50 text-red-700 border border-red-200',
            warning: 'bg-yellow-50 text-yellow-700 border border-yellow-200'
        };
        
        const div = document.createElement('div');
        div.className = `p-3 rounded-lg text-sm mb-3 ${colors[type] || colors.error}`;
        div.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 
                                type === 'warning' ? 'fa-exclamation-triangle' : 
                                'fa-exclamation-circle'} mr-2"></i>
                <span>${message}</span>
            </div>
        `;
        
        container.prepend(div);
        
        // Auto-remove success messages
        if (type === 'success') {
            setTimeout(() => div.remove(), 4000);
        }
    }
});
</script>

<?php include 'include/footer.php'; ?>