<?php
require_once '../config/conn.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    die('<div class="text-center p-4 text-red-500 text-sm">Unauthorized</div>');
}

$student_id = $_SESSION['user_id'];

// Get filter parameters
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$start_date = isset($_GET['start_date']) && $_GET['start_date'] !== '' ? $_GET['start_date'] : null;
$end_date = isset($_GET['end_date']) && $_GET['end_date'] !== '' ? $_GET['end_date'] : null;
$status_filter = isset($_GET['status']) && $_GET['status'] !== '' ? $_GET['status'] : '';
$type_filter = isset($_GET['type']) && $_GET['type'] !== '' ? $_GET['type'] : '';
$sort_order = (isset($_GET['sort']) && $_GET['sort'] === 'asc') ? 'ASC' : 'DESC';

$limit = 15;
$offset = ($page - 1) * $limit;

// Get wallet
$wallet_stmt = $conn->prepare("SELECT id FROM student_wallets WHERE student_id = ?");
$wallet_stmt->bind_param("i", $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();
$wallet = $wallet_result->fetch_assoc();

if (!$wallet) {
    die('<div class="text-center p-4 text-red-500 text-sm">Wallet not found</div>');
}

// Build WHERE clause
$where_conditions = ["t.wallet_id = ?"];
$params = "i";
$param_values = [$wallet['id']];

if ($start_date && $end_date) {
    $where_conditions[] = "DATE(t.created_at) BETWEEN ? AND ?";
    $params .= "ss";
    $param_values[] = $start_date;
    $param_values[] = $end_date;
} elseif ($start_date) {
    $where_conditions[] = "DATE(t.created_at) >= ?";
    $params .= "s";
    $param_values[] = $start_date;
} elseif ($end_date) {
    $where_conditions[] = "DATE(t.created_at) <= ?";
    $params .= "s";
    $param_values[] = $end_date;
}

if ($status_filter) {
    $where_conditions[] = "t.status = ?";
    $params .= "s";
    $param_values[] = $status_filter;
}

if ($type_filter === 'credit') {
    $where_conditions[] = "t.transaction_type IN ('deposit', 'refund', 'credit', 'transfer_in')";
} elseif ($type_filter === 'debit') {
    $where_conditions[] = "t.transaction_type IN ('withdrawal', 'payment', 'transfer_out', 'fee')";
}

$where_clause = implode(" AND ", $where_conditions);

// Get total count
$count_query = "SELECT COUNT(*) as total FROM wallet_transactions t WHERE $where_clause";
$count_stmt = $conn->prepare($count_query);
if (!empty($param_values)) {
    $count_stmt->bind_param($params, ...$param_values);
}
$count_stmt->execute();
$count_result = $count_stmt->get_result();
$total_row = $count_result->fetch_assoc();
$total_rows = $total_row['total'] ?? 0;
$total_pages = ceil($total_rows / $limit);

// Get transactions
$transactions_query = "
    SELECT t.*, b.bank_name, b.account_number 
    FROM wallet_transactions t
    LEFT JOIN student_bank_accounts b ON t.bank_account_id = b.id
    WHERE $where_clause
    ORDER BY t.created_at $sort_order
    LIMIT ? OFFSET ?
";

$params .= "ii";
$param_values[] = $limit;
$param_values[] = $offset;

$transactions_stmt = $conn->prepare($transactions_query);
if ($transactions_stmt) {
    $transactions_stmt->bind_param($params, ...$param_values);
    $transactions_stmt->execute();
    $transactions_result = $transactions_stmt->get_result();
    $transactions = $transactions_result->fetch_all(MYSQLI_ASSOC);
} else {
    die('<div class="text-center p-4 text-red-500 text-sm">Query error</div>');
}

if (empty($transactions)): ?>
    <div class="text-center p-6">
        <div class="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-3">
            <i class="fas fa-exchange-alt text-gray-400 text-sm"></i>
        </div>
        <p class="text-gray-500 text-sm">No transactions found</p>
        <?php if ($start_date || $end_date || $status_filter || $type_filter): ?>
            <p class="text-gray-400 text-xs mt-1 mb-3">Try different filters</p>
            <button onclick="clearFilters()" 
                    class="px-4 py-2 bg-primary-500 text-white text-xs rounded-lg hover:bg-primary-600">
                Clear Filters
            </button>
        <?php endif; ?>
    </div>
<?php else: ?>
    <div class="space-y-1">
        <?php foreach ($transactions as $transaction): 
            $isCredit = in_array($transaction['transaction_type'], ['deposit', 'refund', 'credit', 'transfer_in']);
            $sign = $isCredit ? '+' : '-';
            $amountClass = $isCredit ? 'text-green-600' : 'text-red-600';
            
            // Status color
            $statusColor = 'text-gray-500';
            if (strtolower($transaction['status']) === 'completed') $statusColor = 'text-green-500';
            if (strtolower($transaction['status']) === 'pending') $statusColor = 'text-yellow-500';
            if (in_array(strtolower($transaction['status']), ['failed', 'cancelled'])) $statusColor = 'text-red-500';
            
            // Format dates
            $date = date('M j', strtotime($transaction['created_at']));
            $time = date('g:iA', strtotime($transaction['created_at']));
            
            // Truncate description
            $description = htmlspecialchars($transaction['description']);
            $shortDesc = strlen($description) > 25 ? substr($description, 0, 22) . '...' : $description;
        ?>
            <!-- MOBILE VERSION (320px-640px) -->
            <div data-transaction-id="<?= $transaction['id'] ?>" 
                 class="block sm:hidden bg-white border-b border-gray-100 p-2.5 hover:bg-gray-50 active:bg-gray-100 cursor-pointer"
                 onclick="showTransactionDetails(<?= $transaction['id'] ?>)">
                
                <!-- Single line layout for mobile -->
                <div class="flex items-center justify-between">
                    <!-- Left: Description -->
                    <div class="min-w-0 flex-1 pr-2">
                        <div class="flex items-center gap-1.5 mb-0.5">
                            <!-- Status indicator dot -->
                            <div class="w-1.5 h-1.5 rounded-full <?= $isCredit ? 'bg-green-500' : 'bg-red-500' ?> flex-shrink-0"></div>
                            <!-- Description -->
                            <p class="text-gray-800 font-medium text-xs truncate">
                                <?= $shortDesc ?>
                            </p>
                        </div>
                        
                        <!-- Date, time, status -->
                        <div class="flex items-center gap-1.5 text-gray-500 text-[10px]">
                            <span class="flex items-center">
                                <i class="far fa-clock mr-0.5"></i>
                                <?= $date ?>
                            </span>
                            <span>•</span>
                            <span class="<?= $statusColor ?>">
                                <?= substr(ucfirst($transaction['status']), 0, 4) ?>
                            </span>
                            <span>•</span>
                            <span class="text-gray-400">
                                #<?= $transaction['id'] ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Right: Amount -->
                    <div class="flex-shrink-0 text-right">
                        <p class="font-semibold <?= $amountClass ?> text-xs whitespace-nowrap">
                            <?= $sign ?>₦<?= number_format($transaction['amount'], 2) ?>
                        </p>
                        <p class="text-gray-500 text-[9px] mt-0.5">
                            <?= $isCredit ? 'In' : 'Out' ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- DESKTOP VERSION (640px+) -->
            <div data-transaction-id="<?= $transaction['id'] ?>" 
                 class="hidden sm:block bg-white border border-gray-200 rounded-lg p-3 hover:bg-gray-50 cursor-pointer"
                 onclick="showTransactionDetails(<?= $transaction['id'] ?>)">
                
                <div class="flex items-center justify-between">
                    <!-- Left side -->
                    <div class="flex items-center gap-3 flex-1 min-w-0">
                        <!-- Icon -->
                        <div class="w-8 h-8 rounded-full flex items-center justify-center <?= $isCredit ? 'bg-green-100 text-green-600' : 'bg-red-100 text-red-600' ?> flex-shrink-0">
                            <?php if ($isCredit): ?>
                                <i class="fas fa-arrow-down text-xs"></i>
                            <?php else: ?>
                                <i class="fas fa-arrow-up text-xs"></i>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Details -->
                        <div class="min-w-0 flex-1">
                            <div class="flex items-center justify-between mb-1">
                                <p class="text-gray-800 font-medium text-sm truncate">
                                    <?= htmlspecialchars($transaction['description']) ?>
                                </p>
                                <span class="text-xs px-2 py-1 rounded-full <?= $statusColor ?> bg-opacity-10 ml-2 flex-shrink-0">
                                    <?= ucfirst($transaction['status']) ?>
                                </span>
                            </div>
                            
                            <div class="flex items-center gap-3 text-gray-500 text-xs">
                                <span class="flex items-center">
                                    <i class="far fa-clock mr-1"></i>
                                    <?= date('M j, Y • g:i A', strtotime($transaction['created_at'])) ?>
                                </span>
                                
                                <?php if ($transaction['bank_name']): ?>
                                    <span class="hidden md:inline">
                                        • <?= $isCredit ? 'From' : 'To' ?>: 
                                        <?= htmlspecialchars($transaction['bank_name']) ?>
                                    </span>
                                <?php endif; ?>
                                
                                <span class="text-gray-400">
                                    ID: <?= $transaction['id'] ?>
                                </span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Right side: Amount -->
                    <div class="ml-4 flex-shrink-0 text-right">
                        <p class="font-semibold <?= $amountClass ?> text-sm">
                            <?= $sign ?>₦<?= number_format($transaction['amount'], 2) ?>
                        </p>
                        <p class="text-gray-500 text-xs mt-0.5">
                            <?= $isCredit ? 'Credit' : 'Debit' ?>
                        </p>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <?php if ($total_pages > 1): ?>
        <!-- Pagination -->
        <div class="mt-4 pt-3 border-t border-gray-200">
            <!-- Mobile -->
            <div class="sm:hidden">
                <div class="flex items-center justify-center gap-4">
                    <?php if ($page > 1): ?>
                        <a href="#" data-page="<?= $page - 1 ?>" 
                           class="pagination-link text-primary-600 text-sm">
                            ← Previous
                        </a>
                    <?php endif; ?>
                    
                    <span class="text-gray-600 text-sm font-medium">
                        <?= $page ?>/<?= $total_pages ?>
                    </span>
                    
                    <?php if ($page < $total_pages): ?>
                        <a href="#" data-page="<?= $page + 1 ?>" 
                           class="pagination-link text-primary-600 text-sm">
                            Next →
                        </a>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Desktop -->
            <div class="hidden sm:block">
                <div class="flex justify-between items-center">
                    <div class="text-gray-500 text-sm">
                        Page <?= $page ?> of <?= $total_pages ?>
                    </div>
                    
                    <div class="flex items-center gap-1">
                        <?php if ($page > 1): ?>
                            <a href="#" data-page="1" 
                               class="pagination-link px-3 py-1.5 border rounded text-gray-700 hover:bg-gray-50 text-sm">
                                <i class="fas fa-angle-double-left"></i>
                            </a>
                            <a href="#" data-page="<?= $page - 1 ?>" 
                               class="pagination-link px-3 py-1.5 border rounded text-gray-700 hover:bg-gray-50 text-sm">
                                <i class="fas fa-chevron-left"></i>
                            </a>
                        <?php endif; ?>
                        
                        <span class="px-3 py-1.5 bg-primary-500 text-white text-sm rounded">
                            <?= $page ?>
                        </span>
                        
                        <?php if ($page < $total_pages): ?>
                            <a href="#" data-page="<?= $page + 1 ?>" 
                               class="pagination-link px-3 py-1.5 border rounded text-gray-700 hover:bg-gray-50 text-sm">
                                <i class="fas fa-chevron-right"></i>
                            </a>
                            <a href="#" data-page="<?= $total_pages ?>" 
                               class="pagination-link px-3 py-1.5 border rounded text-gray-700 hover:bg-gray-50 text-sm">
                                <i class="fas fa-angle-double-right"></i>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>
<?php endif; ?>