<?php
require_once '../config/conn.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    die('Unauthorized');
}

$student_id = $_SESSION['user_id'];

// Get student registration number to use as wallet ID
$student_stmt = $conn->prepare("SELECT reg_number FROM students WHERE id = ?");
$student_stmt->bind_param("i", $student_id);
$student_stmt->execute();
$student_result = $student_stmt->get_result();
$student = $student_result->fetch_assoc();
$reg_number = $student['reg_number'];

// Check if user has set transaction PIN
$pin_stmt = $conn->prepare("SELECT * FROM wallet_pins WHERE student_id = ?");
$pin_stmt->bind_param("i", $student_id);
$pin_stmt->execute();
$has_pin = $pin_stmt->get_result()->num_rows > 0;

// Get or create wallet
$wallet_stmt = $conn->prepare("SELECT id, wallet_id, balance FROM student_wallets WHERE student_id = ?");
$wallet_stmt->bind_param("i", $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();

if ($wallet_result->num_rows === 0) {
    // Create wallet if doesn't exist using registration number as wallet ID
    $create_stmt = $conn->prepare("INSERT INTO student_wallets (wallet_id, student_id, balance) VALUES (?, ?, 0.00)");
    $create_stmt->bind_param("si", $reg_number, $student_id);
    
    try {
        $create_stmt->execute();
    } catch (mysqli_sql_exception $e) {
        // Handle potential duplicate wallet_id case
        if ($e->getCode() == 1062) {
            // If wallet_id already exists, try with a modified wallet_id
            $modified_reg_number = $reg_number . '-' . substr(uniqid(), -4);
            $create_stmt = $conn->prepare("INSERT INTO student_wallets (wallet_id, student_id, balance) VALUES (?, ?, 0.00)");
            $create_stmt->bind_param("si", $modified_reg_number, $student_id);
            $create_stmt->execute();
        } else {
            throw $e; // Re-throw other exceptions
        }
    }
    
    // Re-fetch wallet
    $wallet_stmt->execute();
    $wallet_result = $wallet_stmt->get_result();
} else {
    // Wallet exists - ensure wallet_id matches current reg_number
    $wallet = $wallet_result->fetch_assoc();
    if ($wallet['wallet_id'] !== $reg_number) {
        $update_stmt = $conn->prepare("UPDATE student_wallets SET wallet_id = ? WHERE student_id = ?");
        $update_stmt->bind_param("si", $reg_number, $student_id);
        $update_stmt->execute();
    }
}

// Final wallet fetch
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();
$wallet = $wallet_result->fetch_assoc();

// Get recent transactions
$transactions_stmt = $conn->prepare("
    SELECT t.*, b.bank_name, b.account_number 
    FROM wallet_transactions t
    LEFT JOIN student_bank_accounts b ON t.bank_account_id = b.id
    WHERE t.wallet_id = ?
    ORDER BY t.created_at DESC
    LIMIT 5
");
$transactions_stmt->bind_param("i", $wallet['id']);
$transactions_stmt->execute();
$transactions_result = $transactions_stmt->get_result();
$transactions = $transactions_result->fetch_all(MYSQLI_ASSOC);

// Get linked bank accounts
$accounts_stmt = $conn->prepare("
    SELECT * FROM student_bank_accounts
    WHERE student_id = ?
    ORDER BY is_primary DESC, created_at DESC
");
$accounts_stmt->bind_param("i", $student_id);
$accounts_stmt->execute();
$accounts_result = $accounts_stmt->get_result();
$bank_accounts = $accounts_result->fetch_all(MYSQLI_ASSOC);

// Get active vendors (latest vendors) for Quick Actions
$vendors_query = "
    SELECT id, vendor_id, business_name, business_type, status, created_at, profile_picture
    FROM vendors 
    WHERE status = 'active' 
    ORDER BY created_at DESC 
    LIMIT 10
";
$vendors_result = mysqli_query($conn, $vendors_query);
$vendors = [];
if ($vendors_result) {
    $vendors = mysqli_fetch_all($vendors_result, MYSQLI_ASSOC);
}


?>

<!-- Main Container -->
<div class="space-y-4 p-4">
    <!-- Balance Card -->
    <div class="  p-4">
        <div class="flex justify-between items-start mb-4">
            <div>
                <p class="text-gray-500 text-xs mb-1">Wallet ID</p>
                <div class="flex items-center">
                    <p class="text-base font-bold text-[#fbbb01] mb-2" id="wallet-id"><?= $reg_number ?></p>
                    <button id="copy-wallet-id" class="ml-2 text-gray-400 hover:text-[#fbbb01] text-sm">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
                
                <p class="text-gray-500 text-xs mb-1">Available Balance</p>
                <div class="flex items-center">
                    <p class="text-2xl font-bold text-[#fbbb01]" id="balance-amount" style="display: none;">
                        ₦<?= number_format($wallet['balance'], 2) ?>
                    </p>
                    <p class="text-2xl font-bold text-[#fbbb01]" id="balance-mask">
                        ₦••••••
                    </p>
                    <button id="toggle-balance" class="ml-2 text-gray-400 hover:text-[#fbbb01]">
                        <i id="eye-icon" class="fas fa-eye"></i>
                        <i id="eye-off-icon" class="fas fa-eye-slash hidden"></i>
                    </button>
                </div>
                
                <?php if (!$has_pin): ?>
                    <div class="mt-3 bg-yellow-50 border border-yellow-200 p-2 rounded text-xs">
                        <div class="flex items-center">
                            <i class="fas fa-exclamation-triangle text-yellow-400 mr-2"></i>
                            <p class="text-yellow-700">Set transaction PIN to withdraw</p>
                        </div>
                        <a href="pin.php" class="text-yellow-700 underline mt-1 inline-block">
                            Set PIN now
                        </a>
                    </div>
                <?php endif; ?>
            </div>
            <div class="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center">
                <i class="fas fa-wallet text-yellow-500"></i>
            </div>
        </div>
        
        <!-- Action Buttons -->
        <div class="grid grid-cols-2 gap-3 mt-4">
            <a href="deposit.php"
               class="flex items-center justify-center bg-[#fbbb01] hover:bg-[#e0a800] text-white px-3 py-2 rounded-lg text-xs font-medium transition">
                <i class="fas fa-plus mr-1.5"></i>
                Deposit
            </a>
            
            <a href="withdraw.php"
               class="flex items-center justify-center text-[#fbbb01] border border-[#fbbb01] hover:bg-yellow-50 px-3 py-2 rounded-lg text-xs font-medium transition"
               <?= !$has_pin ? 'disabled title="Set transaction PIN first"' : '' ?>>
                <i class="fas fa-minus mr-1.5"></i>
                Withdraw
            </a>
            
            <a href="bank_accounts.php"
               class="flex items-center justify-center bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg text-xs font-medium transition col-span-2">
                <i class="fas fa-university mr-1.5"></i>
                <?= count($bank_accounts) > 0 ? 'Bank Accounts' : 'Add Bank Account' ?>
            </a>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="  p-4">
        <div class="flex justify-between items-center mb-3">
            <h3 class="text-base font-semibold">Quick Actions</h3>
            <button id="scroll-left" class="p-1.5 rounded-full bg-gray-100 text-gray-500 hover:bg-gray-200 text-xs">
                <i class="fas fa-chevron-left"></i>
            </button>
        </div>
        
        <div class="relative overflow-hidden">
            <div id="quick-actions-container" class="flex overflow-x-auto pb-3 space-x-3 scrollbar-hide">
                <div class="flex space-x-3" id="quick-actions-track">
                    <?php if (!empty($vendors)): ?>
                        <?php foreach ($vendors as $vendor): 
                            $business_type = strtolower($vendor['business_type'] ?? 'default');
                            $icon = isset($business_type_icons[$business_type]) ? $business_type_icons[$business_type] : $business_type_icons['default'];
                            $color_class = isset($business_type_colors[$business_type]) ? $business_type_colors[$business_type] : $business_type_colors['default'];
                            $color_parts = explode(' ', $color_class);
                            $bg_color = $color_parts[0] ?? 'bg-gray-100';
                            $text_color = $color_parts[1] ?? 'text-gray-500';
                            
                            $display_name = strlen($vendor['business_name']) > 10 
                                ? substr($vendor['business_name'], 0, 10) . '...' 
                                : $vendor['business_name'];
                                
                            $business_type_display = !empty($vendor['business_type']) ? ucfirst($vendor['business_type']) : 'Vendor';
                            $has_profile_picture = !empty($vendor['profile_picture']);
                            $profile_picture_path = $has_profile_picture ? '../uploads/' . $vendor['profile_picture'] : '';
                        ?>
                        <a href="vendor_details.php?id=<?= $vendor['id'] ?>" 
                           class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                            <?php if ($has_profile_picture && file_exists($profile_picture_path)): ?>
                                <div class="w-10 h-10 <?= $bg_color ?> rounded-full flex items-center justify-center mb-1.5 overflow-hidden">
                                    <img src="<?= $profile_picture_path ?>" 
                                         alt="<?= htmlspecialchars($vendor['business_name']) ?>" 
                                         class="w-full h-full object-cover">
                                </div>
                            <?php else: ?>
                                <div class="w-10 h-10 <?= $bg_color ?> rounded-full flex items-center justify-center mb-1.5">
                                    <i class="fas <?= $icon ?> <?= $text_color ?> text-lg"></i>
                                </div>
                            <?php endif; ?>
                            <span class="text-xs font-medium text-gray-700 text-center leading-tight"><?= htmlspecialchars($display_name) ?></span>
                        </a>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <!-- Default categories -->
                        <a href="vendors.php?category=cafeteria" 
                           class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                            <div class="w-10 h-10 bg-red-100 rounded-full flex items-center justify-center mb-1.5">
                                <i class="fas fa-utensils text-red-500 text-lg"></i>
                            </div>
                            <span class="text-xs font-medium text-gray-700">Cafeteria</span>
                        </a>
                        <a href="vendors.php?category=bookstore" 
                           class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                            <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center mb-1.5">
                                <i class="fas fa-book text-blue-500 text-lg"></i>
                            </div>
                            <span class="text-xs font-medium text-gray-700">Bookstore</span>
                        </a>
                        <a href="vendors.php?category=merchandise" 
                           class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                            <div class="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center mb-1.5">
                                <i class="fas fa-t-shirt text-purple-500 text-lg"></i>
                            </div>
                            <span class="text-xs font-medium text-gray-700">Merchandise</span>
                        </a>
                        <a href="vendors.php?category=transport" 
                           class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                            <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center mb-1.5">
                                <i class="fas fa-bus text-green-500 text-lg"></i>
                            </div>
                            <span class="text-xs font-medium text-gray-700">Transport</span>
                        </a>
                    <?php endif; ?>
                    
                    <!-- More button -->
                    <a href="vendors.php" 
                       class="bg-yellow-50 p-3 rounded-lg flex flex-col items-center hover:bg-yellow-100 transition flex-shrink-0 w-20 no-underline">
                        <div class="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center mb-1.5">
                            <i class="fas fa-ellipsis-h text-gray-500 text-lg"></i>
                        </div>
                        <span class="text-xs font-medium text-gray-700">More</span>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Bank Accounts -->
    <div class="  p-4">
        <div class="flex justify-between items-center mb-3">
            <h3 class="text-base font-semibold">Bank Accounts</h3>
            <a href="bank_accounts.php" class="text-[#fbbb01] hover:text-[#e0a800] text-xs font-medium">
                <?= count($bank_accounts) > 0 ? 'Manage' : 'Add' ?>
            </a>
        </div>
        
        <?php if (empty($bank_accounts)): ?>
            <div class="text-center py-4 text-gray-500 text-sm">
                <i class="fas fa-university text-gray-300 text-2xl mb-2 block"></i>
                <p>No bank accounts</p>
                <a href="bank_accounts.php" class="text-[#fbbb01] hover:text-[#e0a800] mt-1 inline-block text-xs">
                    Add bank account
                </a>
            </div>
        <?php else: ?>
            <div class="space-y-2">
                <?php foreach ($bank_accounts as $account): ?>
                    <div class="flex items-center p-2 bg-gray-50 hover:bg-gray-100 rounded-lg transition">
                        <div class="w-8 h-8 rounded-full bg-blue-100 flex items-center justify-center mr-2">
                            <i class="fas fa-university text-blue-500 text-xs"></i>
                        </div>
                        <div class="flex-1 min-w-0">
                            <p class="font-medium text-sm truncate"><?= htmlspecialchars($account['bank_name']) ?></p>
                            <p class="text-xs text-gray-500 truncate">
                                <?= htmlspecialchars($account['account_name']) ?> • ****<?= substr($account['account_number'], -4) ?>
                                <?= $account['is_primary'] ? ' • Primary' : '' ?>
                            </p>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Recent Transactions -->
    <div class="  p-4">
        <div class="flex justify-between items-center mb-3">
            <h3 class="text-base font-semibold">Recent Transactions</h3>
            <a href="wallet_history.php" class="text-[#fbbb01] hover:text-[#e0a800] text-xs font-medium">View All</a>
        </div>
        <div class="space-y-2">
            <?php if (empty($transactions)): ?>
                <div class="text-center py-4 text-gray-500 text-sm">
                    <i class="fas fa-exchange-alt text-gray-300 text-2xl mb-2 block"></i>
                    <p>No transactions yet</p>
                </div>
            <?php else: ?>
                <?php foreach ($transactions as $transaction): ?>
                    <div class="flex items-center p-2 hover:bg-gray-50 rounded-lg transition">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center mr-2 
                            <?= $transaction['transaction_type'] === 'credit' || $transaction['transaction_type'] === 'deposit' ? 'bg-green-100' : 'bg-red-100' ?>">
                            <?php if ($transaction['transaction_type'] === 'credit' || $transaction['transaction_type'] === 'deposit'): ?>
                                <i class="fas fa-arrow-down text-green-500 text-xs"></i>
                            <?php else: ?>
                                <i class="fas fa-arrow-up text-red-500 text-xs"></i>
                            <?php endif; ?>
                        </div>
                        <div class="flex-1 min-w-0">
                            <p class="font-medium text-sm truncate"><?= htmlspecialchars($transaction['description']) ?></p>
                            <p class="text-xs text-gray-500 truncate">
                                <?= date('M j', strtotime($transaction['created_at'])) ?>
                                • <?= ucfirst($transaction['status']) ?>
                            </p>
                        </div>
                        <p class="font-medium text-sm whitespace-nowrap
                            <?= ($transaction['transaction_type'] === 'credit' || $transaction['transaction_type'] === 'deposit') ? 'text-green-600' : 'text-red-600' ?>">
                            <?= ($transaction['transaction_type'] === 'credit' || $transaction['transaction_type'] === 'deposit') ? '+' : '-' ?>₦<?= number_format($transaction['amount'], 2) ?>
                        </p>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
    .scrollbar-hide::-webkit-scrollbar {
        display: none;
    }
    .scrollbar-hide {
        -ms-overflow-style: none;
        scrollbar-width: none;
    }
    
    #toggle-balance {
        cursor: pointer;
        transition: all 0.2s ease;
        background: none;
        border: none;
        padding: 0;
    }
    
    .no-underline {
        text-decoration: none;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const container = document.getElementById('quick-actions-container');
    const track = document.getElementById('quick-actions-track');
    const scrollLeftBtn = document.getElementById('scroll-left');
    
    let isDown = false;
    let startX;
    let scrollLeft;
    
    // Mouse drag functionality
    container.addEventListener('mousedown', (e) => {
        isDown = true;
        startX = e.pageX - container.offsetLeft;
        scrollLeft = container.scrollLeft;
    });
    
    container.addEventListener('mouseleave', () => {
        isDown = false;
    });
    
    container.addEventListener('mouseup', () => {
        isDown = false;
    });
    
    container.addEventListener('mousemove', (e) => {
        if(!isDown) return;
        e.preventDefault();
        const x = e.pageX - container.offsetLeft;
        const walk = (x - startX) * 2;
        container.scrollLeft = scrollLeft - walk;
    });
    
    // Touch drag functionality
    container.addEventListener('touchstart', (e) => {
        isDown = true;
        startX = e.touches[0].pageX - container.offsetLeft;
        scrollLeft = container.scrollLeft;
    });
    
    container.addEventListener('touchend', () => {
        isDown = false;
    });
    
    container.addEventListener('touchmove', (e) => {
        if(!isDown) return;
        const x = e.touches[0].pageX - container.offsetLeft;
        const walk = (x - startX) * 2;
        container.scrollLeft = scrollLeft - walk;
    });
    
    // Scroll left button
    scrollLeftBtn.addEventListener('click', () => {
        container.scrollBy({
            left: -150,
            behavior: 'smooth'
        });
    });
    
    // Show/hide scroll left button based on scroll position
    container.addEventListener('scroll', () => {
        if(container.scrollLeft > 0) {
            scrollLeftBtn.style.display = 'block';
        } else {
            scrollLeftBtn.style.display = 'none';
        }
    });
    
    // Handle balance toggle
    const toggleBalance = document.getElementById('toggle-balance');
    if (toggleBalance) {
        toggleBalance.addEventListener('click', function() {
            const balanceAmount = document.getElementById('balance-amount');
            const balanceMask = document.getElementById('balance-mask');
            const eyeIcon = document.getElementById('eye-icon');
            const eyeOffIcon = document.getElementById('eye-off-icon');
            
            if (balanceAmount.style.display === 'none') {
                balanceAmount.style.display = 'block';
                balanceMask.style.display = 'none';
                eyeIcon.classList.add('hidden');
                eyeOffIcon.classList.remove('hidden');
            } else {
                balanceAmount.style.display = 'none';
                balanceMask.style.display = 'block';
                eyeIcon.classList.remove('hidden');
                eyeOffIcon.classList.add('hidden');
            }
        });
    }
    
    // Handle wallet ID copy
    const copyWalletId = document.getElementById('copy-wallet-id');
    if (copyWalletId) {
        copyWalletId.addEventListener('click', function() {
            const walletId = document.getElementById('wallet-id').textContent;
            navigator.clipboard.writeText(walletId).then(() => {
                alert('Wallet ID copied!');
            }).catch(err => {
                console.error('Failed to copy: ', err);
            });
        });
    }
});
</script>