<?php 
ob_start();
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['user_id'])) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch student data
$student_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    $_SESSION['error'] = "Student record not found";
    header("Location: dashboard.php");
    exit();
}

ob_end_flush();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto">
  <!-- Success/Error Messages -->
  <div id="messageContainer"></div>

  <div class="dark:bg-gray-800 p-6 border border-gray-100 dark:border-gray-700">
    <div class="flex justify-between items-center mb-6">
      <h2 class="text-xl font-bold dark:text-white">Update Profile Picture</h2>
      <a href="profile.php" class="text-sm text-primary-500 hover:text-primary-600 dark:hover:text-primary-400">
        <i class="fas fa-arrow-left mr-1"></i> Back to Profile
      </a>
    </div>
    
    <form id="profilePictureForm" method="POST" enctype="multipart/form-data" class="space-y-6">
      <div class="flex flex-col items-center space-y-6">
        <div class="relative">
          <img id="imagePreview" src="<?= !empty($student['profile_picture']) ? htmlspecialchars($student['profile_picture']) : 'assets/images/default-avatar.png' ?>" 
               alt="Current Profile Picture" 
               class="w-32 h-32 rounded-full object-cover border-4 border-primary-100 dark:border-primary-800">
          <div id="loadingSpinner" class="absolute inset-0 flex items-center justify-center bg-black bg-opacity-50 rounded-full hidden">
            <div class="w-8 h-8 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
          </div>
        </div>
        
        <div class="w-full max-w-md">
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Choose New Photo</label>
          <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md dark:border-gray-600">
            <div class="space-y-1 text-center">
              <div class="flex text-sm text-gray-600 dark:text-gray-400 justify-center">
                <label for="profile_picture" class="relative cursor-pointer rounded-md font-medium text-primary-600 hover:text-primary-500 focus-within:outline-none">
                  <span>Upload a file</span>
                  <input id="profile_picture" name="profile_picture" type="file" class="sr-only" accept="image/*" onchange="previewImage(this)">
                </label>
              </div>
              <p class="text-xs text-gray-500 dark:text-gray-400">
                PNG, JPG, GIF up to 5MB
              </p>
            </div>
          </div>
          <small class="text-red-500 text-sm mt-1 hidden" id="file_error"></small>
        </div>
      </div>
      
      <div class="flex justify-end space-x-3 pt-4 border-t dark:border-gray-700">
        <a href="profile.php" class="px-4 py-2 text-sm bg-gray-200 hover:bg-gray-300 rounded-lg dark:bg-gray-600 dark:hover:bg-gray-700 dark:text-white">
          Cancel
        </a>
        <button type="submit" 
                id="submitBtn"
                class="px-4 py-2 text-sm bg-primary-500 hover:bg-primary-600 text-white rounded-lg flex items-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed">
          <span id="btnText">Update Photo</span>
          <div id="btnSpinner" class="hidden w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
        </button>
      </div>
    </form>
  </div>
</main>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
function previewImage(input) {
  const fileError = document.getElementById('file_error');
  fileError.classList.add('hidden');
  
  if (input.files && input.files[0]) {
    const file = input.files[0];
    
    // Validate file type
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    if (!allowedTypes.includes(file.type)) {
      fileError.textContent = 'Please select a valid image file (JPEG, PNG, GIF)';
      fileError.classList.remove('hidden');
      input.value = '';
      return;
    }
    
    // Validate file size (5MB)
    if (file.size > 5 * 1024 * 1024) {
      fileError.textContent = 'File size must be less than 5MB';
      fileError.classList.remove('hidden');
      input.value = '';
      return;
    }
    
    var reader = new FileReader();
    reader.onload = function(e) {
      document.getElementById('imagePreview').src = e.target.result;
    }
    reader.readAsDataURL(input.files[0]);
  }
}

// AJAX Form Submission
$(document).ready(function() {
  $('#profilePictureForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const submitBtn = $('#submitBtn');
    const btnText = $('#btnText');
    const btnSpinner = $('#btnSpinner');
    const loadingSpinner = $('#loadingSpinner');
    const fileError = $('#file_error');
    
    // Reset errors
    fileError.addClass('hidden').text('');
    
    // Validate file
    const fileInput = document.getElementById('profile_picture');
    if (!fileInput.files || !fileInput.files[0]) {
      fileError.text('Please select a file to upload').removeClass('hidden');
      return;
    }
    
    // Show loading states
    submitBtn.prop('disabled', true);
    btnText.text('Uploading...');
    btnSpinner.removeClass('hidden');
    loadingSpinner.removeClass('hidden');
    
    // Send AJAX request
    $.ajax({
      url: 'ajax_update_profile_picture.php',
      type: 'POST',
      data: formData,
      processData: false,
      contentType: false,
      success: function(response) {
        // Reset button state
        submitBtn.prop('disabled', false);
        btnText.text('Update Photo');
        btnSpinner.addClass('hidden');
        loadingSpinner.addClass('hidden');
        
        if (response.success) {
          showMessage(response.message, 'success');
          
          // Update profile picture preview
          if (response.new_image_url) {
            $('#imagePreview').attr('src', response.new_image_url);
          }
          
          // Clear file input
          document.getElementById('profile_picture').value = '';
          
          // Update profile picture in navigation if exists
          const navProfilePic = document.querySelector('.nav-profile-pic');
          if (navProfilePic && response.new_image_url) {
            navProfilePic.src = response.new_image_url;
          }
          
        } else {
          showMessage(response.message, 'error');
          if (response.errors) {
            for (const field in response.errors) {
              $(`#${field}_error`).text(response.errors[field]).removeClass('hidden');
            }
          }
        }
      },
      error: function(xhr, status, error) {
        // Reset button state
        submitBtn.prop('disabled', false);
        btnText.text('Update Photo');
        btnSpinner.addClass('hidden');
        loadingSpinner.addClass('hidden');
        
        console.error('AJAX Error:', error);
        showMessage('An error occurred while uploading. Please try again.', 'error');
      }
    });
  });
});

function showMessage(message, type) {
  const messageContainer = document.getElementById('messageContainer');
  const messageClass = type === 'success' 
    ? 'bg-green-100 border-green-400 text-green-700' 
    : 'bg-red-100 border-red-400 text-red-700';
  
  const messageHTML = `
    <div class="mb-4 border px-4 py-3 rounded relative ${messageClass}" role="alert">
      <span class="block sm:inline">${message}</span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.remove()">
        <svg class="fill-current h-6 w-6 ${type === 'success' ? 'text-green-500' : 'text-red-500'}" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
          <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
        </svg>
      </span>
    </div>
  `;
  
  messageContainer.innerHTML = messageHTML;
  
  // Auto-remove success messages after 5 seconds
  if (type === 'success') {
    setTimeout(() => {
      const alert = messageContainer.querySelector('.bg-green-100');
      if (alert) {
        alert.remove();
      }
    }, 5000);
  }
}
</script>

<?php include 'include/footer.php'; ?>