<?php
require '../config/conn.php';
require '../config/session.php';
require '../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use PHPMailer\PHPMailer\SMTP;

// Set content type to JSON
header('Content-Type: application/json');

// Check if it's a POST request and has the required fields
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // Check if register field exists OR if we have the form fields
    if (isset($_POST['register']) || (isset($_POST['full_name']) && isset($_POST['email']))) {
        
        // Retrieve and sanitize form data (removed phone and gender)
        $full_name = trim($_POST['full_name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        $newsletter_opt_in = isset($_POST['newsletter_opt_in']) ? 1 : 0;
        
        // Store form data in session in case of errors
        $_SESSION['form_data'] = [
            'full_name' => $full_name,
            'email' => $email,
            'newsletter_opt_in' => $newsletter_opt_in
        ];
        
        // Validate inputs
        $errors = [];
        
        // Name validation
        if (empty($full_name)) {
            $errors['full_name'] = "Full name is required";
        } elseif (!preg_match("/^[a-zA-Z ]*$/", $full_name)) {
            $errors['full_name'] = "Only letters and spaces allowed in name";
        }
        
        // Email validation
        if (empty($email)) {
            $errors['email'] = "Email is required";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = "Invalid email format";
        } else {
            // Check if email already exists
            $stmt = $conn->prepare("SELECT id FROM students WHERE email = ?");
            $stmt->bind_param("s", $email);
            $stmt->execute();
            $stmt->store_result();
            
            if ($stmt->num_rows > 0) {
                $errors['email'] = "Email already registered";
            }
            $stmt->close();
        }
        
        // Password validation
        if (empty($password)) {
            $errors['password'] = "Password is required";
        } elseif (strlen($password) < 8) {
            $errors['password'] = "Password must be at least 8 characters";
        } elseif (!preg_match("/[A-Z]/", $password)) {
            $errors['password'] = "Password must contain at least one uppercase letter";
        } elseif (!preg_match("/[a-z]/", $password)) {
            $errors['password'] = "Password must contain at least one lowercase letter";
        } elseif (!preg_match("/[0-9]/", $password)) {
            $errors['password'] = "Password must contain at least one number";
        } elseif (!preg_match("/[^a-zA-Z0-9]/", $password)) {
            $errors['password'] = "Password must contain at least one special character";
        } elseif ($password !== $confirm_password) {
            $errors['confirm_password'] = "Passwords do not match";
        }
        
        // Terms acceptance validation
        if (!isset($_POST['accept_terms'])) {
            $errors['terms'] = "You must accept the terms and conditions";
        }
        
        // If there are errors, return them as JSON
        if (!empty($errors)) {
            echo json_encode([
                'success' => false,
                'errors' => $errors,
                'message' => 'Please fix the errors below.',
                'debug' => 'Validation errors found'
            ]);
            exit();
        }
        
        // If no errors, proceed with registration
        // Generate unique student ID starting with ALU (Aletheia University)
        $student_id = 'ALU' . strtoupper(substr(md5(uniqid()), 0, 8));
        
        // Generate registration number
        $student_reg_number = 'REG' . date('Y') . str_pad(mt_rand(1, 99999), 5, '0', STR_PAD_LEFT);
        
        // Generate verification token
        $verification_code = bin2hex(random_bytes(32));
        $verification_code_expiry = date('Y-m-d H:i:s', strtotime('+30 minutes'));
        
        // Hash password
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Insert into database (removed phone and gender fields)
        $stmt = $conn->prepare("INSERT INTO students (student_id, reg_number, full_name, email, password, verification_code, verification_code_expiry, newsletter_opt_in) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("sssssssi", $student_id, $student_reg_number, $full_name, $email, $hashed_password, $verification_code, $verification_code_expiry, $newsletter_opt_in);
        
        if ($stmt->execute()) {
            // Send verification email
            $mail = new PHPMailer(true);
            
            try {
                $mail->isSMTP();
                $mail->Host       = 'smtp.gmail.com';
                $mail->SMTPAuth   = true;
                $mail->Username   = 'olorodemicheal2274@gmail.com';
                $mail->Password   = 'farq ptyc vjtl eses';
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
                $mail->Port       = 465;
                
                // Recipients - Updated to Aletheia University
                $mail->setFrom('noreply@aletheia.edu', 'Aletheia University Campus Services');
                $mail->addAddress($email, $full_name);
                
                // Verification link
                $verification_link = "https://adinoheightschols.com/student/verify.php?code=$verification_code&email=" . urlencode($email);
                
                // Content - Updated with Aletheia University branding
                $mail->isHTML(true);
                $mail->Subject = 'Welcome to Aletheia University Campus Services - Verify Your Account';
                
                // HTML email body with Aletheia University branding
                $mail->Body = "
<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Account Verification - Aletheia University</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            margin: 0;
            padding: 0;
        }
        .container {
            max-width: 600px;
            margin: 0 auto;
            padding: 20px;
        }
        .university-header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 2px solid #2C3E50;
        }
        .university-name {
            font-size: 28px;
            font-weight: bold;
            color: #2C3E50;
            margin: 0;
            letter-spacing: 1px;
        }
        .subtitle {
            font-size: 16px;
            color: #666;
            margin: 5px 0 0 0;
        }
        .greeting {
            color: #2C3E50;
            font-size: 18px;
            margin-top: 0;
        }
        .student-info {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border-left: 4px solid #FBBB01;
        }
        .student-info p {
            margin: 8px 0;
            font-size: 15px;
        }
        .verify-button {
            text-align: center;
            margin: 30px 0;
        }
        .verify-button a {
            background-color: #2C3E50;
            color: #FFFFFF;
            padding: 12px 30px;
            text-decoration: none;
            border-radius: 4px;
            display: inline-block;
            font-weight: bold;
            border: 2px solid #FBBB01;
            font-size: 16px;
        }
        .verify-button a:hover {
            background-color: #1a252f;
        }
        .verification-link {
            background-color: #fff3cd;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
            border: 1px solid #ffeaa7;
            word-break: break-all;
            font-size: 14px;
        }
        .verification-link p {
            margin: 0;
            color: #856404;
        }
        .footer {
            text-align: center;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #eee;
            color: #777;
            font-size: 12px;
        }
        .highlight {
            color: #FBBB01;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <div class='container'>
        <div class='university-header'>
            <h1 class='university-name'>ALETHEIA UNIVERSITY</h1>
            <p class='subtitle'>Campus Services Account Verification</p>
        </div>
        
        <h2 class='greeting'>Welcome, $full_name!</h2>
        
        <p>Thank you for registering with Aletheia University Campus Services. Your account has been created successfully.</p>
        
        <div class='student-info'>
            <p><strong>Your Student ID:</strong> $student_id</p>
            <p><strong>Registration Number:</strong> $student_reg_number</p>
        </div>
        
        <p>To complete your registration and activate your account, please verify your email address by clicking the button below:</p>
        
        <div class='verify-button'>
            <a href='$verification_link'>Verify Email Address</a>
        </div>
        
        <p style='text-align: center; color: #666; font-size: 14px;'>Or copy and paste this link in your browser:</p>
        
        <div class='verification-link'>
            <p>$verification_link</p>
        </div>
        
        <div style='background-color: #e8f4fd; padding: 15px; border-radius: 5px; margin: 25px 0; border-left: 4px solid #2196F3;'>
            <p style='margin: 0; color: #0c5460;'><strong>⚠️ Important:</strong> This verification link will expire in <span class='highlight'>30 minutes</span>.</p>
            <p style='margin: 10px 0 0 0; color: #0c5460;'>If the link expires, you can request a new verification link from the login page.</p>
        </div>
        
        <p>Once verified, you can access all campus services including:</p>
        <ul>
           
            <li>Food Fee payment and receipts</li>

        </ul>
        
        <p>If you have any questions or need assistance, please contact the IT Help Desk.</p>
        
        <p>Best regards,<br>
        <strong>Aletheia University Campus Services</strong></p>
        
        <div class='footer'>
            <p>Aletheia University IT Services</p>
            <p>This is an automated message. Please do not reply to this email.</p>
            <p>© " . date('Y') . " Aletheia University. All rights reserved.</p>
        </div>
    </div>
</body>
</html>
                ";
                
                // Plain text version
                $mail->AltBody = "ALETHEIA UNIVERSITY\nCampus Services Account Verification\n\n" .
                                "Welcome, $full_name!\n\n" .
                                "Thank you for registering with Aletheia University Campus Services.\n\n" .
                                "Your Student ID: $student_id\n" .
                                "Your Registration Number: $student_reg_number\n\n" .
                                "Please verify your email address to activate your account:\n" .
                                "$verification_link\n\n" .
                                "⚠️ Important: This verification link will expire in 30 minutes.\n\n" .
                                "Once verified, you can access all campus services including course registration, fee payment, exam schedules, and campus announcements.\n\n" .
                                "If you have any questions or need assistance, please contact the IT Help Desk.\n\n" .
                                "Best regards,\n" .
                                "Aletheia University Campus Services\n\n" .
                                "This is an automated message. Please do not reply to this email.\n" .
                                "© " . date('Y') . " Aletheia University. All rights reserved.";
                
                $mail->send();
                
                // Clear form data from session
                unset($_SESSION['form_data']);
                
                // Return success response
                echo json_encode([
                    'success' => true,
                    'message' => "Registration successful! We've sent a verification email to $email. Please check your inbox (and spam folder).",
                    'student_id' => $student_id
                ]);
                exit();
                
            } catch (Exception $e) {
                // Email sending failed
                echo json_encode([
                    'success' => false,
                    'message' => "Registration completed but we couldn't send a verification email. Please contact support.",
                    'debug' => 'Email error: ' . $e->getMessage()
                ]);
                exit();
            }
        } else {
            echo json_encode([
                'success' => false,
                'message' => "Registration failed. Please try again.",
                'debug' => 'Database error'
            ]);
            exit();
        }
        
        $stmt->close();
    } else {
        // Missing required fields
        echo json_encode([
            'success' => false,
            'message' => 'Invalid request - missing form data.',
            'debug' => 'POST data: ' . print_r($_POST, true)
        ]);
        exit();
    }
} else {
    // Not a POST request
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method.',
        'debug' => 'Request method: ' . $_SERVER['REQUEST_METHOD']
    ]);
    exit();
}
?>