<?php
// process_withdrawal_fixed.php

// ============================================
// PHASE 1: PREVENT ANY OUTPUT
// ============================================
// Disable ALL error display
ini_set('display_errors', '0');
ini_set('display_startup_errors', '0');
error_reporting(0);

// Start output buffering at the VERY beginning
if (ob_get_level() == 0) {
    ob_start();
}

// ============================================
// PHASE 2: LOAD CONFIG SILENTLY
// ============================================
// Include config file silently
function loadConfigSilently() {
    // Capture any output from config
    ob_start();
    require_once '../config/conn.php';
    $output = ob_get_clean();
    
    if (!empty($output)) {
        error_log("Config file output: " . $output);
        return ['error' => 'Config file has invalid output'];
    }
    
    // Check if connection was established
    if (!isset($GLOBALS['conn']) || !$GLOBALS['conn']) {
        return ['error' => 'Database connection failed'];
    }
    
    return ['success' => true, 'conn' => $GLOBALS['conn']];
}

$configResult = loadConfigSilently();
if (isset($configResult['error'])) {
    // Clear any buffers
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'message' => 'System error: ' . $configResult['error']
    ]);
    exit;
}

$conn = $configResult['conn'];

// ============================================
// PHASE 3: SESSION HANDLING
// ============================================
// Start session without errors
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

// ============================================
// PHASE 4: CLEAR BUFFERS & SET HEADERS
// ============================================
// Clear all buffers
while (ob_get_level()) {
    ob_end_clean();
}

// Set JSON header
header('Content-Type: application/json; charset=utf-8');

// ============================================
// PHASE 5: VALIDATE REQUEST
// ============================================
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

// ============================================
// PHASE 6: PROCESS REQUEST
// ============================================
$response = processWithdrawalRequest($conn, $_SESSION['user_id'], $_POST);

// Output JSON response
echo json_encode($response);
exit;

// ============================================
// PROCESSING FUNCTION
// ============================================
function processWithdrawalRequest($conn, $student_id, $postData) {
    // Validate inputs
    $wallet_id = isset($postData['wallet_id']) ? intval($postData['wallet_id']) : 0;
    $amount = isset($postData['amount']) ? floatval($postData['amount']) : 0;
    $bank_account_id = isset($postData['bank_account_id']) ? intval($postData['bank_account_id']) : 0;
    $transaction_pin = isset($postData['transaction_pin']) ? trim($postData['transaction_pin']) : '';

    // Validation
    if ($amount < 100) {
        return ['success' => false, 'message' => 'Minimum withdrawal is ₦100'];
    }

    if (empty($bank_account_id)) {
        return ['success' => false, 'message' => 'Bank account is required'];
    }

    if (strlen($transaction_pin) !== 4 || !ctype_digit($transaction_pin)) {
        return ['success' => false, 'message' => 'Invalid transaction PIN'];
    }

    try {
        // Verify PIN
        $pin_stmt = $conn->prepare("SELECT pin_hash FROM wallet_pins WHERE student_id = ?");
        if (!$pin_stmt) {
            return ['success' => false, 'message' => 'System error: PIN verification failed'];
        }
        
        $pin_stmt->bind_param("i", $student_id);
        $pin_stmt->execute();
        $pin_result = $pin_stmt->get_result();
        $pin_stmt->close();

        if ($pin_result->num_rows === 0) {
            return ['success' => false, 'message' => 'Transaction PIN not set'];
        }

        $pin_data = $pin_result->fetch_assoc();
        if (!password_verify($transaction_pin, $pin_data['pin_hash'])) {
            return ['success' => false, 'message' => 'Incorrect transaction PIN'];
        }

        // Check wallet
        $wallet_stmt = $conn->prepare("SELECT id, balance FROM student_wallets WHERE id = ? AND student_id = ?");
        if (!$wallet_stmt) {
            return ['success' => false, 'message' => 'System error: Wallet check failed'];
        }
        
        $wallet_stmt->bind_param("ii", $wallet_id, $student_id);
        $wallet_stmt->execute();
        $wallet_result = $wallet_stmt->get_result();
        $wallet_stmt->close();

        if ($wallet_result->num_rows === 0) {
            return ['success' => false, 'message' => 'Wallet not found'];
        }

        $wallet = $wallet_result->fetch_assoc();
        
        // Calculate with fee
        $FIXED_FEE = 25.00;
        $total_deduction = $amount + $FIXED_FEE;
        
        if ($wallet['balance'] < $total_deduction) {
            return ['success' => false, 'message' => 'Insufficient balance (including ₦' . $FIXED_FEE . ' fee)'];
        }

        // Get bank account
        $bank_stmt = $conn->prepare("SELECT * FROM student_bank_accounts WHERE id = ? AND student_id = ?");
        if (!$bank_stmt) {
            return ['success' => false, 'message' => 'System error: Bank account check failed'];
        }
        
        $bank_stmt->bind_param("ii", $bank_account_id, $student_id);
        $bank_stmt->execute();
        $bank_result = $bank_stmt->get_result();
        $bank_stmt->close();

        if ($bank_result->num_rows === 0) {
            return ['success' => false, 'message' => 'Bank account not found'];
        }

        $bank_account = $bank_result->fetch_assoc();

        // Start transaction
        $conn->begin_transaction();

        try {
            // Deduct from wallet
            $update_stmt = $conn->prepare("UPDATE student_wallets SET balance = balance - ? WHERE id = ?");
            $update_stmt->bind_param("di", $total_deduction, $wallet_id);
            
            if (!$update_stmt->execute()) {
                throw new Exception('Failed to update wallet');
            }
            $update_stmt->close();

            // Create transaction record
            $reference = 'WDR' . time() . mt_rand(1000, 9999);
            $description = "Withdrawal to " . $bank_account['bank_name'] . " (****" . substr($bank_account['account_number'], -4) . ")";
            
            // Try to insert into withdrawal_transactions
            $table_check = $conn->query("SHOW TABLES LIKE 'withdrawal_transactions'");
            
            if ($table_check && $table_check->num_rows > 0) {
                $transaction_stmt = $conn->prepare("
                    INSERT INTO withdrawal_transactions (
                        student_id, wallet_id, bank_account_id, amount, fee, 
                        total_deduction, bank_name, account_number, account_name, 
                        bank_code, description, status, reference, created_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?, NOW())
                ");
                
                if ($transaction_stmt) {
                    $transaction_stmt->bind_param(
                        "iiiddsssssss", 
                        $student_id, $wallet_id, $bank_account_id, $amount, $FIXED_FEE,
                        $total_deduction, $bank_account['bank_name'], $bank_account['account_number'],
                        $bank_account['account_name'], $bank_account['bank_code'], $description, $reference
                    );
                }
            } else {
                // Fallback
                $transaction_stmt = $conn->prepare("
                    INSERT INTO wallet_transactions (
                        student_id, wallet_id, bank_account_id, amount, fee,
                        transaction_type, description, status, reference, created_at
                    ) VALUES (?, ?, ?, ?, ?, 'withdrawal', ?, 'pending', ?, NOW())
                ");
                
                if ($transaction_stmt) {
                    $transaction_stmt->bind_param(
                        "iiiddsss",
                        $student_id, $wallet_id, $bank_account_id, $total_deduction,
                        $FIXED_FEE, $description, $reference
                    );
                }
            }
            
            if (!$transaction_stmt || !$transaction_stmt->execute()) {
                throw new Exception('Failed to create transaction record');
            }
            
            $transaction_id = $conn->insert_id;
            $transaction_stmt->close();
            
            // Record fee earnings
            $fee_stmt = $conn->prepare("
                INSERT INTO business_earnings (transaction_type, amount, student_id, description, created_at)
                VALUES ('withdrawal_fee', ?, ?, 'Withdrawal fee', NOW())
            ");
            
            if ($fee_stmt) {
                $fee_stmt->bind_param("di", $FIXED_FEE, $student_id);
                $fee_stmt->execute();
                $fee_stmt->close();
            }
            
            $conn->commit();
            
            return [
                'success' => true,
                'message' => 'Withdrawal request submitted successfully',
                'transaction_id' => $transaction_id,
                'reference' => $reference,
                'amount' => $amount,
                'fee' => $FIXED_FEE,
                'total_deduction' => $total_deduction
            ];
            
        } catch (Exception $e) {
            $conn->rollback();
            return ['success' => false, 'message' => 'Transaction failed: ' . $e->getMessage()];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'System error: ' . $e->getMessage()];
    }
}