<?php 
ob_start();
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['user_id'])) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch student data
$student_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    $_SESSION['error'] = "Student record not found";
    header("Location: dashboard.php");
    exit();
}

ob_end_flush();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto" style="margin-bottom:80px">
  <!-- Success/Error Messages -->
  <div id="messageContainer"></div>

  <div class="dark:bg-gray-800 p-6 border border-gray-100 dark:border-gray-700">
    <div class="flex justify-between items-center mb-6">
      <h2 class="text-xl font-bold dark:text-white">Edit Profile</h2>
      <a href="profile.php" class="text-sm text-primary-500 hover:text-primary-600 dark:hover:text-primary-400">
        <i class="fas fa-arrow-left mr-1"></i> Back to Profile
      </a>
    </div>
    
    <form id="profileEditForm" method="POST" class="space-y-6">
      <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
        <!-- Personal Information -->
        <div>
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Full Name</label>
          <input type="text" name="full_name" value="<?= htmlspecialchars($student['full_name']) ?>" 
                 class="w-full px-3 py-2 border rounded-lg dark:bg-gray-700 dark:border-gray-600 dark:text-white" required>
          <small class="text-red-500 text-sm mt-1 hidden" id="full_name_error"></small>
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Phone Number</label>
          <input type="text" name="phone" value="<?= htmlspecialchars($student['phone']) ?>" 
                 class="w-full px-3 py-2 border rounded-lg dark:bg-gray-700 dark:border-gray-600 dark:text-white">
          <small class="text-red-500 text-sm mt-1 hidden" id="phone_error"></small>
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Gender</label>
          <select name="gender" class="w-full px-3 py-2 border rounded-lg dark:bg-gray-700 dark:border-gray-600 dark:text-white">
            <option value="">Select Gender</option>
            <option value="male" <?= $student['gender'] == 'male' ? 'selected' : '' ?>>Male</option>
            <option value="female" <?= $student['gender'] == 'female' ? 'selected' : '' ?>>Female</option>
            <option value="other" <?= $student['gender'] == 'other' ? 'selected' : '' ?>>Other</option>
          </select>
          <small class="text-red-500 text-sm mt-1 hidden" id="gender_error"></small>
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Date of Birth</label>
          <input type="date" name="dob" value="<?= !empty($student['dob']) ? htmlspecialchars($student['dob']) : '' ?>" 
                 class="w-full px-3 py-2 border rounded-lg dark:bg-gray-700 dark:border-gray-600 dark:text-white">
          <small class="text-red-500 text-sm mt-1 hidden" id="dob_error"></small>
        </div>
        
        <div>
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Nationality</label>
          <input type="text" name="nationality" value="<?= !empty($student['nationality']) ? htmlspecialchars($student['nationality']) : '' ?>" 
                 class="w-full px-3 py-2 border rounded-lg dark:bg-gray-700 dark:border-gray-600 dark:text-white">
          <small class="text-red-500 text-sm mt-1 hidden" id="nationality_error"></small>
        </div>
        
        <!-- Academic Information REMOVED -->
      </div>
      
      <div class="flex justify-end space-x-3 pt-4 border-t dark:border-gray-700">
        <a href="profile.php" class="px-4 py-2 text-sm bg-gray-200 hover:bg-gray-300 rounded-lg dark:bg-gray-600 dark:hover:bg-gray-700 dark:text-white">
          Cancel
        </a>
        <button type="submit" 
                id="submitBtn"
                class="px-4 py-2 text-sm bg-primary-500 hover:bg-primary-600 text-white rounded-lg flex items-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed">
          <span id="btnText">Save Changes</span>
          <div id="btnSpinner" class="hidden w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
        </button>
      </div>
    </form>
  </div>
</main>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
$(document).ready(function() {
  $('#profileEditForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = $(this).serialize();
    const submitBtn = $('#submitBtn');
    const btnText = $('#btnText');
    const btnSpinner = $('#btnSpinner');
    
    // Clear previous errors
    $('.text-red-500').addClass('hidden').text('');
    
    // Show loading state
    submitBtn.prop('disabled', true);
    btnText.text('Saving...');
    btnSpinner.removeClass('hidden');
    
    // Send AJAX request
    $.ajax({
      url: 'ajax_update_profile.php',
      type: 'POST',
      data: formData,
      success: function(response) {
        // Reset button state
        submitBtn.prop('disabled', false);
        btnText.text('Save Changes');
        btnSpinner.addClass('hidden');
        
        if (response.success) {
          showMessage(response.message, 'success');
          
          // Redirect after success if needed
          if (response.redirect) {
            setTimeout(() => {
              window.location.href = response.redirect;
            }, 1500);
          }
        } else {
          showMessage(response.message, 'error');
          
          // Display field-specific errors
          if (response.errors) {
            for (const field in response.errors) {
              $(`#${field}_error`).text(response.errors[field]).removeClass('hidden');
            }
          }
        }
      },
      error: function(xhr, status, error) {
        // Reset button state
        submitBtn.prop('disabled', false);
        btnText.text('Save Changes');
        btnSpinner.addClass('hidden');
        
        console.error('AJAX Error:', error);
        showMessage('An error occurred while saving. Please try again.', 'error');
      }
    });
  });
  
  // Real-time validation for phone number
  $('input[name="phone"]').on('input', function() {
    const phone = $(this).val();
    const phoneError = $('#phone_error');
    
    if (phone && !/^[\d\s\-\+\(\)]{10,15}$/.test(phone)) {
      phoneError.text('Please enter a valid phone number').removeClass('hidden');
    } else {
      phoneError.addClass('hidden');
    }
  });
  
  // Real-time validation for full name
  $('input[name="full_name"]').on('input', function() {
    const name = $(this).val();
    const nameError = $('#full_name_error');
    
    if (name && !/^[a-zA-Z\s]{2,50}$/.test(name)) {
      nameError.text('Please enter a valid name (letters and spaces only)').removeClass('hidden');
    } else {
      nameError.addClass('hidden');
    }
  });
  
  // Date validation - ensure not future date
  $('input[name="dob"]').on('change', function() {
    const dob = new Date($(this).val());
    const today = new Date();
    const dobError = $('#dob_error');
    
    if (dob > today) {
      dobError.text('Date of birth cannot be in the future').removeClass('hidden');
    } else {
      dobError.addClass('hidden');
    }
  });
});

function showMessage(message, type) {
  const messageContainer = document.getElementById('messageContainer');
  const messageClass = type === 'success' 
    ? 'bg-green-100 border-green-400 text-green-700' 
    : 'bg-red-100 border-red-400 text-red-700';
  
  const messageHTML = `
    <div class="mb-4 border px-4 py-3 rounded relative ${messageClass}" role="alert">
      <span class="block sm:inline">${message}</span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.remove()">
        <svg class="fill-current h-6 w-6 ${type === 'success' ? 'text-green-500' : 'text-red-500'}" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
          <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
        </svg>
      </span>
    </div>
  `;
  
  messageContainer.innerHTML = messageHTML;
  
  // Auto-remove success messages after 5 seconds
  if (type === 'success') {
    setTimeout(() => {
      const alert = messageContainer.querySelector('.bg-green-100');
      if (alert) {
        alert.remove();
      }
    }, 5000);
  }
}
</script>

<?php include 'include/footer.php'; ?>