<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
require '../config/conn.php';

// Set timezone to UTC for consistency
date_default_timezone_set('UTC');

// Also set MySQL timezone to UTC
if (isset($conn) && $conn) {
    $conn->query("SET time_zone = '+00:00'");
}

// Check if token is valid
if (!isset($_GET['token']) || !isset($_GET['email'])) {
    $_SESSION['reset_error'] = "Invalid password reset link";
    header("Location: forgot_password.php");
    exit();
}

$token = $_GET['token'];
$email = $_GET['email'];

// Check if token exists and is not expired - USE UTC_TIMESTAMP() instead of NOW()
$query = "SELECT id, reset_token_expiry, full_name FROM students 
          WHERE email = ? AND reset_token = ? AND reset_token_expiry > UTC_TIMESTAMP()";
$stmt = $conn->prepare($query);
$stmt->bind_param("ss", $email, $token);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    $_SESSION['reset_error'] = "Invalid or expired password reset link";
    header("Location: forgot_password.php");
    exit();
}

// Store user info in session
$user = $result->fetch_assoc();
$_SESSION['reset_user_id'] = $user['id'];
$_SESSION['reset_user_name'] = $user['full_name'];

// Initialize variables
$errors = [
    'password' => '',
    'confirm_password' => ''
];
$success_message = '';
$show_form = true;

// If form was submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reset_password'])) {
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    // Validate password
    if (empty($password)) {
        $errors['password'] = "Password is required";
    } elseif (strlen($password) < 8) {
        $errors['password'] = "Password must be at least 8 characters";
    } elseif (!preg_match("/[A-Z]/", $password)) {
        $errors['password'] = "Password must contain at least one uppercase letter";
    } elseif (!preg_match("/[a-z]/", $password)) {
        $errors['password'] = "Password must contain at least one lowercase letter";
    } elseif (!preg_match("/[0-9]/", $password)) {
        $errors['password'] = "Password must contain at least one number";
    } elseif (!preg_match("/[^A-Za-z0-9]/", $password)) {
        $errors['password'] = "Password must contain at least one special character";
    }
    
    // Validate confirm password
    if (empty($confirm_password)) {
        $errors['confirm_password'] = "Please confirm your password";
    } elseif ($password !== $confirm_password) {
        $errors['confirm_password'] = "Passwords do not match";
    }
    
    // If no errors, update password
    if (empty($errors['password']) && empty($errors['confirm_password'])) {
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Update password and clear reset token - verify token again to prevent tampering
        $update_query = "UPDATE students SET password = ?, reset_token = NULL, reset_token_expiry = NULL 
                         WHERE email = ? AND reset_token = ? AND id = ?";
        $update_stmt = $conn->prepare($update_query);
        $update_stmt->bind_param("sssi", $hashed_password, $email, $token, $_SESSION['reset_user_id']);
        
        if ($update_stmt->execute()) {
            // Password reset successful - show success message on same page
            $success_message = "Password reset successfully! You can now login with your new password.";
            $show_form = false; // Hide the form
            
            // Clear session data
            unset($_SESSION['reset_user_id']);
            unset($_SESSION['reset_user_name']);
        } else {
            $_SESSION['error_messages'] = ["Failed to update password. Please try again."];
            header("Location: reset_password.php?token=$token&email=$email");
            exit();
        }
    } else {
        $_SESSION['reset_errors'] = $errors;
        header("Location: reset_password.php?token=$token&email=$email");
        exit();
    }
}

// If there are validation errors from form submission, display them
if (isset($_SESSION['reset_errors'])) {
    $errors = array_merge($errors, $_SESSION['reset_errors']);
    unset($_SESSION['reset_errors']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Campus Services - Set New Password</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Poppins', sans-serif; }
    </style>
</head>
<body class=" bg-gray-100 flex items-center justify-center p-4">
    <div class="w-full max-w-sm">
        <!-- Logo and Header -->
        <div class="text-center mb-6">
            <div class="w-16 h-16 mx-auto mb-3 flex items-center justify-center">
                <img src="../img/schools.png" alt="School Logo" class="max-w-full max-h-full">
            </div>
            <h1 class="text-xl font-bold text-gray-800 mb-1">Set New Password</h1>
            <p class="text-gray-500 text-sm">Create a new password for your account</p>
        </div>

        <!-- Display success message -->
        <?php if (!empty($success_message)): ?>
            <div class="bg-green-50 border border-green-200 p-4 mb-4 text-green-700 text-sm ">
                <div class="flex items-center">
                    <i class="fas fa-check-circle mr-2"></i>
                    <span><?php echo htmlspecialchars($success_message); ?></span>
                </div>
            </div>
            <div class="bg-white  text-center">
                <div class="mb-4">
                    <i class="fas fa-check-circle text-green-500 text-4xl mb-3"></i>
                    <h2 class="text-lg font-semibold text-gray-800 mb-2">Password Reset Successful!</h2>
                    <p class="text-gray-600 text-sm mb-4">Your password has been changed successfully.</p>
                </div>
                <a href="login.php" 
                   class="w-full py-2.5 bg-[#fbbb01] text-white text-sm font-medium rounded-lg hover:bg-[#e0a800] transition inline-block text-center">
                    Go to Login
                </a>
            </div>
        <?php elseif ($show_form): ?>
            <!-- Display session error messages -->
            <?php 
            if (isset($_SESSION['error_messages'])) {
                foreach ($_SESSION['error_messages'] as $error) {
                    echo '<div class="bg-red-50 border border-red-200 p-3 mb-4 text-red-700 text-sm rounded-lg">' . htmlspecialchars($error) . '</div>';
                }
                unset($_SESSION['error_messages']);
            }
            ?>
            
            <!-- Reset Password Form -->
            <div class="bg-white  p-6">
                <form id="resetForm" method="POST">
                    <input type="hidden" name="reset_password" value="1">
                    
                    <!-- New Password -->
                    <div class="mb-4">
                        <label class="block text-gray-700 text-sm font-medium mb-1">New Password</label>
                        <div class="relative">
                            <i class="fas fa-lock absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm"></i>
                            <input type="password" id="password" name="password" 
                                   class="w-full pl-9 pr-9 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                                   placeholder="••••••••"
                                   required>
                            <i class="fas fa-eye absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 cursor-pointer text-sm" 
                               onclick="togglePassword('password', this)"></i>
                        </div>
                        <small class="text-red-500 text-xs mt-1 block" id="password_error"><?php echo htmlspecialchars($errors['password']); ?></small>
                        <small class="text-gray-600 text-xs mt-1 block">
                            Must be 8+ characters with uppercase, lowercase, number, and special character
                        </small>
                    </div>
                    
                    <!-- Confirm Password -->
                    <div class="mb-5">
                        <label class="block text-gray-700 text-sm font-medium mb-1">Confirm Password</label>
                        <div class="relative">
                            <i class="fas fa-lock absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm"></i>
                            <input type="password" id="confirm_password" name="confirm_password" 
                                   class="w-full pl-9 pr-9 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                                   placeholder="••••••••"
                                   required>
                            <i class="fas fa-eye absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 cursor-pointer text-sm" 
                               onclick="togglePassword('confirm_password', this)"></i>
                        </div>
                        <small class="text-red-500 text-xs mt-1 block" id="confirm_password_error"><?php echo htmlspecialchars($errors['confirm_password']); ?></small>
                    </div>
                    
                    <!-- Submit Button -->
                    <button type="submit" id="submitBtn" name="reset_password"
                            class="w-full py-2.5 bg-[#fbbb01] text-white text-sm font-medium rounded-lg hover:bg-[#e0a800] transition disabled:opacity-70 disabled:cursor-not-allowed">
                        <span id="btnText">Update Password</span>
                        <span id="loadingSpinner" class="hidden ml-2">
                            <i class="fas fa-spinner fa-spin"></i>
                        </span>
                    </button>
                    
                    <!-- Back to Login -->
                    <div class="text-center mt-4">
                        <p class="text-gray-500 text-xs">
                            Remember your password? 
                            <a href="login.php" class="text-[#fbbb01] hover:text-[#e0a800] font-medium">Login here</a>
                        </p>
                    </div>
                </form>
            </div>
        <?php endif; ?>
    </div>

    <script>
    function togglePassword(inputId, icon) {
        const input = document.getElementById(inputId);
        if (input.type === 'password') {
            input.type = 'text';
            icon.classList.replace('fa-eye', 'fa-eye-slash');
        } else {
            input.type = 'password';
            icon.classList.replace('fa-eye-slash', 'fa-eye');
        }
    }

    function validatePassword(password) {
        if (password.length < 8) return 'Password must be at least 8 characters';
        if (!/[A-Z]/.test(password)) return 'Must contain at least one uppercase letter';
        if (!/[a-z]/.test(password)) return 'Must contain at least one lowercase letter';
        if (!/[0-9]/.test(password)) return 'Must contain at least one number';
        if (!/[^A-Za-z0-9]/.test(password)) return 'Must contain at least one special character';
        return '';
    }

    <?php if ($show_form): ?>
    document.getElementById('resetForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const password = document.getElementById('password').value;
        const confirmPassword = document.getElementById('confirm_password').value;
        const passwordError = document.getElementById('password_error');
        const confirmError = document.getElementById('confirm_password_error');
        
        // Clear errors
        passwordError.textContent = '';
        confirmError.textContent = '';
        
        // Validate
        const passwordValidation = validatePassword(password);
        if (passwordValidation) {
            passwordError.textContent = passwordValidation;
            return;
        }
        
        if (password !== confirmPassword) {
            confirmError.textContent = 'Passwords do not match';
            return;
        }
        
        const btn = document.getElementById('submitBtn');
        const btnText = document.getElementById('btnText');
        const spinner = document.getElementById('loadingSpinner');
        
        btn.disabled = true;
        btnText.textContent = 'Updating...';
        spinner.classList.remove('hidden');
        
        // Submit form
        this.submit();
    });

    // Real-time password validation
    document.getElementById('password').addEventListener('blur', function() {
        const error = validatePassword(this.value);
        document.getElementById('password_error').textContent = error || '';
    });

    document.getElementById('confirm_password').addEventListener('blur', function() {
        const password = document.getElementById('password').value;
        const confirmPassword = this.value;
        
        if (password && confirmPassword && password !== confirmPassword) {
            document.getElementById('confirm_password_error').textContent = 'Passwords do not match';
        } else {
            document.getElementById('confirm_password_error').textContent = '';
        }
    });

    document.addEventListener('DOMContentLoaded', function() {
        document.getElementById('password').focus();
    });
    <?php endif; ?>
    </script>
</body>
</html>

<?php
if (isset($_SESSION['reset_errors'])) {
    unset($_SESSION['reset_errors']);
}
?>