<?php 
ob_start();
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['user_id'])) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch student data
$student_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    $_SESSION['error'] = "Student record not found";
    header("Location: dashboard.php");
    exit();
}

// Handle photo upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_FILES['profile_picture']['name'])) {
    $target_dir = "../uploads/";
    if (!file_exists($target_dir)) {
        mkdir($target_dir, 0777, true);
    }
    
    $file_extension = pathinfo($_FILES["profile_picture"]["name"], PATHINFO_EXTENSION);
    $new_filename = "profile_" . $student_id . "_" . time() . "." . $file_extension;
    $target_file = $target_dir . $new_filename;
    
    // Check if image file is a actual image
    $check = getimagesize($_FILES["profile_picture"]["tmp_name"]);
    if ($check === false) {
        $_SESSION['error'] = "File is not an image.";
    } elseif ($_FILES["profile_picture"]["size"] > 5000000) {
        $_SESSION['error'] = "Sorry, your file is too large (max 5MB).";
    } else {
        $allowed_types = ['jpg', 'jpeg', 'png', 'gif'];
        if (!in_array(strtolower($file_extension), $allowed_types)) {
            $_SESSION['error'] = "Sorry, only JPG, JPEG, PNG & GIF files are allowed.";
        } else {
            if (move_uploaded_file($_FILES["profile_picture"]["tmp_name"], $target_file)) {
                // Delete old profile picture if it's not the default
                if (!empty($student['profile_picture']) && $student['profile_picture'] != 'assets/images/default-avatar.png') {
                    @unlink($student['profile_picture']);
                }
                
                $update_stmt = $conn->prepare("UPDATE students SET profile_picture=? WHERE id=?");
                $update_stmt->bind_param("si", $target_file, $student_id);
                
                if ($update_stmt->execute()) {
                    $_SESSION['success'] = "Profile picture updated successfully!";
                    header("Location: profile.php");
                    exit();
                } else {
                    $_SESSION['error'] = "Error updating profile picture in database.";
                }
            } else {
                $_SESSION['error'] = "Sorry, there was an error uploading your file.";
            }
        }
    }
}

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    
    // Verify current password
    if (!password_verify($current_password, $student['password'])) {
        $_SESSION['error'] = "Current password is incorrect";
    } elseif ($new_password !== $confirm_password) {
        $_SESSION['error'] = "New passwords don't match";
    } elseif (strlen($new_password) < 8) {
        $_SESSION['error'] = "Password must be at least 8 characters long";
    } else {
        // Update password
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        $update_stmt = $conn->prepare("UPDATE students SET password=?, password_changed_at=NOW() WHERE id=?");
        $update_stmt->bind_param("si", $hashed_password, $student_id);
        
        if ($update_stmt->execute()) {
            $_SESSION['success'] = "Password changed successfully!";
            header("Location: profile.php");
            exit();
        } else {
            $_SESSION['error'] = "Error updating password";
        }
    }
}
ob_end_flush();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-6 bg-gray-50 dark:bg-gray-900 overflow-y-auto" style="margin-bottom:80px">
  <!-- Success/Error Messages -->
  <?php if (isset($_SESSION['error'])): ?>
    <div class="mb-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
      <span class="block sm:inline"><?= $_SESSION['error'] ?></span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
      </span>
    </div>
    <?php unset($_SESSION['error']); ?>
  <?php endif; ?>

  <?php if (isset($_SESSION['success'])): ?>
    <div class="mb-4 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
      <span class="block sm:inline"><?= $_SESSION['success'] ?></span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
      </span>
    </div>
    <?php unset($_SESSION['success']); ?>
  <?php endif; ?>

  <div class=" dark:bg-gray-800   p-6 border border-gray-100 dark:border-gray-700">
    <div class="flex justify-between items-center mb-6">
      <h2 class="text-xl font-bold dark:text-white">Update Profile Picture</h2>
      <a href="profile.php" class="text-sm text-primary-500 hover:text-primary-600 dark:hover:text-primary-400">
        <i class="fas fa-arrow-left mr-1"></i> Back to Profile
      </a>
    </div>
    
    <form method="POST" action="" enctype="multipart/form-data" class="space-y-6">
      <div class="flex flex-col items-center space-y-6">
        <div class="relative">
          <img id="imagePreview" src="<?= !empty($student['profile_picture']) ? htmlspecialchars($student['profile_picture']) : 'assets/images/default-avatar.png' ?>" 
               alt="Current Profile Picture" 
               class="w-32 h-32 rounded-full object-cover border-4 border-primary-100 dark:border-primary-800">
        </div>
        
        <div class="w-full max-w-md">
          <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Choose New Photo</label>
          <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md dark:border-gray-600">
            <div class="space-y-1 text-center">
              <div class="flex text-sm text-gray-600 dark:text-gray-400 justify-center">
                <label for="profile_picture" class="relative cursor-pointer  rounded-md font-medium text-primary-600 hover:text-primary-500 focus-within:outline-none">
                  <span>Upload a file</span>
                  <input id="profile_picture" name="profile_picture" type="file" class="sr-only" accept="image/*" onchange="previewImage(this)">
                </label>
              </div>
              <p class="text-xs text-gray-500 dark:text-gray-400">
                PNG, JPG, GIF up to 5MB
              </p>
            </div>
          </div>
        </div>
      </div>
      
      <div class="flex justify-end space-x-3 pt-4 border-t dark:border-gray-700">
        <a href="profile.php" class="px-4 py-2 text-sm bg-gray-200 hover:bg-gray-300 rounded-lg dark:bg-gray-600 dark:hover:bg-gray-700 dark:text-white">
          Cancel
        </a>
        <button type="submit" 
                class="px-4 py-2 text-sm bg-primary-500 hover:bg-primary-600 text-white rounded-lg">
          Update Photo
        </button>
      </div>
    </form>
  </div>

  <!-- Password Change Form -->
  <div class=" dark:bg-gray-800   p-6 border border-gray-100 dark:border-gray-700 mt-6">
    <div class="flex justify-between items-center mb-6">
        <h2 class="text-xl font-bold dark:text-white">Change Password</h2>
    </div>
    
    <form method="POST" action="" class="space-y-6">
        <div class="grid grid-cols-1 gap-6">
            <div>
                <label for="current_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Current Password</label>
                <input type="password" id="current_password" name="current_password" required 
                       class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md  focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
            </div>
            
            <div>
                <label for="new_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">New Password</label>
                <input type="password" id="new_password" name="new_password" required minlength="8"
                       class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md  focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                <div id="password-requirements" class="text-xs text-gray-500 dark:text-gray-400 mt-1 hidden">
                    Password must be at least 8 characters long
                </div>
            </div>
            
            <div>
                <label for="confirm_password" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Confirm New Password</label>
                <input type="password" id="confirm_password" name="confirm_password" required minlength="8"
                       class="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                <div id="password-match" class="text-xs mt-1 hidden"></div>
            </div>
        </div>
        
        <div class="flex justify-end space-x-3 pt-4 border-t dark:border-gray-700">
            <button type="reset" class="px-4 py-2 text-sm bg-gray-200 hover:bg-gray-300 rounded-lg dark:bg-gray-600 dark:hover:bg-gray-700 dark:text-white">
                Reset
            </button>
            <button type="submit" name="change_password" 
                    class="px-4 py-2 text-sm bg-primary-500 hover:bg-primary-600 text-white rounded-lg">
                Change Password
            </button>
        </div>
    </form>
  </div>
</main>

<script>
function previewImage(input) {
  if (input.files && input.files[0]) {
    var reader = new FileReader();
    reader.onload = function(e) {
      document.getElementById('imagePreview').src = e.target.result;
    }
    reader.readAsDataURL(input.files[0]);
  }
}

// Password validation
document.addEventListener('DOMContentLoaded', function() {
    const newPassword = document.getElementById('new_password');
    const confirmPassword = document.getElementById('confirm_password');
    const requirements = document.getElementById('password-requirements');
    const matchIndicator = document.getElementById('password-match');
    
    function validatePassword() {
        if (newPassword.value !== confirmPassword.value) {
            confirmPassword.setCustomValidity("Passwords don't match");
            matchIndicator.textContent = "Passwords don't match";
            matchIndicator.className = "text-xs text-red-500 mt-1";
            matchIndicator.classList.remove('hidden');
        } else {
            confirmPassword.setCustomValidity('');
            matchIndicator.textContent = "Passwords match!";
            matchIndicator.className = "text-xs text-green-500 mt-1";
            matchIndicator.classList.remove('hidden');
        }
    }
    
    newPassword.addEventListener('focus', function() {
        requirements.classList.remove('hidden');
    });
    
    newPassword.addEventListener('blur', function() {
        if (this.value === '') {
            requirements.classList.add('hidden');
        }
    });
    
    newPassword.addEventListener('change', validatePassword);
    confirmPassword.addEventListener('keyup', validatePassword);
});
</script>

<?php include 'include/footer.php'; ?>