<?php
require_once '../config/conn.php';
session_start();

if (!isset($_SESSION['user_id']) || !isset($_GET['id'])) {
    die('<div class="p-4 text-red-500">Unauthorized</div>');
}

$transaction_id = intval($_GET['id']);
$student_id = $_SESSION['user_id'];

// Verify the transaction belongs to the student
$stmt = $conn->prepare("
    SELECT t.*, b.bank_name, b.account_number, b.account_name 
    FROM wallet_transactions t
    LEFT JOIN student_bank_accounts b ON t.bank_account_id = b.id
    JOIN student_wallets w ON t.wallet_id = w.id
    WHERE t.id = ? AND w.student_id = ?
");
$stmt->bind_param("ii", $transaction_id, $student_id);
$stmt->execute();
$result = $stmt->get_result();
$transaction = $result->fetch_assoc();

if (!$transaction) {
    die('<div class="p-4 text-red-500">Transaction not found</div>');
}

$isCredit = in_array($transaction['transaction_type'], ['deposit', 'refund', 'credit', 'transfer_in']);
$amountFormatted = number_format($transaction['amount'], 2);
?>

<div id="transaction-details-page" class="fixed inset-0 z-50 overflow-y-auto bg-white">
    <!-- Header -->
    <div class="sticky top-0 bg-white border-b border-gray-200 z-10">
        <div class="flex items-center justify-between p-4">
            <button id="back-to-transactions" class="flex items-center text-gray-600 hover:text-gray-800">
                <i class="fas fa-arrow-left mr-2"></i>
                <span class="text-sm font-medium">Back</span>
            </button>
            <div class="text-center">
                <h1 class="text-sm font-semibold text-gray-800">Transaction Details</h1>
                <p class="text-xs text-gray-500">ID: <?= $transaction['id'] ?></p>
            </div>
            <div class="w-6"></div> <!-- Spacer for centering -->
        </div>
    </div>

    <!-- Main Content -->
    <div class="p-4">
        <!-- Amount Card -->
        <div class="bg-gradient-to-r <?= $isCredit ? 'from-green-50 to-emerald-50' : 'from-red-50 to-rose-50' ?> rounded-xl p-5 mb-4 text-center">
            <p class="text-sm text-gray-600 mb-1"><?= $isCredit ? 'Credit' : 'Debit' ?> Amount</p>
            <p class="text-3xl font-bold <?= $isCredit ? 'text-green-600' : 'text-red-600' ?>">
                <?= $isCredit ? '+' : '-' ?>₦<?= $amountFormatted ?>
            </p>
            <div class="inline-flex items-center mt-2 px-3 py-1 rounded-full text-xs font-medium <?= 
                $transaction['status'] === 'completed' ? 'bg-green-100 text-green-800' : 
                ($transaction['status'] === 'pending' ? 'bg-yellow-100 text-yellow-800' : 
                'bg-red-100 text-red-800') ?>">
                <i class="fas <?= 
                    $transaction['status'] === 'completed' ? 'fa-check-circle' : 
                    ($transaction['status'] === 'pending' ? 'fa-clock' : 
                    'fa-times-circle') ?> mr-1"></i>
                <?= ucfirst($transaction['status']) ?>
            </div>
        </div>

        <!-- Details Card -->
        <div class="bg-white rounded-lg border border-gray-200 divide-y divide-gray-100">
            <!-- Basic Info -->
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-info-circle text-primary-500 mr-2"></i>
                    Transaction Information
                </h3>
                <div class="space-y-3">
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Type</span>
                        <span class="text-sm font-medium capitalize"><?= $transaction['transaction_type'] ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Date</span>
                        <span class="text-sm"><?= date('M j, Y', strtotime($transaction['created_at'])) ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Time</span>
                        <span class="text-sm"><?= date('g:i A', strtotime($transaction['created_at'])) ?></span>
                    </div>
                </div>
            </div>

            <!-- Description -->
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-file-alt text-primary-500 mr-2"></i>
                    Description
                </h3>
                <p class="text-sm text-gray-700 bg-gray-50 p-3 rounded-lg"><?= htmlspecialchars($transaction['description']) ?></p>
            </div>

            <!-- Reference -->
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-hashtag text-primary-500 mr-2"></i>
                    Reference
                </h3>
                <div class="flex items-center justify-between bg-gray-50 p-3 rounded-lg">
                    <code class="text-sm font-mono text-gray-700 truncate" id="reference-text">
                        <?= htmlspecialchars($transaction['reference']) ?>
                    </code>
                    <button onclick="copyReference()" 
                            class="text-primary-600 hover:text-primary-700 ml-2 flex-shrink-0">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
            </div>

            <!-- Payment Method -->
            <?php if ($transaction['gateway'] || $transaction['payment_gateway']): ?>
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-credit-card text-primary-500 mr-2"></i>
                    Payment Method
                </h3>
                <div class="flex items-center">
                    <div class="w-8 h-8 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center mr-3">
                        <i class="fas fa-university"></i>
                    </div>
                    <span class="text-sm"><?= htmlspecialchars($transaction['gateway'] ?: $transaction['payment_gateway']) ?></span>
                </div>
            </div>
            <?php endif; ?>

            <!-- Bank Details -->
            <?php if ($transaction['bank_name']): ?>
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-university text-primary-500 mr-2"></i>
                    Bank Details
                </h3>
                <div class="space-y-3 bg-gray-50 p-3 rounded-lg">
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Bank</span>
                        <span class="text-sm font-medium"><?= htmlspecialchars($transaction['bank_name']) ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Account Name</span>
                        <span class="text-sm"><?= htmlspecialchars($transaction['account_name']) ?></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-sm text-gray-600">Account Number</span>
                        <span class="text-sm font-mono">****<?= substr($transaction['account_number'], -4) ?></span>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Additional Info -->
            <?php if ($transaction['metadata']): 
                $metadata = json_decode($transaction['metadata'], true);
                if ($metadata && is_array($metadata)):
            ?>
            <div class="p-4">
                <h3 class="text-sm font-semibold text-gray-800 mb-3 flex items-center">
                    <i class="fas fa-list-ul text-primary-500 mr-2"></i>
                    Additional Information
                </h3>
                <div class="bg-gray-50 p-3 rounded-lg space-y-2">
                    <?php foreach ($metadata as $key => $value): 
                        if (!empty($value)):
                    ?>
                    <div class="flex justify-between items-start">
                        <span class="text-xs text-gray-600 capitalize"><?= htmlspecialchars(str_replace('_', ' ', $key)) ?>:</span>
                        <span class="text-xs text-gray-700 text-right ml-2"><?= htmlspecialchars(is_array($value) ? json_encode($value) : $value) ?></span>
                    </div>
                    <?php endif; endforeach; ?>
                </div>
            </div>
            <?php endif; endif; ?>
        </div>

        <!-- Transaction ID Footer -->
        <div class="mt-4 text-center">
            <p class="text-xs text-gray-400">Transaction ID: <?= $transaction['id'] ?></p>
            <p class="text-xs text-gray-400 mt-1"><?= date('F j, Y \a\t g:i A', strtotime($transaction['created_at'])) ?></p>
        </div>

        <!-- Single Print Button -->
        <div class="mt-6">
            <button onclick="printTransaction()" 
                    class="w-full flex items-center justify-center py-3 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 active:bg-gray-100 transition-colors">
                <i class="fas fa-print mr-2"></i>
                <span class="text-sm font-medium">Print Receipt</span>
            </button>
        </div>
    </div>
</div>

<style>
    #transaction-details-page {
        -webkit-overflow-scrolling: touch;
    }
    
    @media print {
        #transaction-details-page {
            position: static !important;
            overflow: visible !important;
        }
        
        button {
            display: none !important;
        }
        
        body * {
            visibility: hidden;
        }
        
        #transaction-details-page, #transaction-details-page * {
            visibility: visible;
        }
        
        #transaction-details-page {
            position: absolute;
            left: 0;
            top: 0;
            width: 100%;
        }
    }
</style>

<script>
    // Copy reference to clipboard
    function copyReference() {
        const reference = document.getElementById('reference-text').textContent;
        copyToClipboard(reference, 'Reference copied to clipboard!');
    }
    
    // Generic copy function
    function copyToClipboard(text, successMessage = 'Copied to clipboard!') {
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText(text).then(() => {
                showToast(successMessage, 'success');
            }).catch(err => {
                console.error('Failed to copy: ', err);
                fallbackCopy(text, successMessage);
            });
        } else {
            fallbackCopy(text, successMessage);
        }
    }
    
    // Fallback copy for older browsers
    function fallbackCopy(text, successMessage) {
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.opacity = '0';
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        
        try {
            const successful = document.execCommand('copy');
            if (successful) {
                showToast(successMessage, 'success');
            } else {
                showToast('Failed to copy', 'error');
            }
        } catch (err) {
            console.error('Fallback copy failed: ', err);
            showToast('Failed to copy', 'error');
        }
        
        document.body.removeChild(textArea);
    }
    
    // Print transaction
    function printTransaction() {
        window.print();
    }
    
    // Show toast notification
    function showToast(message, type = 'info') {
        // Remove existing toast
        const existingToast = document.querySelector('.toast-notification');
        if (existingToast) {
            existingToast.remove();
        }
        
        // Create toast
        const toast = document.createElement('div');
        toast.className = `toast-notification fixed top-4 left-1/2 transform -translate-x-1/2 z-50 px-4 py-2 rounded-lg shadow-lg text-white text-sm font-medium ${
            type === 'success' ? 'bg-green-500' :
            type === 'error' ? 'bg-red-500' :
            'bg-blue-500'
        }`;
        toast.textContent = message;
        
        document.body.appendChild(toast);
        
        // Auto remove after 3 seconds
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
    
    // Close on escape key
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            closeDetailsPage();
        }
    });
    
    // Close on overlay click
    document.getElementById('transaction-details-page').addEventListener('click', (e) => {
        if (e.target.id === 'transaction-details-page') {
            closeDetailsPage();
        }
    });
    
    // Back button
    document.getElementById('back-to-transactions').addEventListener('click', (e) => {
        e.preventDefault();
        closeDetailsPage();
    });
    
    function closeDetailsPage() {
        const detailsPage = document.getElementById('transaction-details-page');
        if (detailsPage) {
            detailsPage.remove();
            document.querySelector('main').style.opacity = '1';
            document.querySelector('main').style.pointerEvents = 'auto';
        }
    }
    
    // Initialize when page loads
    document.addEventListener('DOMContentLoaded', () => {
        // Add click handler for back button
        const backBtn = document.getElementById('back-to-transactions');
        if (backBtn) {
            backBtn.addEventListener('click', (e) => {
                e.preventDefault();
                closeDetailsPage();
            });
        }
    });
</script>