<?php 
include 'include/header.php';

// Verify session data
if (!isset($_SESSION['user_id'])) {
    session_destroy();
    header("Location: login.php");
    exit();
}

require_once '../config/conn.php';

// Search and filter
$search = $_GET['search'] ?? '';
$business_type = $_GET['business_type'] ?? '';

// Build query
$query = "SELECT id, vendor_id, business_name, business_type, address, profile_picture 
          FROM vendors 
          WHERE is_verified = 1 AND is_approved = 1";

$params = [];
$types = '';

if (!empty($search)) {
    $query .= " AND (business_name LIKE ? OR business_type LIKE ? OR address LIKE ?)";
    $search_term = "%$search%";
    $params = array_merge($params, [$search_term, $search_term, $search_term]);
    $types .= 'sss';
}

if (!empty($business_type)) {
    $query .= " AND business_type = ?";
    $params[] = $business_type;
    $types .= 's';
}

$query .= " ORDER BY business_name ASC";

$stmt = $conn->prepare($query);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$vendors = $stmt->get_result();

// Get unique business types
$type_stmt = $conn->query("SELECT DISTINCT business_type FROM vendors WHERE business_type IS NOT NULL AND business_type != '' ORDER BY business_type");
$business_types = $type_stmt->fetch_all(MYSQLI_ASSOC);
?>

<div class="max-w-6xl mx-auto p-4 mb-6">
    <!-- Header -->
    <div class="mb-6">
        <div class="flex items-center mb-3">
            <i class="fas fa-store text-[#fbbb01] text-xl mr-2"></i>
            <h1 class="text-xl font-bold text-gray-800">Browse Vendors</h1>
        </div>
        <p class="text-gray-500 text-sm">Find and purchase from trusted campus vendors</p>
    </div>

    <!-- Search and Filter -->
    <div class=" p-4 mb-6">
        <form id="filterForm" method="GET" class="space-y-3">
            <!-- Search -->
            <div>
                <label class="block text-gray-700 text-sm font-medium mb-1">Search Vendors</label>
                <div class="relative">
                    <i class="fas fa-search absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm"></i>
                    <input type="text" 
                           name="search" 
                           value="<?= htmlspecialchars($search) ?>" 
                           placeholder="Search vendors..."
                           class="w-full pl-9 pr-3 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]">
                </div>
            </div>
            
            <!-- Business Type Filter -->
            <div>
                <label class="block text-gray-700 text-sm font-medium mb-1">Business Type</label>
                <div class="relative">
                    <i class="fas fa-tag absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm z-10"></i>
                    <select name="business_type" 
                            class="w-full pl-9 pr-9 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01] appearance-none bg-white">
                        <option value="">All Types</option>
                        <?php foreach($business_types as $type): ?>
                            <option value="<?= htmlspecialchars($type['business_type']) ?>" 
                                    <?= $business_type === $type['business_type'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($type['business_type']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <i class="fas fa-chevron-down absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 pointer-events-none"></i>
                </div>
            </div>
            
            <!-- Buttons -->
            <div class="flex space-x-2 pt-2">
                <button type="submit" 
                        class="flex-1 bg-[#fbbb01] text-white text-sm font-medium py-2 rounded-lg hover:bg-[#e0a800] transition">
                    Search
                </button>
                <a href="vendors.php" 
                   class="flex-1 bg-gray-100 text-gray-700 text-sm font-medium py-2 rounded-lg hover:bg-gray-200 transition text-center">
                    Reset
                </a>
            </div>
        </form>
    </div>

    <!-- Results Count -->
    <div class="mb-4">
        <p class="text-gray-500 text-sm">
            <?php 
            $count = $vendors->num_rows;
            echo $count . ' vendor' . ($count !== 1 ? 's' : '') . ' found';
            ?>
        </p>
    </div>

    <!-- Vendors Grid -->
    <div id="vendorsContainer">
        <?php if ($vendors->num_rows > 0): ?>
            <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 gap-3">
                <?php while($vendor = $vendors->fetch_assoc()): 
                    // Truncate business name for mobile
                    $display_name = strlen($vendor['business_name']) > 14 
                        ? substr($vendor['business_name'], 0, 14) . '...' 
                        : $vendor['business_name'];
                ?>
                <div class=" overflow-hidden transition">
                    <!-- Vendor Image -->
                    <div class="h-32 bg-gray-100 relative">
                        <?php if (!empty($vendor['profile_picture'])): 
                            $image_path = strpos($vendor['profile_picture'], 'http') === 0 
                                ? $vendor['profile_picture'] 
                                : '../uploads/' . $vendor['profile_picture'];
                        ?>
                            <img src="<?= htmlspecialchars($image_path) ?>" 
                                 alt="<?= htmlspecialchars($vendor['business_name']) ?>" 
                                 class="w-full h-full object-cover"
                                 onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect width=%22100%22 height=%22100%22 fill=%22%23f3f4f6%22/><text x=%2250%22 y=%2255%22 font-family=%22Arial%22 font-size=%2214%22 text-anchor=%22middle%22 fill=%22%239ca3af%22>Store</text></svg>'">
                        <?php else: ?>
                            <div class="w-full h-full flex items-center justify-center bg-gray-100">
                                <i class="fas fa-store text-3xl text-gray-300"></i>
                            </div>
                        <?php endif; ?>
                        <div class="absolute top-2 right-2">
                            <span class="bg-[#fbbb01] text-white text-xs font-medium px-2 py-1 rounded-full">
                                <?= !empty($vendor['business_type']) ? substr($vendor['business_type'], 0, 8) : 'Vendor' ?>
                            </span>
                        </div>
                    </div>
                    
                    <!-- Vendor Info -->
                    <div class="p-3">
                        <h3 class="text-sm font-medium text-gray-800 mb-3 truncate" title="<?= htmlspecialchars($vendor['business_name']) ?>">
                            <?= htmlspecialchars($display_name) ?>
                        </h3>
                        
                        <!-- Action Buttons -->
                        <div class="space-y-2">
                            <a href="vendor_details.php?id=<?= $vendor['id'] ?>" 
                               class="block bg-gray-100 hover:bg-gray-200 text-gray-700 text-xs font-medium py-1.5 rounded text-center transition">
                                <i class="fas fa-info-circle mr-1"></i> Details
                            </a>
                            <a href="vendor-products.php?vendor_id=<?= $vendor['id'] ?>" 
                               class="block bg-[#fbbb01] hover:bg-[#e0a800] text-white text-xs font-medium py-1.5 rounded text-center transition">
                                <i class="fas fa-shopping-cart mr-1"></i> Buy Now
                            </a>
                        </div>
                    </div>
                </div>
                <?php endwhile; ?>
            </div>
        <?php else: ?>
            <!-- No Results -->
            <div class=" p-8 text-center">
                <i class="fas fa-store-slash text-4xl text-gray-300 mb-3"></i>
                <h3 class="text-lg font-medium text-gray-800 mb-2">No Vendors Found</h3>
                <p class="text-gray-500 text-sm mb-4">Try adjusting your search or filters</p>
                <a href="vendors.php" class="inline-block bg-[#fbbb01] text-white text-sm font-medium px-4 py-2 rounded-lg hover:bg-[#e0a800] transition">
                    Reset Filters
                </a>
            </div>
        <?php endif; ?>
    </div>

    <!-- Loading Indicator (hidden by default) -->
    <div id="loadingIndicator" class="hidden text-center py-8">
        <div class="inline-block w-8 h-8 border-2 border-[#fbbb01] border-t-transparent rounded-full animate-spin mb-3"></div>
        <p class="text-gray-500 text-sm">Loading vendors...</p>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const filterForm = document.getElementById('filterForm');
    const vendorsContainer = document.getElementById('vendorsContainer');
    const loadingIndicator = document.getElementById('loadingIndicator');
    
    // Function to update URL without page reload
    function updateURL(params) {
        const url = new URL(window.location);
        url.search = params.toString();
        window.history.pushState({}, '', url);
    }
    
    // Function to update vendors via AJAX
    function updateVendors() {
        const formData = new FormData(filterForm);
        const params = new URLSearchParams(formData);
        
        // Show loading, hide current content
        loadingIndicator.classList.remove('hidden');
        vendorsContainer.classList.add('opacity-50');
        
        // Update URL
        updateURL(params);
        
        // AJAX request
        fetch('vendors.php?' + params.toString(), {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => response.text())
        .then(html => {
            // Create temp element to parse response
            const temp = document.createElement('div');
            temp.innerHTML = html;
            
            // Extract new content
            const newContainer = temp.querySelector('#vendorsContainer');
            const newResults = temp.querySelector('.mb-4 p');
            
            // Update content
            if (newContainer) {
                vendorsContainer.innerHTML = newContainer.innerHTML;
                vendorsContainer.classList.remove('opacity-50');
            }
            
            if (newResults) {
                document.querySelector('.mb-4 p').innerHTML = newResults.innerHTML;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            vendorsContainer.innerHTML = `
                <div class="bg-white rounded-lg shadow p-8 text-center">
                    <i class="fas fa-exclamation-triangle text-4xl text-red-400 mb-3"></i>
                    <h3 class="text-lg font-medium text-gray-800 mb-2">Error Loading</h3>
                    <p class="text-gray-500 text-sm">Please try again</p>
                </div>
            `;
            vendorsContainer.classList.remove('opacity-50');
        })
        .finally(() => {
            loadingIndicator.classList.add('hidden');
        });
    }
    
    // Form submission
    filterForm.addEventListener('submit', function(e) {
        e.preventDefault();
        updateVendors();
    });
    
    // Real-time search with debounce
    let searchTimer;
    const searchInput = document.querySelector('input[name="search"]');
    searchInput.addEventListener('input', function() {
        clearTimeout(searchTimer);
        searchTimer = setTimeout(() => {
            filterForm.dispatchEvent(new Event('submit'));
        }, 500);
    });
    
    // Auto-submit when filter changes
    document.querySelector('select[name="business_type"]').addEventListener('change', function() {
        filterForm.dispatchEvent(new Event('submit'));
    });
    
    // Handle browser back/forward
    window.addEventListener('popstate', function() {
        // Get current URL params
        const urlParams = new URLSearchParams(window.location.search);
        
        // Update form fields
        searchInput.value = urlParams.get('search') || '';
        document.querySelector('select[name="business_type"]').value = urlParams.get('business_type') || '';
        
        // Update vendors
        updateVendors();
    });
});
</script>

<?php include 'include/footer.php'; ?>