<?php
// verify.php - Debug version
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

echo "<pre>";
echo "Starting verification process...\n";

// Start session
session_start();
echo "Session started\n";

// Check for verification parameters
if (isset($_GET['code']) && isset($_GET['email'])) {
    echo "Code and email parameters found\n";
    echo "Code: " . htmlspecialchars($_GET['code']) . "\n";
    echo "Email: " . htmlspecialchars($_GET['email']) . "\n";
    
    // Try to connect to database
    try {
        echo "Attempting database connection...\n";
        
        // Check if config file exists
        if (!file_exists('../config/conn.php')) {
            throw new Exception("Database config file not found!");
        }
        
        require '../config/conn.php';
        
        if (!isset($conn)) {
            throw new Exception("Database connection not established!");
        }
        
        echo "Database connected successfully\n";
        
        // Clean inputs
        $verification_code = $conn->real_escape_string($_GET['code']);
        $email = $conn->real_escape_string($_GET['email']);
        
        echo "Checking database for verification code...\n";
        
        // Check if vendor exists with this verification code
        $query = "SELECT id, vendor_id, business_name, contact_person, verification_code_expiry, email_verified_at 
                  FROM vendors 
                  WHERE verification_code = ? AND email = ?";
        
        $stmt = $conn->prepare($query);
        
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $conn->error);
        }
        
        $stmt->bind_param("ss", $verification_code, $email);
        
        if (!$stmt->execute()) {
            throw new Exception("Execute failed: " . $stmt->error);
        }
        
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            echo "No matching verification code found\n";
            $_SESSION['verification_error'] = "Invalid verification link. Please check your email for the correct link.";
        } else {
            $vendor = $result->fetch_assoc();
            echo "Vendor found: " . $vendor['business_name'] . "\n";
            
            // Check if already verified
            if ($vendor['email_verified_at'] !== null) {
                echo "Email already verified\n";
                $_SESSION['verification_error'] = "This email has already been verified. You can now login.";
            } else {
                // Check if verification code has expired
                $current_time = time();
                $expiry_time = strtotime($vendor['verification_code_expiry']);
                
                echo "Current time: " . date('Y-m-d H:i:s', $current_time) . "\n";
                echo "Expiry time: " . $vendor['verification_code_expiry'] . "\n";
                
                if ($current_time > $expiry_time) {
                    echo "Verification code expired\n";
                    $_SESSION['verification_error'] = "Verification link has expired. Please request a new verification email.";
                } else {
                    // Verify the email
                    echo "Verifying email...\n";
                    $verify_time = date('Y-m-d H:i:s');
                    
                    $update_query = "UPDATE vendors 
                                    SET email_verified_at = ?, 
                                        verification_code = NULL, 
                                        verification_code_expiry = NULL 
                                    WHERE id = ?";
                    
                    $update_stmt = $conn->prepare($update_query);
                    
                    if (!$update_stmt) {
                        throw new Exception("Update prepare failed: " . $conn->error);
                    }
                    
                    $update_stmt->bind_param("si", $verify_time, $vendor['id']);
                    
                    if ($update_stmt->execute()) {
                        echo "Email verified successfully!\n";
                        $_SESSION['verification_success'] = "Email verified successfully! Your account is now under review by our procurement team.";
                        $_SESSION['vendor_id'] = $vendor['vendor_id'];
                        $_SESSION['vendor_name'] = $vendor['business_name'];
                    } else {
                        throw new Exception("Update failed: " . $update_stmt->error);
                    }
                    
                    $update_stmt->close();
                }
            }
        }
        
        $stmt->close();
        $conn->close();
        
    } catch (Exception $e) {
        echo "ERROR: " . $e->getMessage() . "\n";
        $_SESSION['verification_error'] = "An error occurred during verification. Please try again later.";
    }
} else {
    echo "Missing verification parameters\n";
    $_SESSION['verification_error'] = "Invalid verification link. Missing parameters.";
}

echo "Redirecting to login page...\n";
echo "</pre>";

// Redirect to login page
header("Location: vendor-login.php");
exit();
?>