<?php
ob_start();
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

require '../config/conn.php';
require '../vendor/autoload.php';

if (!isset($_SESSION['2fa_user_id'])) {
    header("Location: login.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Verify Code</title>

<script src="https://cdn.tailwindcss.com"></script>
<link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;500;600&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">

<style>
    body { font-family: 'Poppins', sans-serif; }
    input[type=number]::-webkit-inner-spin-button,
    input[type=number]::-webkit-outer-spin-button { 
        -webkit-appearance: none; 
        margin: 0; 
    }
    input[type="number"] {
        -moz-appearance: textfield;
    }
</style>
</head>

<body class=" bg-gray-50 flex items-center justify-center px-4">

<div class="w-full max-w-sm bg-white p-6">

    <!-- Header -->
    <div class="text-center mb-6">
        <div class="w-14 h-14 mx-auto bg-yellow-100 rounded-full flex items-center justify-center mb-3">
            <i class="fas fa-shield-alt text-yellow-500 text-xl"></i>
        </div>
        <h2 class="text-xl font-semibold text-gray-800">Verify Code</h2>
        <p class="text-sm text-gray-500 mt-1">
            Enter the 6-digit code sent to your email
        </p>
    </div>

    <!-- Message -->
    <div id="messageContainer" class="mb-4"></div>

    <!-- Form -->
    <form id="verifyForm">

        <!-- OTP Inputs -->
        <div class="flex justify-between gap-2 mb-6">
            <?php for ($i = 0; $i < 6; $i++): ?>
                <input type="number"
                       maxlength="1"
                       class="otp w-11 h-12 text-center text-lg font-semibold border border-gray-300 rounded-lg focus:border-yellow-500 focus:ring-1 focus:ring-yellow-400 outline-none appearance-none"
                       inputmode="numeric"
                       oninput="this.value=this.value.slice(0,1)"
                       autocomplete="off">
            <?php endfor; ?>
        </div>

        <input type="hidden" name="code" id="code">

        <!-- Verify Button -->
        <button type="submit"
                id="verifyBtn"
                class="w-full py-3 bg-yellow-500 text-white rounded-lg font-medium hover:bg-yellow-600 transition disabled:opacity-50"
                disabled>
            Verify
        </button>

        <!-- Resend -->
        <div class="text-center mt-4 text-sm">
            <span class="text-gray-500">Didn't get it?</span>
            <button type="button" 
                    id="resendBtn" 
                    class="text-yellow-500 font-medium ml-1 hover:text-yellow-600 disabled:text-gray-400"
                    disabled>
                Resend
            </button>
            <span id="countdown" class="block text-xs text-gray-400 mt-1"></span>
        </div>

    </form>
</div>

<script>
const inputs = document.querySelectorAll('.otp');
const codeInput = document.getElementById('code');
const form = document.getElementById('verifyForm');
const verifyBtn = document.getElementById('verifyBtn');
const msgBox = document.getElementById('messageContainer');
const resendBtn = document.getElementById('resendBtn');
const countdownEl = document.getElementById('countdown');

let resendTimer = 30; // 30 seconds cooldown
let timerInterval;

// Initialize timer on page load
function startResendTimer() {
    resendBtn.disabled = true;
    updateCountdown();
    
    timerInterval = setInterval(() => {
        resendTimer--;
        updateCountdown();
        
        if (resendTimer <= 0) {
            clearInterval(timerInterval);
            resendBtn.disabled = false;
            countdownEl.textContent = '';
        }
    }, 1000);
}

function updateCountdown() {
    if (resendTimer > 0) {
        countdownEl.textContent = `Resend available in ${resendTimer}s`;
    }
}

// OTP input handling
inputs.forEach((input, index) => {
    input.addEventListener('input', (e) => {
        // Only allow single digit
        if (e.target.value.length > 1) {
            e.target.value = e.target.value.slice(0, 1);
        }
        
        // Auto-focus next input
        if (e.target.value && index < inputs.length - 1) {
            inputs[index + 1].focus();
        }
        
        updateCode();
    });
    
    input.addEventListener('keydown', (e) => {
        // Handle backspace
        if (e.key === 'Backspace' && !input.value && index > 0) {
            inputs[index - 1].focus();
        }
        
        // Handle paste
        if (e.key === 'v' && (e.ctrlKey || e.metaKey)) {
            setTimeout(() => {
                handlePaste(index);
            }, 10);
        }
    });
    
    input.addEventListener('paste', (e) => {
        e.preventDefault();
        handlePaste(index);
    });
});

function handlePaste(startIndex) {
    navigator.clipboard.readText().then(text => {
        const numbers = text.replace(/\D/g, '');
        for (let i = 0; i < Math.min(6 - startIndex, numbers.length); i++) {
            if (startIndex + i < inputs.length) {
                inputs[startIndex + i].value = numbers[i];
            }
        }
        updateCode();
    });
}

function updateCode() {
    const code = Array.from(inputs).map(input => input.value).join('');
    codeInput.value = code;
    
    // Enable/disable verify button based on code length
    verifyBtn.disabled = code.length !== 6;
}

// Form submission
form.addEventListener('submit', async (e) => {
    e.preventDefault();
    
    if (codeInput.value.length !== 6) {
        showMessage('Please enter all 6 digits', 'error');
        return;
    }
    
    verifyBtn.disabled = true;
    verifyBtn.textContent = 'Verifying...';
    
    try {
        const formData = new FormData();
        formData.append('code', codeInput.value);
        formData.append('verify', '1');

        const response = await fetch('process_verify_code.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        showMessage(data.message, data.success ? 'success' : 'error');
        
        if (data.success && data.redirect) {
            setTimeout(() => {
                window.location.href = data.redirect;
            }, 1500);
        }
    } catch (error) {
        showMessage('Network error. Please try again.', 'error');
    } finally {
        verifyBtn.disabled = false;
        verifyBtn.textContent = 'Verify';
    }
});

// Resend button
resendBtn.addEventListener('click', async () => {
    if (resendTimer > 0) return;
    
    resendBtn.disabled = true;
    resendTimer = 30;
    startResendTimer();
    
    try {
        const formData = new FormData();
        formData.append('resend', '1');

        const response = await fetch('process_verify_code.php', {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            showMessage('New verification code sent to your email', 'success');
            // Clear OTP inputs
            inputs.forEach(input => input.value = '');
            updateCode();
            inputs[0].focus();
        } else {
            showMessage(data.message || 'Failed to resend code', 'error');
        }
    } catch (error) {
        showMessage('Network error. Please try again.', 'error');
    }
});

function showMessage(text, type) {
    msgBox.innerHTML = `
        <div class="p-3 rounded-lg text-sm ${
            type === 'success' 
            ? 'bg-green-100 text-green-700 border border-green-200' 
            : 'bg-red-100 text-red-700 border border-red-200'
        }">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
            ${text}
        </div>`;
    
    // Auto-hide message after 5 seconds
    setTimeout(() => {
        msgBox.innerHTML = '';
    }, 5000);
}

// Initialize
window.addEventListener('DOMContentLoaded', () => {
    startResendTimer();
    inputs[0].focus();
});
</script>

</body>
</html>

<?php ob_end_flush(); ?>