<?php
require_once '../config/conn.php';
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Authenticate request
session_start();
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

$input = json_decode(file_get_contents('php://input'), true);
$reference = $input['reference'] ?? '';
$gateway = $input['gateway'] ?? '';

if (empty($reference) || empty($gateway)) {
    echo json_encode(['success' => false, 'message' => 'Missing reference or gateway']);
    exit();
}

$student_id = $_SESSION['user_id'];

try {
    if ($gateway === 'paystack') {
        // Initialize cURL for Paystack verification
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://api.paystack.co/transaction/verify/" . rawurlencode($reference),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "accept: application/json",
                "authorization: Bearer sk_test_e961ecf6ab503d4b5d63853a2dfc0c9ec6ac78e6", // Replace with your actual secret key
                "cache-control: no-cache"
            ],
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        if ($err) {
            throw new Exception('cURL Error: ' . $err);
        }

        $result = json_decode($response, true);

        if (!$result['status'] || $http_code !== 200) {
            throw new Exception('Paystack API Error: ' . ($result['message'] ?? 'Unknown error'));
        }

        $amount = $result['data']['amount'] / 100; // Convert to Naira
        $status = $result['data']['status'];
        $paid_at = $result['data']['paid_at'];
        $customer_email = $result['data']['customer']['email'];
        
        // Verify the email matches the student's email
        $email_stmt = $conn->prepare("SELECT email FROM students WHERE id = ?");
        $email_stmt->bind_param("i", $student_id);
        $email_stmt->execute();
        $email_result = $email_stmt->get_result();
        $student_email = $email_result->fetch_assoc()['email'];
        
        if ($customer_email !== $student_email) {
            throw new Exception('Email verification failed');
        }
    } else {
        throw new Exception('Unsupported payment gateway');
    }

    if ($status !== 'success') {
        throw new Exception('Payment not successful. Status: ' . $status);
    }

    // Check if transaction already exists
    $check_txn_stmt = $conn->prepare("SELECT id FROM wallet_transactions WHERE reference = ?");
    $check_txn_stmt->bind_param("s", $reference);
    $check_txn_stmt->execute();
    $check_txn_result = $check_txn_stmt->get_result();
    
    if ($check_txn_result->num_rows > 0) {
        throw new Exception('Transaction already processed');
    }

    $conn->begin_transaction();
    
    // Get wallet ID
    $wallet_stmt = $conn->prepare("SELECT id, balance FROM student_wallets WHERE student_id = ?");
    $wallet_stmt->bind_param("i", $student_id);
    $wallet_stmt->execute();
    $wallet_result = $wallet_stmt->get_result();
    $wallet = $wallet_result->fetch_assoc();
    
    if (!$wallet) {
        throw new Exception("Wallet not found");
    }
    
    // Update wallet balance - using POSITIVE amount for deposit
    $update_stmt = $conn->prepare("UPDATE student_wallets SET balance = balance + ?, updated_at = NOW() WHERE student_id = ?");
    $update_stmt->bind_param("di", $amount, $student_id);
    $update_stmt->execute();
    
    if ($update_stmt->affected_rows === 0) {
        throw new Exception("Failed to update wallet balance");
    }
    
    // Create transaction record - using 'credit' type for deposit
   // In verify_payment.php - ensure this uses 'deposit' not 'credit'
$txn_stmt = $conn->prepare("INSERT INTO wallet_transactions 
    (student_id, wallet_id, amount, transaction_type, reference, status, description, gateway, created_at) 
    VALUES (?, ?, ?, 'deposit', ?, 'completed', 'Wallet deposit via Paystack', ?, ?)");
    $txn_stmt->bind_param("iidsss", $student_id, $wallet['id'], $amount, $reference, $gateway, $paid_at);
    $txn_stmt->execute();
    
    $conn->commit();
    
    // Get updated balance
    $new_balance_stmt = $conn->prepare("SELECT balance FROM student_wallets WHERE student_id = ?");
    $new_balance_stmt->bind_param("i", $student_id);
    $new_balance_stmt->execute();
    $new_balance_result = $new_balance_stmt->get_result();
    $new_balance = $new_balance_result->fetch_assoc()['balance'];
    
    echo json_encode([
        'success' => true, 
        'message' => 'Payment verified and wallet credited successfully',
        'amount' => $amount,
        'new_balance' => $new_balance,
        'reference' => $reference
    ]);
    
} catch (Exception $e) {
    if (isset($conn) && $conn) {
        $conn->rollback();
    }
    error_log("Payment verification error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>