<?php 

require_once '../config/conn.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}


$student_id = $_SESSION['user_id'];
$wallet_stmt = $conn->prepare("
    SELECT w.* FROM student_wallets w
    JOIN students s ON w.student_id = s.id
    WHERE s.id = ?
");
$wallet_stmt->bind_param("i", $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();
$wallet = $wallet_result->fetch_assoc();

if (!$wallet) {
    header("Location: wallet.php");
    exit();
}

// Get current date for default filter values
$today = date('Y-m-d');
$lastMonth = date('Y-m-d', strtotime('-30 days'));
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction History</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .fade-in {
            animation: fadeIn 0.3s ease-out;
        }
        .transaction-item:hover {
            background-color: #f9fafb;
        }
        @media (max-width: 640px) {
            .mobile-text-sm {
                font-size: 0.875rem;
            }
            .mobile-text-xs {
                font-size: 0.75rem;
            }
        }
    </style>
</head>
<body class="bg-gray-50">
    <?php include 'include/header.php'; ?>

    <main class="min-h-screen mb-16">
        <div class="container mx-auto px-4 py-4 max-w-4xl">
            <!-- Header -->
            <div class="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3 mb-4">
                <div>
                    <h1 class="text-xl font-bold text-gray-800">Transaction History</h1>
                    <p class="text-gray-500 text-sm">View and filter your transactions</p>
                </div>
                <a href="wallet.php" class="text-sm text-primary-600 hover:text-primary-700 flex items-center">
                    <i class="fas fa-arrow-left mr-1"></i>
                    Back to Wallet
                </a>
            </div>

            <!-- Filters Card -->
            <div class="bg-white  p-4 mb-4 border border-gray-200">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-3">
                    <!-- Custom Date Range -->
                    <div>
                        <label class="block text-gray-700 text-sm font-medium mb-1">Date Range</label>
                        <div class="flex gap-2">
                            <input type="date" 
                                   id="startDate" 
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-primary-500"
                                   value="<?= $lastMonth ?>"
                                   max="<?= $today ?>">
                            <span class="flex items-center text-gray-400">to</span>
                            <input type="date" 
                                   id="endDate" 
                                   class="flex-1 px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-primary-500"
                                   value="<?= $today ?>"
                                   max="<?= $today ?>">
                        </div>
                    </div>
                    
                    <!-- Status Filter -->
                    <div>
                        <label class="block text-gray-700 text-sm font-medium mb-1">Status</label>
                        <select id="statusFilter" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-primary-500">
                            <option value="">All Status</option>
                            <option value="completed">Completed</option>
                            <option value="pending">Pending</option>
                            <option value="failed">Failed</option>
                        </select>
                    </div>
                </div>
                
                <!-- Second Row Filters -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <!-- Type Filter -->
                    <div>
                        <label class="block text-gray-700 text-sm font-medium mb-1">Transaction Type</label>
                        <select id="typeFilter" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-primary-500">
                            <option value="">All Types</option>
                            <option value="credit">Credits Only</option>
                            <option value="debit">Debits Only</option>
                        </select>
                    </div>
                    
                    <!-- Sort Order -->
                    <div>
                        <label class="block text-gray-700 text-sm font-medium mb-1">Sort By</label>
                        <select id="sortOrder" 
                                class="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:outline-none focus:ring-1 focus:ring-primary-500">
                            <option value="desc">Newest First</option>
                            <option value="asc">Oldest First</option>
                        </select>
                    </div>
                </div>
                
                <!-- Action Buttons -->
                <div class="flex gap-2 mt-4 pt-4 border-t border-gray-200">
                    <button onclick="applyFilters()" 
                            class="flex-1 px-4 py-2 bg-primary-500 text-white text-sm rounded-lg hover:bg-primary-600 font-medium">
                        <i class="fas fa-filter mr-2"></i>
                        Apply Filters
                    </button>
                    <button onclick="clearFilters()" 
                            class="px-4 py-2 border border-gray-300 text-gray-700 text-sm rounded-lg hover:bg-gray-50">
                        Clear
                    </button>
                </div>
            </div>

            <!-- Transactions Container -->
          <!-- Transactions Container -->
<div class="bg-white rounded-lg shadow-sm border border-gray-200">
    <div class="p-3 border-b border-gray-200 bg-gray-50">
        <div class="flex justify-between items-center">
            <h2 class="font-medium text-gray-700 text-sm">Transactions</h2>
            <div id="filter-summary" class="text-gray-500 text-xs truncate max-w-[50%]">
                Showing last 30 days
            </div>
        </div>
    </div>
    <div class="p-2">
        <div id="transactions-container" class="min-w-0"> <!-- Added min-w-0 here -->
            <div class="flex flex-col items-center justify-center py-8">
                <div class="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-primary-500 mb-3"></div>
                <p class="text-gray-600 text-sm">Loading transactions...</p>
            </div>
        </div>
    </div>
</div>
        </div>
    </main>

    <!-- Loading Overlay -->
    <div id="loading-overlay" class="hidden fixed inset-0 bg-black bg-opacity-40 z-50 flex items-center justify-center">
        <div class="bg-white p-5 rounded-lg shadow-lg">
            <div class="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-primary-500 mx-auto mb-3"></div>
            <p class="text-gray-700 text-sm">Loading...</p>
        </div>
    </div>

    <script>
    // Default filter values
    let currentFilters = {
        startDate: '<?= $lastMonth ?>',
        endDate: '<?= $today ?>',
        status: '',
        type: '',
        sort: 'desc',
        page: 1
    };

    document.addEventListener('DOMContentLoaded', function() {
        loadTransactions();
        setupEventListeners();
    });

    function setupEventListeners() {
        // Load transactions when Enter key is pressed in date fields
        document.getElementById('startDate').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') applyFilters();
        });
        
        document.getElementById('endDate').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') applyFilters();
        });
        
        // Pagination
        document.addEventListener('click', function(e) {
            if (e.target.closest('.pagination-link')) {
                e.preventDefault();
                const link = e.target.closest('.pagination-link');
                const page = link.getAttribute('data-page');
                if (page && !link.classList.contains('active')) {
                    currentFilters.page = parseInt(page);
                    loadTransactions();
                }
            }
            
            // Transaction item clicks
            if (e.target.closest('.transaction-item')) {
                const item = e.target.closest('.transaction-item');
                const transactionId = item.getAttribute('data-transaction-id');
                if (transactionId) {
                    showTransactionDetails(transactionId);
                }
            }
        });
    }

    function applyFilters() {
        const startDate = document.getElementById('startDate').value;
        const endDate = document.getElementById('endDate').value;
        
        if (startDate && endDate && startDate > endDate) {
            alert('Start date cannot be after end date');
            return;
        }
        
        // Update current filters
        currentFilters.startDate = startDate;
        currentFilters.endDate = endDate;
        currentFilters.status = document.getElementById('statusFilter').value;
        currentFilters.type = document.getElementById('typeFilter').value;
        currentFilters.sort = document.getElementById('sortOrder').value;
        currentFilters.page = 1;
        
        loadTransactions();
    }

    function loadTransactions() {
        showLoading();
        const container = document.getElementById('transactions-container');
        
        // Build query string
        const params = new URLSearchParams({
            page: currentFilters.page,
            start_date: currentFilters.startDate || '',
            end_date: currentFilters.endDate || '',
            status: currentFilters.status,
            type: currentFilters.type,
            sort: currentFilters.sort
        }).toString();
        
        console.log('Loading transactions with params:', params); // Debug
        
        fetch(`load_transactions.php?${params}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error('Network response was not ok');
                }
                return response.text();
            })
            .then(html => {
                container.innerHTML = html;
                hideLoading();
                updateFilterSummary();
                
                // Animate items
                const items = container.querySelectorAll('.transaction-item');
                items.forEach((item, index) => {
                    item.style.animationDelay = `${index * 0.03}s`;
                    item.classList.add('fade-in');
                });
            })
            .catch(error => {
                console.error('Error loading transactions:', error);
                container.innerHTML = `
                    <div class="text-center py-8">
                        <i class="fas fa-exclamation-circle text-2xl text-red-400 mb-3"></i>
                        <p class="text-gray-600 mb-3">Failed to load transactions</p>
                        <p class="text-gray-400 text-sm mb-3">Please check your connection</p>
                        <button onclick="loadTransactions()" 
                                class="px-4 py-2 bg-primary-500 text-white text-sm rounded-lg hover:bg-primary-600">
                            <i class="fas fa-redo mr-2"></i>Retry
                        </button>
                    </div>
                `;
                hideLoading();
            });
    }

    function clearFilters() {
        // Reset all filters to default
        document.getElementById('startDate').value = '<?= $lastMonth ?>';
        document.getElementById('endDate').value = '<?= $today ?>';
        document.getElementById('statusFilter').value = '';
        document.getElementById('typeFilter').value = '';
        document.getElementById('sortOrder').value = 'desc';
        
        // Reset filter object
        currentFilters = {
            startDate: '<?= $lastMonth ?>',
            endDate: '<?= $today ?>',
            status: '',
            type: '',
            sort: 'desc',
            page: 1
        };
        
        loadTransactions();
    }

    function updateFilterSummary() {
        const summary = document.getElementById('filter-summary');
        
        if (!currentFilters.startDate || !currentFilters.endDate) {
            summary.textContent = 'Showing all transactions';
            return;
        }
        
        const start = new Date(currentFilters.startDate);
        const end = new Date(currentFilters.endDate);
        
        // Format date nicely
        const formatDate = (date) => {
            const options = { 
                month: 'short', 
                day: 'numeric',
                year: date.getFullYear() !== new Date().getFullYear() ? 'numeric' : undefined
            };
            return date.toLocaleDateString('en-US', options);
        };
        
        let text = `${formatDate(start)} - ${formatDate(end)}`;
        
        if (currentFilters.status) {
            text += ` • ${currentFilters.status}`;
        }
        
        if (currentFilters.type) {
            text += currentFilters.type === 'credit' ? ' • Credits' : ' • Debits';
        }
        
        summary.textContent = text;
    }

    function showTransactionDetails(transactionId) {
        showLoading();
        fetch(`transaction_details.php?id=${transactionId}`)
            .then(response => {
                if (!response.ok) throw new Error('Failed to load details');
                return response.text();
            })
            .then(html => {
                const detailsPage = document.createElement('div');
                detailsPage.id = 'transaction-details-page';
                detailsPage.innerHTML = html;
                document.body.appendChild(detailsPage);
                
                document.querySelector('main').style.opacity = '0.5';
                document.querySelector('main').style.pointerEvents = 'none';
                
                const backButton = detailsPage.querySelector('#back-to-transactions');
                if (backButton) {
                    backButton.addEventListener('click', hideTransactionDetails);
                }
                
                hideLoading();
            })
            .catch(error => {
                hideLoading();
                alert('Failed to load transaction details. Please try again.');
                console.error('Details error:', error);
            });
    }

    function hideTransactionDetails() {
        const detailsPage = document.getElementById('transaction-details-page');
        if (detailsPage) detailsPage.remove();
        
        const main = document.querySelector('main');
        main.style.opacity = '1';
        main.style.pointerEvents = 'auto';
    }

    function showLoading() {
        document.getElementById('loading-overlay').classList.remove('hidden');
    }

    function hideLoading() {
        document.getElementById('loading-overlay').classList.add('hidden');
    }
    </script>

    <?php include 'include/footer.php'; ?>
</body>
</html>