<?php
// webhooks/paystack.php
require_once '../config/conn.php';

// Get input
$input = file_get_contents('php://input');
$event = json_decode($input);

// Verify signature
$signature = $_SERVER['HTTP_X_PAYSTACK_SIGNATURE'] ?? '';
$expected_signature = hash_hmac('sha512', $input, PAYSTACK_SECRET_KEY);

if ($signature !== $expected_signature) {
    http_response_code(401);
    exit('Invalid signature');
}

// Handle events
switch ($event->event) {
    case 'transfer.success':
        handleTransferSuccess($event->data, $conn);
        break;
        
    case 'transfer.failed':
        handleTransferFailed($event->data, $conn);
        break;
        
    case 'transfer.reversed':
        handleTransferReversed($event->data, $conn);
        break;
}

function handleTransferSuccess($data, $conn) {
    $reference = $conn->real_escape_string($data->reference);
    
    // Update transaction status
    $stmt = $conn->prepare("
        UPDATE withdrawal_transactions 
        SET status = 'completed', 
            description = CONCAT(description, ' - Transfer completed'),
            processed_at = NOW()
        WHERE paystack_reference = ? OR paystack_transfer_code = ?
    ");
    $stmt->bind_param("ss", $reference, $reference);
    $stmt->execute();
    
    http_response_code(200);
    echo 'Webhook processed';
}

function handleTransferFailed($data, $conn) {
    $reference = $conn->real_escape_string($data->reference);
    
    // Get transaction to refund
    $stmt = $conn->prepare("
        SELECT * FROM withdrawal_transactions 
        WHERE paystack_reference = ? OR paystack_transfer_code = ?
    ");
    $stmt->bind_param("ss", $reference, $reference);
    $stmt->execute();
    $transaction = $stmt->get_result()->fetch_assoc();
    
    if ($transaction) {
        // Refund wallet
        $refund_stmt = $conn->prepare("
            UPDATE student_wallets 
            SET balance = balance + ? 
            WHERE id = ?
        ");
        $refund_stmt->bind_param("di", $transaction['total_deduction'], $transaction['wallet_id']);
        $refund_stmt->execute();
        
        // Update transaction status
        $update_stmt = $conn->prepare("
            UPDATE withdrawal_transactions 
            SET status = 'failed', 
                description = CONCAT(description, ' - Transfer failed: ', ?)
            WHERE id = ?
        ]);
        $reason = $data->reason ?? 'Unknown reason';
        $update_stmt->bind_param("si", $reason, $transaction['id']);
        $update_stmt->execute();
    }
    
    http_response_code(200);
    echo 'Webhook processed';
}

http_response_code(200);
echo 'Webhook processed';
?>