<?php
// Start output buffering at the VERY beginning
if (!ob_get_level()) {
    ob_start();
}

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include config
require_once '../config/conn.php';

// Debug: Log everything
error_log("=========================================");
error_log("WITHDRAW PAGE ACCESSED - " . date('Y-m-d H:i:s'));

// Authentication check
if (!isset($_SESSION['user_id'])) {
    // Check if it's an AJAX request
    if (isset($_POST['ajax']) && $_POST['ajax'] == '1') {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Unauthorized', 'redirect' => 'login.php']);
        exit();
    } else {
        error_log("No session found, redirecting to login");
        header("Location: login.php");
        exit();
    }
}

$student_id = $_SESSION['user_id'];

// ========== AJAX REQUEST HANDLING ==========
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax']) && $_POST['ajax'] == '1') {
    error_log("=== AJAX REQUEST STARTED ===");
    
    $response = ['success' => false, 'message' => '', 'data' => null];
    
    // Check if user has set transaction PIN
    $pin_stmt = $conn->prepare("SELECT * FROM wallet_pins WHERE student_id = ?");
    $pin_stmt->bind_param("i", $student_id);
    $pin_stmt->execute();
    $has_pin = $pin_stmt->get_result()->num_rows > 0;

    if (!$has_pin) {
        $response['message'] = 'PIN not set';
        $response['redirect'] = 'pin.php';
        header('Content-Type: application/json');
        echo json_encode($response);
        exit();
    }

    // Get wallet info
    $wallet_stmt = $conn->prepare("SELECT id, wallet_id, balance FROM student_wallets WHERE student_id = ?");
    $wallet_stmt->bind_param("i", $student_id);
    $wallet_stmt->execute();
    $wallet_result = $wallet_stmt->get_result();
    $wallet = $wallet_result->fetch_assoc();

    if (!$wallet) {
        $response['message'] = "Wallet not found!";
        error_log("ERROR: Wallet not found for student_id: $student_id");
        header('Content-Type: application/json');
        echo json_encode($response);
        exit();
    }
    
    // Get form data
    $amount = isset($_POST['amount']) && is_numeric($_POST['amount']) ? floatval($_POST['amount']) : 0;
    $bank_account_id = isset($_POST['bank_account_id']) && is_numeric($_POST['bank_account_id']) ? intval($_POST['bank_account_id']) : 0;
    $transaction_pin = $_POST['transaction_pin'] ?? '';
    $wallet_id = isset($_POST['wallet_id']) && is_numeric($_POST['wallet_id']) ? intval($_POST['wallet_id']) : 0;
    
    // Validation
    $valid = true;
    
    if ($amount < 100) {
        $response['message'] = "Minimum withdrawal is ₦100";
        $valid = false;
    } elseif ($amount > $wallet['balance']) {
        $response['message'] = "Insufficient balance";
        $valid = false;
    } elseif ($bank_account_id <= 0) {
        $response['message'] = "Please select a bank account";
        $valid = false;
    } elseif (strlen($transaction_pin) !== 4 || !is_numeric($transaction_pin)) {
        $response['message'] = "PIN must be 4 digits";
        $valid = false;
    }
    
    if ($valid) {
        // Verify transaction PIN
        $pin_check = $conn->prepare("SELECT pin_hash FROM wallet_pins WHERE student_id = ?");
        $pin_check->bind_param("i", $student_id);
        $pin_check->execute();
        $pin_result = $pin_check->get_result();
        
        if ($pin_result->num_rows === 0) {
            $response['message'] = "Transaction PIN not set";
        } else {
            $pin_data = $pin_result->fetch_assoc();
            if (!password_verify($transaction_pin, $pin_data['pin_hash'])) {
                $response['message'] = "Invalid transaction PIN";
            } else {
                // Verify bank account
                $bank_check = $conn->prepare("SELECT * FROM student_bank_accounts WHERE id = ? AND student_id = ?");
                $bank_check->bind_param("ii", $bank_account_id, $student_id);
                $bank_check->execute();
                $bank_result = $bank_check->get_result();
                
                if ($bank_result->num_rows === 0) {
                    $response['message'] = "Bank account not found";
                } else {
                    $bank_account = $bank_result->fetch_assoc();
                    
                    try {
                        // Generate references
                        $reference = 'WDL-' . time() . '-' . strtoupper(uniqid());
                        $fee = 0.00;
                        $total_deduction = $amount;
                        
                        error_log("Creating withdrawal transaction: $reference");
                        
                        // Start transaction
                        $conn->begin_transaction();
                        
                        // Insert withdrawal transaction
                        $description = "Withdrawal to {$bank_account['bank_name']}";
                        $withdraw_stmt = $conn->prepare("
                            INSERT INTO withdrawal_transactions 
                            (student_id, wallet_id, bank_account_id, reference, 
                            amount, total_deduction, status, description, attempted_at)
                            VALUES (?, ?, ?, ?, ?, ?, 'pending', ?, NOW())
                        ");
                        
                        $withdraw_stmt->bind_param(
                            "iiisdds",
                            $student_id, 
                            $wallet_id, 
                            $bank_account_id, 
                            $reference, 
                            $amount, 
                            $total_deduction,
                            $description
                        );
                        
                        if ($withdraw_stmt->execute()) {
                            $transaction_id = $conn->insert_id;
                            error_log("Withdrawal transaction created with ID: $transaction_id");
                            
                            // Create wallet transaction
                            $balance_before = $wallet['balance'];
                            $balance_after = $balance_before - $total_deduction;
                            
                            $wallet_trans_stmt = $conn->prepare("
                                INSERT INTO wallet_transactions 
                                (student_id, wallet_id, transaction_type, amount, balance_before, 
                                balance_after, reference, description)
                                VALUES (?, ?, 'withdrawal', ?, ?, ?, ?, ?)
                            ");
                            $wallet_trans_stmt->bind_param(
                                "iidddss", 
                                $student_id, 
                                $wallet_id, 
                                $total_deduction, 
                                $balance_before, 
                                $balance_after, 
                                $reference, 
                                $description
                            );
                            
                            if (!$wallet_trans_stmt->execute()) {
                                throw new Exception("Wallet transaction failed: " . $wallet_trans_stmt->error);
                            }
                            
                            // Update wallet balance
                            $update_wallet = $conn->prepare("
                                UPDATE student_wallets 
                                SET balance = balance - ? 
                                WHERE id = ? AND student_id = ?
                            ");
                            $update_wallet->bind_param("dii", $total_deduction, $wallet_id, $student_id);
                            
                            if (!$update_wallet->execute()) {
                                throw new Exception("Wallet update failed: " . $update_wallet->error);
                            }
                            
                            // Commit transaction
                            $conn->commit();
                            
                            // Prepare success response
                            $response['success'] = true;
                            $response['message'] = "Withdrawal initiated successfully!";
                            $response['data'] = [
                                'reference' => $reference,
                                'amount' => $amount,
                                'fee' => $fee,
                                'total_deduction' => $total_deduction,
                                'status' => 'pending',
                                'bank_name' => $bank_account['bank_name'],
                                'account_last4' => substr($bank_account['account_number'], -4),
                                'balance_after' => $balance_after,
                                'balance_before' => $balance_before
                            ];
                            
                            error_log("SUCCESS: Withdrawal recorded");
                            
                        } else {
                            throw new Exception("Failed to create transaction: " . $withdraw_stmt->error);
                        }
                        
                    } catch (Exception $e) {
                        // Rollback transaction on error
                        $conn->rollback();
                        $response['message'] = "System error: " . $e->getMessage();
                        error_log("Exception: " . $e->getMessage());
                    }
                }
            }
        }
    }
    
    error_log("=== AJAX REQUEST COMPLETE ===");
    header('Content-Type: application/json');
    echo json_encode($response);
    exit();
}

// ========== NORMAL PAGE RENDERING (Non-AJAX) ==========
// Check if user has set transaction PIN
$pin_stmt = $conn->prepare("SELECT * FROM wallet_pins WHERE student_id = ?");
$pin_stmt->bind_param("i", $student_id);
$pin_stmt->execute();
$has_pin = $pin_stmt->get_result()->num_rows > 0;

if (!$has_pin) {
    header("Location: pin.php");
    exit();
}

// Get wallet info
$wallet_stmt = $conn->prepare("SELECT id, wallet_id, balance FROM student_wallets WHERE student_id = ?");
$wallet_stmt->bind_param("i", $student_id);
$wallet_stmt->execute();
$wallet_result = $wallet_stmt->get_result();
$wallet = $wallet_result->fetch_assoc();

if (!$wallet) {
    $error = "Wallet not found!";
    error_log("ERROR: Wallet not found for student_id: $student_id");
}

// Get linked bank accounts
$accounts_stmt = $conn->prepare("
    SELECT * FROM student_bank_accounts
    WHERE student_id = ?
    ORDER BY is_primary DESC, created_at DESC
");
$accounts_stmt->bind_param("i", $student_id);
$accounts_stmt->execute();
$accounts_result = $accounts_stmt->get_result();
$bank_accounts = $accounts_result->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Withdraw Funds</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Mobile-optimized styles */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .fade-in {
            animation: fadeIn 0.3s ease-out;
        }
        
        /* Prevent zoom on input focus on mobile */
        input, select, textarea {
            font-size: 16px !important;
        }
        
        /* Better touch targets for mobile */
        button, a, select, input {
            min-height: 44px;
        }
        
        /* Mobile-specific optimizations */
        @media (max-width: 640px) {
            .mobile-text-sm { font-size: 0.875rem; }
            .mobile-text-xs { font-size: 0.75rem; }
            .mobile-p-3 { padding: 0.75rem; }
            .mobile-py-3 { padding-top: 0.75rem; padding-bottom: 0.75rem; }
            .mobile-px-3 { padding-left: 0.75rem; padding-right: 0.75rem; }
            .mobile-space-y-3 > * + * { margin-top: 0.75rem; }
            .mobile-text-lg { font-size: 1.125rem; }
            .mobile-text-xl { font-size: 1.25rem; }
            
            /* Adjust input sizes for mobile */
            input[type="number"], input[type="password"], select {
                padding: 12px !important;
                height: 48px !important;
            }
            
            /* Improve button touch area */
            .mobile-btn {
                padding: 14px 16px !important;
                font-size: 16px !important;
            }
            
            /* Reduce margins on mobile */
            .mobile-mb-2 { margin-bottom: 0.5rem; }
            .mobile-mb-3 { margin-bottom: 0.75rem; }
            .mobile-mb-4 { margin-bottom: 1rem; }
            .mobile-mt-2 { margin-top: 0.5rem; }
            
            /* Adjust spacing for mobile */
            .mobile-container {
                padding-left: 0.75rem;
                padding-right: 0.75rem;
            }
        }
        
        /* Hide number input spinners */
        input[type=number]::-webkit-inner-spin-button,
        input[type=number]::-webkit-outer-spin-button {
            -webkit-appearance: none;
            margin: 0;
        }
        input[type=number] {
            -moz-appearance: textfield;
        }
        
        /* Smooth scrolling */
        html {
            scroll-behavior: smooth;
        }
    </style>
</head>

<body class="bg-gray-50">
<?php include 'include/header.php'; ?>

<main class=" mb-20">
    <div class="container mx-auto px-3 py-3 max-w-md mobile-container">
        <!-- Messages Container (for AJAX responses) -->
        <div id="messagesContainer" class="mobile-mb-3"></div>

        <!-- Header -->
        <div class="mobile-mb-4" id="pageHeader">
            <div class="flex items-center justify-between mobile-mb-3">
                <a href="wallet.php" class="text-gray-600 hover:text-gray-800 flex items-center mobile-text-sm">
                    <i class="fas fa-arrow-left mr-2"></i>
                    <span class="font-medium">Back</span>
                </a>
                <div class="text-right">
                    <p class="text-gray-500 mobile-text-xs">Available Balance</p>
                    <p id="currentBalance" class="mobile-text-xl font-bold text-gray-800">₦<?= number_format($wallet['balance'], 2) ?></p>
                </div>
            </div>
            
            <div class="text-center mobile-mb-4">
                <div class="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center mx-auto mobile-mb-2">
                    <i class="fas fa-arrow-up text-red-500 text-lg"></i>
                </div>
                <h1 class="mobile-text-lg font-bold text-gray-800">Withdraw Funds</h1>
                <p class="text-gray-500 mobile-text-xs mobile-mt-1">Transfer to your bank account</p>
            </div>
        </div>

        <!-- Withdrawal Form -->
        <form method="POST" action="" id="withdrawForm" class="space-y-3 mobile-space-y-3">
            <input type="hidden" name="wallet_id" value="<?= $wallet['id'] ?>">
            <input type="hidden" name="ajax" value="0" id="ajaxFlag">
            
            <!-- Amount Input -->
            <div class=" border border-gray-200 mobile-p-3 shadow-sm">
                <label class="block text-gray-700 mobile-text-sm font-medium mobile-mb-2">Amount (₦)</label>
                <div class="relative">
                    <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 font-medium"></span>
                    <input type="number" 
                           name="amount" 
                           id="amount"
                           min="100" 
                           step="100" 
                           max="<?= $wallet['balance'] ?>" 
                           class="w-full pl-10 pr-3 mobile-text-lg font-medium border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-primary-500"
                           placeholder="0.00" 
                           required
                           value=""
                           oninput="updateSummary(this.value)"
                           inputmode="decimal">
                </div>
                <div class="flex justify-between mobile-mt-2">
                    <span class="mobile-text-xs text-gray-500">Min: ₦100</span>
                    <span class="mobile-text-xs text-gray-500">Available: ₦<?= number_format($wallet['balance'], 2) ?></span>
                </div>
            </div>

            <!-- Bank Account Selection -->
            <div class=" border border-gray-200 mobile-p-3 shadow-sm">
                <label class="block text-gray-700 mobile-text-sm font-medium mobile-mb-2">Bank Account</label>
                <?php if (empty($bank_accounts)): ?>
                    <div class="text-center mobile-py-3">
                        <div class="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center mx-auto mobile-mb-2">
                            <i class="fas fa-university text-gray-400"></i>
                        </div>
                        <p class="text-gray-600 mobile-text-sm mobile-mb-2">No bank accounts added</p>
                        <a href="add-bank-account.php" 
                           class="inline-flex items-center mobile-px-3 mobile-py-2 bg-primary-500 text-white mobile-text-sm rounded-lg hover:bg-primary-600 mobile-btn">
                            <i class="fas fa-plus mr-2"></i>
                            Add Bank Account
                        </a>
                    </div>
                <?php else: ?>
                    <select name="bank_account_id" 
                            id="bank_account_id"
                            class="w-full mobile-px-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-primary-500 mobile-text-sm"
                            required>
                        <option value="">Select Bank Account</option>
                        <?php foreach ($bank_accounts as $account): ?>
                            <option value="<?= $account['id'] ?>" 
                                    <?= $account['is_primary'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($account['bank_name']) ?> • ****<?= substr($account['account_number'], -4) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="mobile-mt-2 text-right">
                        <a href="add-bank-account.php" class="text-primary-600 hover:text-primary-700 mobile-text-xs">
                            <i class="fas fa-plus mr-1"></i> Add Another Account
                        </a>
                    </div>
                <?php endif; ?>
            </div>

            <!-- PIN Input -->
            <div class=" border border-gray-200 mobile-p-3 shadow-sm">
                <label class="block text-gray-700 mobile-text-sm font-medium mobile-mb-2">Transaction PIN</label>
                <input type="password" 
                       name="transaction_pin" 
                       id="transaction_pin"
                       maxlength="4" 
                       inputmode="numeric" 
                       pattern="\d{4}"
                       class="w-full mobile-px-3 text-center mobile-text-lg tracking-widest border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-primary-500 font-mono"
                       placeholder="••••" 
                       required
                       autocomplete="off">
                <p class="mobile-text-xs text-gray-500 mobile-mt-2 text-center">Enter your 4-digit security PIN</p>
            </div>

            <!-- Summary Card -->
            <div id="withdrawalSummary" class="bg-gray-50 border border-gray-200 rounded-lg mobile-p-3" style="display: none;">
                <h4 class="font-semibold text-gray-800 mobile-text-sm mobile-mb-2 flex items-center">
                    <i class="fas fa-receipt text-primary-500 mr-2"></i>
                    Withdrawal Summary
                </h4>
                <div class="space-y-2">
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600 mobile-text-sm">Amount to withdraw:</span>
                        <span id="summaryAmount" class="font-medium text-gray-800 mobile-text-sm"></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600 mobile-text-sm">Transaction fee:</span>
                        <span id="summaryFee" class="font-medium text-gray-800 mobile-text-sm">₦0.00</span>
                    </div>
                    <div class="border-t border-gray-200 mobile-pt-2 mobile-mt-2">
                        <div class="flex justify-between items-center">
                            <span class="text-gray-800 font-semibold mobile-text-sm">Total deduction:</span>
                            <span id="summaryTotal" class="text-red-600 font-bold mobile-text-lg"></span>
                        </div>
                        <div class="flex justify-between items-center mobile-mt-2">
                            <span class="text-gray-600 mobile-text-xs">Balance after:</span>
                            <span id="summaryBalanceAfter" class="text-gray-600 mobile-text-xs"></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Info Notice -->
            <div class="bg-blue-50 border border-blue-200 rounded-lg mobile-p-3">
                <div class="flex items-start">
                    <div class="w-5 h-5 bg-blue-100 rounded-full flex items-center justify-center mr-2 flex-shrink-0">
                        <i class="fas fa-info text-blue-500 text-xs"></i>
                    </div>
                    <div>
                        <p class="text-blue-800 mobile-text-sm font-medium">Important Information</p>
                        <ul class="text-blue-600 mobile-text-xs mobile-mt-1 space-y-1">
                            <li class="flex items-start">
                                <span class="mr-1">•</span>
                                <span>Withdrawals may take 1-3 business days</span>
                            </li>
                            <li class="flex items-start">
                                <span class="mr-1">•</span>
                                <span>No transaction fees apply</span>
                            </li>
                            <li class="flex items-start">
                                <span class="mr-1">•</span>
                                <span>Minimum withdrawal: ₦100</span>
                            </li>
                            <li class="flex items-start">
                                <span class="mr-1">•</span>
                                <span>Ensure bank details are correct</span>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Submit Button -->
            <div class="mobile-pt-2">
                <button type="submit" 
                        name="withdraw"
                        value="1"
                        id="submitBtn"
                        class="w-full bg-primary-500 text-white mobile-py-3 mobile-px-4  hover:bg-primary-600 active:bg-primary-700 font-medium mobile-text-sm transition-colors flex items-center justify-center disabled:opacity-50 disabled:cursor-not-allowed mobile-btn "
                        <?= empty($bank_accounts) ? 'disabled' : '' ?>>
                    <i class="fas fa-paper-plane mr-2"></i>
                    <span>Proceed Withdrawal</span>
                </button>
            </div>
        </form>
    </div>
</main>

<!-- Loading Overlay -->
<div id="loadingOverlay" class="hidden fixed inset-0 bg-black bg-opacity-40 z-50 flex items-center justify-center">
    <div class="bg-white p-4 rounded-lg shadow-lg mx-4">
        <div class="animate-spin rounded-full h-8 w-8 border-t-2 border-b-2 border-primary-500 mx-auto mobile-mb-2"></div>
        <p class="text-gray-700 mobile-text-sm text-center">Processing withdrawal...</p>
    </div>
</div>

<!-- Success Modal Template -->
<div id="successModalTemplate" class="hidden">
    <div class="fixed inset-0 bg-black bg-opacity-40 z-50 flex items-center justify-center p-4">
        <div class="bg-white  w-full max-w-sm">
            <div class="p-5">
                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mobile-mb-3">
                    <i class="fas fa-check text-green-500 text-2xl"></i>
                </div>
                <h3 class="text-lg font-bold text-gray-800 text-center mobile-mb-2">Success!</h3>
                <p class="text-gray-600 mobile-text-sm text-center mobile-mb-4" id="modalMessage"></p>
                
                <div class="bg-gray-50 rounded-lg p-3 mobile-mb-4">
                    <div class="flex justify-between items-center mobile-mb-1">
                        <span class="text-gray-600 mobile-text-xs">Reference:</span>
                        <span class="font-medium mobile-text-xs" id="modalRef"></span>
                    </div>
                    <div class="flex justify-between items-center mobile-mb-1">
                        <span class="text-gray-600 mobile-text-xs">Amount:</span>
                        <span class="font-medium mobile-text-xs" id="modalAmount"></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <span class="text-gray-600 mobile-text-xs">Status:</span>
                        <span class="px-2 py-1 mobile-text-xs rounded-full bg-yellow-100 text-yellow-800">Pending</span>
                    </div>
                </div>
                
                <div class="space-y-2">
                    <a href="wallet.php" 
                       class="block w-full bg-primary-500 text-white py-3 px-4 rounded-lg hover:bg-primary-600 font-medium mobile-text-sm text-center">
                        Back to Wallet
                    </a>
                   
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    const currentBalance = parseFloat(<?= $wallet['balance'] ?>);
    
    function updateSummary(amount) {
        const parsedAmount = parseFloat(amount) || 0;
        const summaryDiv = document.getElementById('withdrawalSummary');
        
        if (parsedAmount >= 100) {
            const totalDeduction = parsedAmount;
            const balanceAfter = currentBalance - totalDeduction;
            
            document.getElementById('summaryAmount').textContent = '₦' + parsedAmount.toFixed(2);
            document.getElementById('summaryTotal').textContent = '₦' + totalDeduction.toFixed(2);
            document.getElementById('summaryBalanceAfter').textContent = '₦' + balanceAfter.toFixed(2);
            
            if (balanceAfter < 0) {
                summaryDiv.classList.add('border-red-300', 'bg-red-50');
                summaryDiv.classList.remove('border-gray-200', 'bg-gray-50');
                document.getElementById('submitBtn').disabled = true;
            } else {
                summaryDiv.classList.remove('border-red-300', 'bg-red-50');
                summaryDiv.classList.add('border-gray-200', 'bg-gray-50');
                document.getElementById('submitBtn').disabled = false;
            }
            
            summaryDiv.style.display = 'block';
            
            // Smooth scroll to summary
            setTimeout(() => {
                summaryDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            }, 300);
        } else {
            summaryDiv.style.display = 'none';
            document.getElementById('submitBtn').disabled = false;
        }
    }
    
    // AJAX Form Submission
    document.getElementById('withdrawForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Set ajax flag
        document.getElementById('ajaxFlag').value = '1';
        
        // Get form data
        const formData = new FormData(this);
        
        // Validate form
        const amount = parseFloat(document.getElementById('amount').value) || 0;
        const bankAccountId = parseInt(document.getElementById('bank_account_id').value) || 0;
        const transactionPin = document.getElementById('transaction_pin').value;
        
        if (amount < 100) {
            showMessage('error', 'Minimum withdrawal is ₦100');
            return;
        }
        
        if (amount > currentBalance) {
            showMessage('error', 'Insufficient balance');
            return;
        }
        
        if (bankAccountId <= 0) {
            showMessage('error', 'Please select a bank account');
            return;
        }
        
        if (transactionPin.length !== 4 || !/^\d+$/.test(transactionPin)) {
            showMessage('error', 'PIN must be 4 digits');
            return;
        }
        
        // Show loading
        showLoading(true);
        
        // Disable submit button
        document.getElementById('submitBtn').disabled = true;
        
        // Send AJAX request
        fetch(window.location.href, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then(data => {
            // Hide loading
            showLoading(false);
            
            if (data.success) {
                // Show success modal
                showSuccessModal(data);
                
                // Update balance display
                document.getElementById('currentBalance').textContent = 
                    '₦' + parseFloat(data.data.balance_after).toFixed(2);
                
                // Disable form
                disableForm();
                
                // Clear PIN field
                document.getElementById('transaction_pin').value = '';
            } else {
                // Show error message
                showMessage('error', data.message);
                
                // Re-enable submit button
                document.getElementById('submitBtn').disabled = false;
                
                // Check if redirect needed
                if (data.redirect) {
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 2000);
                }
            }
        })
        .catch(error => {
            // Hide loading
            showLoading(false);
            
            // Re-enable submit button
            document.getElementById('submitBtn').disabled = false;
            
            showMessage('error', 'Network error. Please try again.');
            console.error('Error:', error);
        });
    });
    
    function showSuccessModal(data) {
        // Clone and show modal template
        const template = document.getElementById('successModalTemplate');
        const modal = template.cloneNode(true);
        modal.id = 'successModal';
        modal.classList.remove('hidden');
        
        // Fill modal data
        modal.querySelector('#modalMessage').textContent = data.message;
        modal.querySelector('#modalRef').textContent = data.data.reference;
        modal.querySelector('#modalAmount').textContent = '₦' + parseFloat(data.data.amount).toFixed(2);
        
        // Add close functionality
        modal.addEventListener('click', function(e) {
            if (e.target === this || e.target.classList.contains('close-modal')) {
                document.body.removeChild(this);
            }
        });
        
        // Add to page
        document.body.appendChild(modal);
        
        // Clear form
        document.getElementById('amount').value = '';
        document.getElementById('withdrawalSummary').style.display = 'none';
    }
    
    function showMessage(type, message) {
        // Clear existing messages
        clearMessages();
        
        const messageHtml = `
            <div class="fade-in">
                <div class="${type === 'error' ? 'bg-red-50 border-red-200' : 'bg-blue-50 border-blue-200'} border rounded-lg mobile-p-3 mobile-mb-3">
                    <div class="flex items-start">
                        <div class="w-6 h-6 ${type === 'error' ? 'bg-red-100' : 'bg-blue-100'} rounded-full flex items-center justify-center mr-2 flex-shrink-0">
                            <i class="fas ${type === 'error' ? 'fa-exclamation text-red-500' : 'fa-info text-blue-500'} text-sm"></i>
                        </div>
                        <div>
                            <p class="font-medium ${type === 'error' ? 'text-red-800' : 'text-blue-800'} mobile-text-sm">${type === 'error' ? 'Error' : 'Notice'}</p>
                            <p class="${type === 'error' ? 'text-red-600' : 'text-blue-600'} mobile-text-xs mobile-mt-1">${message}</p>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Insert into messages container
        document.getElementById('messagesContainer').innerHTML = messageHtml;
        
        // Scroll to messages
        document.getElementById('messagesContainer').scrollIntoView({ 
            behavior: 'smooth', 
            block: 'start' 
        });
        
        // Auto-remove error messages after 5 seconds
        if (type === 'error') {
            setTimeout(() => {
                clearMessages();
            }, 5000);
        }
    }
    
    function clearMessages() {
        document.getElementById('messagesContainer').innerHTML = '';
    }
    
    function disableForm() {
        // Additional cleanup if needed
        document.getElementById('amount').value = '';
        document.getElementById('transaction_pin').value = '';
        document.getElementById('withdrawalSummary').style.display = 'none';
    }
    
    function showLoading(show) {
        const overlay = document.getElementById('loadingOverlay');
        if (show) {
            overlay.classList.remove('hidden');
        } else {
            overlay.classList.add('hidden');
        }
    }
    
    // Mobile-specific optimizations
    document.addEventListener('DOMContentLoaded', function() {
        const amountInput = document.getElementById('amount');
        if (amountInput) {
            // Auto-focus amount input on mobile
            setTimeout(() => {
                amountInput.focus();
                // For mobile, this will trigger the numeric keyboard
                amountInput.setAttribute('inputmode', 'decimal');
            }, 300);
            
            amountInput.addEventListener('input', function() {
                updateSummary(this.value);
            });
        }
        
        // Prevent form submission on Enter key for PIN field
        document.getElementById('transaction_pin').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
            }
        });
        
        // Auto-advance PIN input (if you want 4 separate inputs)
        // This is optional but improves mobile UX
        
        // Add touch feedback to buttons
        const buttons = document.querySelectorAll('button, a');
        buttons.forEach(btn => {
            btn.addEventListener('touchstart', function() {
                this.classList.add('active:scale-95');
            });
            btn.addEventListener('touchend', function() {
                this.classList.remove('active:scale-95');
            });
        });
    });
</script>

<?php include 'include/footer.php'; ?>
</body>
</html>