<?php
require_once '../../config/conn.php';
session_start();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

if (!isset($_SESSION['vendor_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit();
}

$vendor_id = $_SESSION['vendor_id'];
$bank_code = $_POST['bank_code'] ?? '';
$account_number = $_POST['account_number'] ?? '';
$account_name = $_POST['account_name'] ?? '';
$account_id = $_POST['account_id'] ?? '';

// Validate inputs
$errors = [];
if (empty($bank_code)) $errors['bank_code'] = 'Please select a bank';
if (empty($account_number) || !preg_match('/^\d{10}$/', $account_number)) {
    $errors['account_number'] = 'Please enter a valid 10-digit account number';
}
if (empty($account_name)) $errors['account_name'] = 'Account name is required';
if (!isset($_POST['confirm_name'])) $errors['confirm_name'] = 'Please confirm the account name';

if (!empty($errors)) {
    echo json_encode(['success' => false, 'errors' => $errors]);
    exit();
}

// Get bank name from bank code (you might want to store this in a separate table)
$bank_name = '';
$bank_response = json_decode(file_get_contents('https://api.paystack.co/bank/' . $bank_code), true);
if ($bank_response['status']) {
    $bank_name = $bank_response['data']['name'];
}

try {
    $conn->begin_transaction();
    
    if (empty($account_id)) {
        // Check if vendor already has an account
        $check_stmt = $conn->prepare("SELECT id FROM vendor_bank_accounts WHERE vendor_id = ?");
        $check_stmt->bind_param("i", $vendor_id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows > 0) {
            echo json_encode(['success' => false, 'message' => 'You already have a bank account. Please delete it first.']);
            exit();
        }
        
        // Insert new account
        $stmt = $conn->prepare("
            INSERT INTO vendor_bank_accounts (vendor_id, bank_name, bank_code, account_name, account_number, is_verified)
            VALUES (?, ?, ?, ?, ?, 1)
        ");
        $stmt->bind_param("issss", $vendor_id, $bank_name, $bank_code, $account_name, $account_number);
    } else {
        // Update existing account
        $stmt = $conn->prepare("
            UPDATE vendor_bank_accounts 
            SET bank_name = ?, bank_code = ?, account_name = ?, account_number = ?, is_verified = 1
            WHERE id = ? AND vendor_id = ?
        ");
        $stmt->bind_param("ssssii", $bank_name, $bank_code, $account_name, $account_number, $account_id, $vendor_id);
    }
    
    if ($stmt->execute()) {
        $conn->commit();
        echo json_encode([
            'success' => true,
            'message' => empty($account_id) ? 'Bank account added successfully!' : 'Bank account updated successfully!'
        ]);
    } else {
        $conn->rollback();
        echo json_encode(['success' => false, 'message' => 'Failed to save bank account']);
    }
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>