<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
require '../config/conn.php';

header('Content-Type: application/json');

$response = [
    'success' => false,
    'message' => ''
];

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }

    $token = $_POST['token'] ?? '';
    $email = $_POST['email'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    // Validate inputs
    if (empty($token) || empty($email) || empty($new_password) || empty($confirm_password)) {
        throw new Exception('All fields are required');
    }

    if ($new_password !== $confirm_password) {
        throw new Exception('Passwords do not match');
    }

    // Validate password strength
    if (strlen($new_password) < 8) {
        throw new Exception('Password must be at least 8 characters long');
    }
    if (!preg_match('/[A-Z]/', $new_password)) {
        throw new Exception('Password must contain at least one uppercase letter');
    }
    if (!preg_match('/[a-z]/', $new_password)) {
        throw new Exception('Password must contain at least one lowercase letter');
    }
    if (!preg_match('/[0-9]/', $new_password)) {
        throw new Exception('Password must contain at least one number');
    }
    if (!preg_match('/[!@#$%^&*(),.?":{}|<>]/', $new_password)) {
        throw new Exception('Password must contain at least one special character');
    }

    // Check if token is valid and not expired
    $stmt = $conn->prepare("
        SELECT pt.id, pt.vendor_id, pt.expires_at, v.email 
        FROM password_reset_tokens pt 
        JOIN vendors v ON pt.vendor_id = v.id 
        WHERE pt.token = ? 
        AND v.email = ? 
        AND pt.used = 0 
        AND pt.expires_at > NOW()
        LIMIT 1
    ");

    if (!$stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }

    $stmt->bind_param("ss", $token, $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        throw new Exception('Invalid or expired reset token');
    }

    $data = $result->fetch_assoc();
    $token_id = $data['id'];
    $vendor_id = $data['vendor_id'];
    $stmt->close();

    // Hash the new password
    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);

    // Update vendor password
    $update_stmt = $conn->prepare("UPDATE vendors SET password = ? WHERE id = ?");
    if (!$update_stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }

    $update_stmt->bind_param("si", $hashed_password, $vendor_id);
    
    if ($update_stmt->execute()) {
        // Mark token as used
        $token_stmt = $conn->prepare("UPDATE password_reset_tokens SET used = 1, used_at = NOW() WHERE id = ?");
        $token_stmt->bind_param("i", $token_id);
        $token_stmt->execute();
        $token_stmt->close();

        // Log the password reset
        $log_check = $conn->query("SHOW TABLES LIKE 'vendor_activity_log'");
        if ($log_check->num_rows > 0) {
            $log_stmt = $conn->prepare("INSERT INTO vendor_activity_log (vendor_id, activity_type, description) VALUES (?, 'password_reset', 'Vendor reset password via reset link')");
            $log_stmt->bind_param("i", $vendor_id);
            $log_stmt->execute();
            $log_stmt->close();
        }

        $response['success'] = true;
        $response['message'] = 'Password reset successful! You can now login with your new password.';
    } else {
        throw new Exception('Failed to update password');
    }

} catch (Exception $e) {
    error_log("Password reset error: " . $e->getMessage());
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
exit();
?>