<?php
session_start();
require '../config/conn.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $verification_code = trim($_POST['verification_code'] ?? '');
    
    // Check if 2FA session exists
    if (!isset($_SESSION['2fa_vendor_id'])) {
        echo json_encode([
            'success' => false,
            'message' => 'Session expired. Please login again.',
            'redirect' => 'vendor-login.php'
        ]);
        exit();
    }

    $vendor_id = $_SESSION['2fa_vendor_id'];
    $remember_me = $_SESSION['2fa_remember_me'] ?? 0;

    // Validate code
    if (empty($verification_code) || strlen($verification_code) !== 6) {
        echo json_encode([
            'success' => false,
            'message' => 'Please enter a valid 6-digit code.',
            'error_field' => 'code'
        ]);
        exit();
    }

    // Check verification code in database
    $stmt = $conn->prepare("
        SELECT vc.*, v.business_name, v.email 
        FROM verification_codes vc 
        JOIN vendors v ON vc.user_id = v.id 
        WHERE vc.user_id = ? AND vc.code = ? AND vc.user_type = 'vendor' 
        AND vc.expires_at > NOW() AND vc.used_at IS NULL
        ORDER BY vc.created_at DESC 
        LIMIT 1
    ");
    $stmt->bind_param("is", $vendor_id, $verification_code);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 1) {
        $verification_data = $result->fetch_assoc();
        
        // Mark code as used
        $update_stmt = $conn->prepare("UPDATE verification_codes SET used_at = NOW() WHERE id = ?");
        $update_stmt->bind_param("i", $verification_data['id']);
        $update_stmt->execute();

        // Get vendor data
        $vendor_stmt = $conn->prepare("SELECT id, vendor_id, business_name, email, is_approved FROM vendors WHERE id = ?");
        $vendor_stmt->bind_param("i", $vendor_id);
        $vendor_stmt->execute();
        $vendor_result = $vendor_stmt->get_result();
        $vendor = $vendor_result->fetch_assoc();

        // Set session variables
        $_SESSION['vendor_logged_in'] = true;
        $_SESSION['vendor_id'] = $vendor['id'];
        $_SESSION['vendor_email'] = $vendor['email'];
        $_SESSION['vendor_name'] = $vendor['business_name'];
        $_SESSION['vendor_status'] = $vendor['is_approved'];

        // Set remember me cookie if requested
        if ($remember_me) {
            $token = bin2hex(random_bytes(32));
            $expiry = time() + (30 * 24 * 60 * 60); // 30 days
            
            $token_stmt = $conn->prepare("UPDATE vendors SET remember_token = ?, token_expiry = ? WHERE id = ?");
            $token_stmt->bind_param("ssi", $token, date('Y-m-d H:i:s', $expiry), $vendor_id);
            $token_stmt->execute();
            
            setcookie('vendor_remember', $token, $expiry, '/', '', true, true);
        }

        // Clear 2FA session
        unset($_SESSION['2fa_vendor_id']);
        unset($_SESSION['2fa_remember_me']);
        unset($_SESSION['2fa_email']);
        unset($_SESSION['2fa_created_at']);

        echo json_encode([
            'success' => true,
            'message' => 'Verification successful! Logging you in...',
            'redirect' => 'index.php'
        ]);
        
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid or expired verification code.',
            'error_field' => 'code',
            'clear_inputs' => true
        ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method.'
    ]);
}
?>