<?php
ob_start();
// product.php - Vendor Product Management Page
include 'include/header.php';

// Verify session
if (!isset($_SESSION['vendor_id'])) {
    header("Location: login.php");
    exit();
}

require_once '../config/conn.php';

$vendor_id = $_SESSION['vendor_id'];
$message = '';
$error = '';

// Handle product addition
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_product'])) {
    // Get form data
    $name = trim($_POST['name']);
    $description = trim($_POST['description']);
    $price = floatval($_POST['price']);
    $stock_status = $_POST['stock_status']; // 'in_stock' or 'out_of_stock'
    
    // Validate inputs
    if (empty($name)) {
        $error = "Product name is required.";
    } elseif ($price <= 0) {
        $error = "Price must be greater than 0.";
    } elseif (!in_array($stock_status, ['in_stock', 'out_of_stock'])) {
        $error = "Invalid stock status selected.";
    } else {
        // Handle image upload
        $image_path = '';
        if (isset($_FILES['product_image']) && $_FILES['product_image']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = '../uploads/products/';
            
            // Create upload directory if it doesn't exist
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            // Generate unique filename
            $file_extension = strtolower(pathinfo($_FILES['product_image']['name'], PATHINFO_EXTENSION));
            $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
            
            if (in_array($file_extension, $allowed_extensions)) {
                $unique_name = uniqid() . '_' . time() . '.' . $file_extension;
                $target_file = $upload_dir . $unique_name;
                
                // Validate image size (max 5MB)
                if ($_FILES['product_image']['size'] <= 5 * 1024 * 1024) {
                    if (move_uploaded_file($_FILES['product_image']['tmp_name'], $target_file)) {
                        $image_path = 'uploads/products/' . $unique_name;
                    } else {
                        $error = "Failed to upload image. Please try again.";
                    }
                } else {
                    $error = "Image size must be less than 5MB.";
                }
            } else {
                $error = "Only JPG, JPEG, PNG, GIF, and WebP files are allowed.";
            }
        } elseif ($_FILES['product_image']['error'] !== UPLOAD_ERR_NO_FILE) {
            // Handle other upload errors
            switch ($_FILES['product_image']['error']) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $error = "Image file is too large.";
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $error = "Image was only partially uploaded.";
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                    $error = "Missing temporary folder.";
                    break;
                case UPLOAD_ERR_CANT_WRITE:
                    $error = "Failed to write image to disk.";
                    break;
                case UPLOAD_ERR_EXTENSION:
                    $error = "Image upload stopped by extension.";
                    break;
                default:
                    $error = "Unknown upload error.";
            }
        }
        
        if (empty($error)) {
            // Insert product into database
            $stmt = $conn->prepare("INSERT INTO products (vendor_id, name, description, price, stock_status, image_path, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())");
            
            if ($stmt) {
                $stmt->bind_param("issdss", $vendor_id, $name, $description, $price, $stock_status, $image_path);
                
                if ($stmt->execute()) {
                    $message = "Product added successfully!";
                    // Clear form values
                    $_POST = array();
                } else {
                    $error = "Error adding product: " . $stmt->error;
                }
                $stmt->close();
            } else {
                $error = "Database error: " . $conn->error;
            }
        }
    }
}

// Handle product deletion
if (isset($_GET['delete_id'])) {
    $delete_id = intval($_GET['delete_id']);
    
    // Verify the product belongs to this vendor
    $check_stmt = $conn->prepare("SELECT id, image_path FROM products WHERE id = ? AND vendor_id = ?");
    $check_stmt->bind_param("ii", $delete_id, $vendor_id);
    $check_stmt->execute();
    $result = $check_stmt->get_result();
    
    if ($result->num_rows > 0) {
        $product = $result->fetch_assoc();
        
        // Delete the product
        $delete_stmt = $conn->prepare("DELETE FROM products WHERE id = ? AND vendor_id = ?");
        $delete_stmt->bind_param("ii", $delete_id, $vendor_id);
        
        if ($delete_stmt->execute()) {
            // Delete associated image if exists
            if (!empty($product['image_path']) && file_exists('../' . $product['image_path'])) {
                unlink('../' . $product['image_path']);
            }
            $message = "Product deleted successfully!";
        } else {
            $error = "Error deleting product.";
        }
        $delete_stmt->close();
    } else {
        $error = "Product not found or you don't have permission to delete it.";
    }
    $check_stmt->close();
    
    // Redirect to remove delete_id from URL
    header("Location: product.php");
    exit();
}

// Handle stock status update
if (isset($_GET['toggle_stock'])) {
    $product_id = intval($_GET['toggle_stock']);
    
    // Verify the product belongs to this vendor
    $check_stmt = $conn->prepare("SELECT id, stock_status FROM products WHERE id = ? AND vendor_id = ?");
    $check_stmt->bind_param("ii", $product_id, $vendor_id);
    $check_stmt->execute();
    $result = $check_stmt->get_result();
    
    if ($result->num_rows > 0) {
        $product = $result->fetch_assoc();
        $new_stock_status = ($product['stock_status'] == 'in_stock') ? 'out_of_stock' : 'in_stock';
        
        $update_stmt = $conn->prepare("UPDATE products SET stock_status = ?, updated_at = NOW() WHERE id = ? AND vendor_id = ?");
        $update_stmt->bind_param("sii", $new_stock_status, $product_id, $vendor_id);
        
        if ($update_stmt->execute()) {
            $message = "Stock status updated successfully!";
        } else {
            $error = "Error updating stock status.";
        }
        $update_stmt->close();
    } else {
        $error = "Product not found or you don't have permission to update it.";
    }
    $check_stmt->close();
    
    // Redirect to remove toggle_stock from URL
    header("Location: product.php");
    exit();
}

// Fetch vendor's products with search and filter
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$stock_filter = isset($_GET['stock']) ? $_GET['stock'] : '';

// Build query conditions
$conditions = ["vendor_id = ?"];
$params = [$vendor_id];
$param_types = "i";

if (!empty($search)) {
    $conditions[] = "(name LIKE ? OR description LIKE ?)";
    $search_param = "%{$search}%";
    $params[] = $search_param;
    $params[] = $search_param;
    $param_types .= "ss";
}

if (!empty($stock_filter) && in_array($stock_filter, ['in_stock', 'out_of_stock'])) {
    $conditions[] = "stock_status = ?";
    $params[] = $stock_filter;
    $param_types .= "s";
}

$where_clause = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

// Get total products count
$count_query = "SELECT COUNT(*) as total FROM products {$where_clause}";
$count_stmt = $conn->prepare($count_query);
if ($count_stmt) {
    if (!empty($params)) {
        $count_stmt->bind_param($param_types, ...$params);
    }
    $count_stmt->execute();
    $count_result = $count_stmt->get_result();
    $total_products = $count_result->fetch_assoc()['total'];
    $total_pages = ceil($total_products / $limit);
    $count_stmt->close();
} else {
    $total_products = 0;
    $total_pages = 1;
}

// Fetch products with pagination
$products = [];
$query = "SELECT * FROM products {$where_clause} ORDER BY created_at DESC LIMIT ? OFFSET ?";

$product_stmt = $conn->prepare($query);
if ($product_stmt) {
    // Add limit and offset to params
    $params[] = $limit;
    $params[] = $offset;
    $param_types .= "ii";
    
    $product_stmt->bind_param($param_types, ...$params);
    $product_stmt->execute();
    $product_result = $product_stmt->get_result();
    
    while ($row = $product_result->fetch_assoc()) {
        $products[] = $row;
    }
    $product_stmt->close();
}
ob_end_flush();
?>

<main class="main-content flex-1 pb-8">
    <div class="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <!-- Success/Error Messages -->
        <?php if ($message): ?>
        <div class="mb-4 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
            <span class="block sm:inline"><?= htmlspecialchars($message) ?></span>
            <button type="button" class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
                <svg class="fill-current h-6 w-6 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
                    <title>Close</title>
                    <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
                </svg>
            </button>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="mb-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
            <span class="block sm:inline"><?= htmlspecialchars($error) ?></span>
            <button type="button" class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
                <svg class="fill-current h-6 w-6 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
                    <title>Close</title>
                    <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
                </svg>
            </button>
        </div>
        <?php endif; ?>

        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Left Column: Add Product Form -->
            <div class="lg:col-span-1">
                <div class="bg-white dark:bg-gray-800 shadow rounded-lg">
                    <div class="px-4 py-5 sm:px-6 border-b border-gray-200 dark:border-gray-700">
                        <h3 class="text-lg leading-6 font-medium text-gray-900 dark:text-white">Add New Product</h3>
                        <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Fill in the details to add a new product to your store.</p>
                    </div>
                    <div class="px-4 py-5 sm:p-6">
                        <form method="POST" action="" enctype="multipart/form-data" id="productForm">
                            <div class="space-y-4">
                                <div>
                                    <label for="name" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Product Name <span class="text-red-500">*</span>
                                    </label>
                                    <input type="text" name="name" id="name" required 
                                           value="<?= isset($_POST['name']) ? htmlspecialchars($_POST['name']) : '' ?>"
                                           class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Enter the name of your product</p>
                                </div>

                                <div>
                                    <label for="description" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Description</label>
                                    <textarea name="description" id="description" rows="3"
                                              class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white"><?= isset($_POST['description']) ? htmlspecialchars($_POST['description']) : '' ?></textarea>
                                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Describe your product features and details</p>
                                </div>

                                <div class="grid grid-cols-2 gap-4">
                                    <div>
                                        <label for="price" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Price (₦) <span class="text-red-500">*</span>
                                        </label>
                                        <input type="number" name="price" id="price" required step="0.01" min="0.01"
                                               value="<?= isset($_POST['price']) ? htmlspecialchars($_POST['price']) : '' ?>"
                                               class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                        <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Enter price in Naira</p>
                                    </div>

                                    <div>
                                        <label for="stock_status" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Stock Status <span class="text-red-500">*</span>
                                        </label>
                                        <select name="stock_status" id="stock_status" required
                                                class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                            <option value="">Select Status</option>
                                            <option value="in_stock" <?= (isset($_POST['stock_status']) && $_POST['stock_status'] == 'in_stock') ? 'selected' : '' ?>>In Stock</option>
                                            <option value="out_of_stock" <?= (isset($_POST['stock_status']) && $_POST['stock_status'] == 'out_of_stock') ? 'selected' : '' ?>>Out of Stock</option>
                                        </select>
                                    </div>
                                </div>

                                <div>
                                    <label for="product_image" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Product Image</label>
                                    <div class="mt-1 flex items-center">
                                        <input type="file" name="product_image" id="product_image" accept="image/*"
                                               class="block w-full text-sm text-gray-500 dark:text-gray-400
                                                      file:mr-4 file:py-2 file:px-4
                                                      file:rounded-md file:border-0
                                                      file:text-sm file:font-semibold
                                                      file:bg-primary-50 file:text-primary-700 dark:file:bg-primary-900 dark:file:text-primary-200
                                                      hover:file:bg-primary-100 dark:hover:file:bg-primary-800">
                                    </div>
                                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Max size: 5MB. Allowed: JPG, PNG, GIF, WebP</p>
                                    <div id="imagePreview" class="mt-2 hidden">
                                        <img id="previewImage" class="max-w-full h-48 object-cover rounded-lg shadow-md">
                                    </div>
                                </div>

                                <div class="pt-4">
                                    <button type="submit" name="add_product" class="w-full bg-primary-600 border border-transparent rounded-md shadow-sm py-3 px-4 inline-flex justify-center text-sm font-medium text-white hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 transition duration-150 ease-in-out">
                                        <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path>
                                        </svg>
                                        Add Product
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Right Column: Product List -->
            <div class="lg:col-span-2">
                <div class="bg-white dark:bg-gray-800 shadow rounded-lg">
                    <div class="px-4 py-5 sm:px-6 border-b border-gray-200 dark:border-gray-700">
                        <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-4">
                            <div>
                                <h3 class="text-lg leading-6 font-medium text-gray-900 dark:text-white">Your Products</h3>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Manage your product listings.</p>
                            </div>
                            <div class="flex items-center">
                                <span class="text-sm text-gray-500 dark:text-gray-400 mr-3">
                                    Total: <span class="font-semibold"><?= $total_products ?></span> products
                                </span>
                                <a href="product.php" class="bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-3 rounded-md text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                                    <svg class="w-4 h-4 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path>
                                    </svg>
                                    Refresh
                                </a>
                            </div>
                        </div>
                        
                        <!-- Search and Filter -->
                        <div class="mt-4">
                            <form method="GET" action="" class="flex flex-col sm:flex-row gap-3">
                                <div class="flex-1">
                                    <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" 
                                           placeholder="Search products..."
                                           class="w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                </div>
                                <div class="flex gap-3">
                                    <select name="stock" class="border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                        <option value="">All Stock</option>
                                        <option value="in_stock" <?= $stock_filter == 'in_stock' ? 'selected' : '' ?>>In Stock</option>
                                        <option value="out_of_stock" <?= $stock_filter == 'out_of_stock' ? 'selected' : '' ?>>Out of Stock</option>
                                    </select>
                                    
                                    <button type="submit" class="bg-primary-600 text-white py-2 px-4 rounded-md hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                                        Filter
                                    </button>
                                    
                                    <?php if ($search || $stock_filter): ?>
                                        <a href="product.php" class="bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-4 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                                            Clear
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <?php if (empty($products)): ?>
                        <div class="px-4 py-12 text-center">
                            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"/>
                            </svg>
                            <h3 class="mt-2 text-sm font-medium text-gray-900 dark:text-white">No products found</h3>
                            <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                <?= $search || $stock_filter ? 'Try changing your filters.' : 'Get started by adding your first product.' ?>
                            </p>
                        </div>
                    <?php else: ?>
                        <div class="overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead class="bg-gray-50 dark:bg-gray-900">
                                    <tr>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Product</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Price</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Stock</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Added</th>
                                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">Actions</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                    <?php foreach ($products as $product): ?>
                                        <tr class="hover:bg-gray-50 dark:hover:bg-gray-750">
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="flex items-center">
                                                    <div class="flex-shrink-0 h-12 w-12">
                                                        <?php if (!empty($product['image_path'])): ?>
                                                            <img class="h-12 w-12 rounded-lg object-cover" src="../<?= htmlspecialchars($product['image_path']) ?>" alt="<?= htmlspecialchars($product['name']) ?>">
                                                        <?php else: ?>
                                                            <div class="h-12 w-12 rounded-lg bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                                                                <svg class="h-6 w-6 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                                </svg>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="ml-4">
                                                        <div class="text-sm font-medium text-gray-900 dark:text-white">
                                                            <?= htmlspecialchars($product['name']) ?>
                                                        </div>
                                                        <div class="text-sm text-gray-500 dark:text-gray-400 truncate max-w-xs">
                                                            <?= htmlspecialchars(substr($product['description'], 0, 60)) ?><?= strlen($product['description']) > 60 ? '...' : '' ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <div class="text-sm font-semibold text-gray-900 dark:text-white">
                                                    ₦<?= number_format($product['price'], 2) ?>
                                                </div>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap">
                                                <?php 
                                                $stock_colors = [
                                                    'in_stock' => 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200',
                                                    'out_of_stock' => 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'
                                                ];
                                                $stock_text = [
                                                    'in_stock' => 'In Stock',
                                                    'out_of_stock' => 'Out of Stock'
                                                ];
                                                $stock_status = isset($product['stock_status']) ? $product['stock_status'] : 'in_stock';
                                                ?>
                                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?= $stock_colors[$stock_status] ?>">
                                                    <?= $stock_text[$stock_status] ?>
                                                </span>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                <?= date('M d, Y', strtotime($product['created_at'])) ?>
                                            </td>
                                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                                <div class="flex space-x-3">
                                                    <a href="edit-product.php?id=<?= $product['id'] ?>" 
                                                       class="text-primary-600 hover:text-primary-900 dark:text-primary-400 dark:hover:text-primary-300"
                                                       title="Edit Product">
                                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                                                        </svg>
                                                    </a>
                                                    <a href="product.php?toggle_stock=<?= $product['id'] ?>" 
                                                       class="text-blue-600 hover:text-blue-900 dark:text-blue-400 dark:hover:text-blue-300"
                                                       title="<?= (isset($product['stock_status']) && $product['stock_status'] == 'in_stock') ? 'Mark Out of Stock' : 'Mark In Stock' ?>"
                                                       onclick="return confirm('Are you sure you want to mark this product as <?= (isset($product['stock_status']) && $product['stock_status'] == 'in_stock') ? 'Out of Stock' : 'In Stock' ?>?')">
                                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4"></path>
                                                        </svg>
                                                    </a>
                                                    <a href="product.php?delete_id=<?= $product['id'] ?>" 
                                                       class="text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                                       title="Delete Product"
                                                       onclick="return confirm('Are you sure you want to delete this product? This action cannot be undone.')">
                                                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                                                        </svg>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Pagination -->
                        <?php if ($total_pages > 1): ?>
                            <div class="px-4 py-3 flex items-center justify-between border-t border-gray-200 dark:border-gray-700 sm:px-6">
                                <div class="flex-1 flex justify-between sm:hidden">
                                    <?php if ($page > 1): ?>
                                        <a href="product.php?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>" 
                                           class="relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700">
                                            Previous
                                        </a>
                                    <?php endif; ?>
                                    <?php if ($page < $total_pages): ?>
                                        <a href="product.php?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>" 
                                           class="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700">
                                            Next
                                        </a>
                                    <?php endif; ?>
                                </div>
                                <div class="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                    <div>
                                        <p class="text-sm text-gray-700 dark:text-gray-300">
                                            Showing <span class="font-medium"><?= ($offset + 1) ?></span> 
                                            to <span class="font-medium"><?= min($offset + $limit, $total_products) ?></span> 
                                            of <span class="font-medium"><?= $total_products ?></span> results
                                        </p>
                                    </div>
                                    <div>
                                        <nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                                            <?php if ($page > 1): ?>
                                                <a href="product.php?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>" 
                                                   class="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 text-sm font-medium text-gray-500 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700">
                                                    <span class="sr-only">Previous</span>
                                                    <svg class="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                        <path fill-rule="evenodd" d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z" clip-rule="evenodd" />
                                                    </svg>
                                                </a>
                                            <?php endif; ?>
                                            
                                            <?php 
                                            // Show page numbers
                                            $start_page = max(1, $page - 2);
                                            $end_page = min($total_pages, $page + 2);
                                            
                                            for ($i = $start_page; $i <= $end_page; $i++): 
                                            ?>
                                                <a href="product.php?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>" 
                                                   class="<?= $i == $page ? 'z-10 bg-primary-50 dark:bg-primary-900 border-primary-500 dark:border-primary-600 text-primary-600 dark:text-primary-300' : 'bg-white dark:bg-gray-800 border-gray-300 dark:border-gray-600 text-gray-500 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700' ?> relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                                                    <?= $i ?>
                                                </a>
                                            <?php endfor; ?>
                                            
                                            <?php if ($page < $total_pages): ?>
                                                <a href="product.php?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>" 
                                                   class="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-800 text-sm font-medium text-gray-500 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-700">
                                                    <span class="sr-only">Next</span>
                                                    <svg class="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                                        <path fill-rule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clip-rule="evenodd" />
                                                    </svg>
                                                </a>
                                            <?php endif; ?>
                                        </nav>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>

<script>
// Image preview functionality
document.getElementById('product_image').addEventListener('change', function(e) {
    const previewContainer = document.getElementById('imagePreview');
    const previewImage = document.getElementById('previewImage');
    
    if (this.files && this.files[0]) {
        const file = this.files[0];
        
        // Validate file size (5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('Image size must be less than 5MB.');
            this.value = '';
            previewContainer.classList.add('hidden');
            return;
        }
        
        // Validate file type
        const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!validTypes.includes(file.type)) {
            alert('Only JPG, JPEG, PNG, GIF, and WebP files are allowed.');
            this.value = '';
            previewContainer.classList.add('hidden');
            return;
        }
        
        const reader = new FileReader();
        
        reader.onload = function(e) {
            previewImage.src = e.target.result;
            previewContainer.classList.remove('hidden');
        }
        
        reader.onerror = function() {
            alert('Error reading image file.');
            previewContainer.classList.add('hidden');
        }
        
        reader.readAsDataURL(file);
    } else {
        previewContainer.classList.add('hidden');
    }
});

// Form validation
document.getElementById('productForm').addEventListener('submit', function(e) {
    const name = document.getElementById('name').value.trim();
    const price = document.getElementById('price').value;
    const stockStatus = document.getElementById('stock_status').value;
    
    // Clear previous error messages
    const errorElements = document.querySelectorAll('.error-message');
    errorElements.forEach(el => el.remove());
    
    let hasError = false;
    
    // Validate name
    if (!name) {
        showError('name', 'Product name is required.');
        hasError = true;
    }
    
    // Validate price
    if (!price || parseFloat(price) <= 0) {
        showError('price', 'Price must be greater than 0.');
        hasError = true;
    }
    
    // Validate stock status
    if (!stockStatus) {
        showError('stock_status', 'Stock status is required.');
        hasError = true;
    }
    
    if (hasError) {
        e.preventDefault();
        return false;
    }
    
    return true;
});

function showError(fieldId, message) {
    const field = document.getElementById(fieldId);
    const errorElement = document.createElement('p');
    errorElement.className = 'mt-1 text-sm text-red-600 dark:text-red-400 error-message';
    errorElement.textContent = message;
    
    // Insert error message after the field
    field.parentNode.appendChild(errorElement);
    
    // Highlight the field
    field.classList.add('border-red-500', 'dark:border-red-400');
    field.classList.remove('border-gray-300', 'dark:border-gray-600');
    
    // Remove error highlighting when field is focused
    field.addEventListener('input', function() {
        this.classList.remove('border-red-500', 'dark:border-red-400');
        this.classList.add('border-gray-300', 'dark:border-gray-600');
        const errorMsg = this.parentNode.querySelector('.error-message');
        if (errorMsg) {
            errorMsg.remove();
        }
    }, { once: true });
}

// Price formatting
document.getElementById('price').addEventListener('blur', function() {
    const value = parseFloat(this.value);
    if (!isNaN(value) && value > 0) {
        this.value = value.toFixed(2);
    }
});

// Confirm before leaving page if form has changes
let formChanged = false;
const form = document.getElementById('productForm');
const formInputs = form.querySelectorAll('input, textarea, select');

formInputs.forEach(input => {
    input.addEventListener('input', () => {
        formChanged = true;
    });
});

window.addEventListener('beforeunload', function(e) {
    if (formChanged) {
        e.preventDefault();
        e.returnValue = '';
    }
});

// Reset form changed flag on submit
form.addEventListener('submit', function() {
    formChanged = false;
});
</script>

<?php include 'include/footer.php'; ?>