<?php 
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['vendor_id'])) {
    session_destroy();
    header("Location: vendor-login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch vendor data with admin name
$vendor_id = $_SESSION['vendor_id'];

// Debug: Check the actual database structure
// DESCRIBE admins; - to see column names
// Based on your table structure, admins table has 'id' as primary key, not 'admin_id'

$stmt = $conn->prepare("
    SELECT v.*, 
           CASE 
               WHEN v.approved_by IS NOT NULL AND a.full_name IS NOT NULL 
               THEN a.full_name 
               ELSE 'System Admin' 
           END as approved_by_name
    FROM vendors v 
    LEFT JOIN admins a ON v.approved_by = a.id  -- Changed from a.admin_id to a.id
    WHERE v.id = ?
");
$stmt->bind_param("i", $vendor_id);
$stmt->execute();
$result = $stmt->get_result();
$vendor = $result->fetch_assoc();

// Debug output (remove in production)
/*
echo "<pre>";
echo "Vendor Data:\n";
print_r($vendor);
echo "\n\nDebug Info:";
echo "\nVendor ID: " . $vendor_id;
echo "\nApproved By Value: " . ($vendor['approved_by'] ?? 'NULL');
echo "\nApproved By Name: " . ($vendor['approved_by_name'] ?? 'NULL');
echo "\nIs Approved: " . ($vendor['is_approved'] ?? '0');
echo "</pre>";
*/

if (!$vendor) {
    // Handle case where vendor doesn't exist
    $_SESSION['error'] = "Vendor record not found";
    header("Location:index.php");
    exit();
}

// Helper functions
function format_date($date) {
    if (empty($date) || $date == '0000-00-00 00:00:00') return 'Never';
    return date('M j, Y g:i A', strtotime($date));
}

// Get login history
$login_stmt = $conn->prepare("SELECT * FROM login_history WHERE user_id = ? ORDER BY login_time DESC LIMIT 1");
$login_stmt->bind_param("i", $vendor_id);
$login_stmt->execute();
$login_result = $login_stmt->get_result();
$last_login = $login_result->fetch_assoc();

// Close statements
$stmt->close();
if ($login_stmt) $login_stmt->close();
$conn->close();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-3 sm:p-4 md:p-6 dark:bg-gray-900 overflow-y-auto mb-20">
    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success'])): ?>
    <div class="mb-3 bg-green-100 border border-green-400 text-green-700 px-3 py-2 rounded text-sm relative" role="alert">
        <span class="block sm:inline"><?= $_SESSION['success'] ?></span>
        <span class="absolute top-0 bottom-0 right-0 px-3 py-2" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-4 w-4 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
        </span>
    </div>
    <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
    <div class="mb-3 bg-red-100 border border-red-400 text-red-700 px-3 py-2 rounded text-sm relative" role="alert">
        <span class="block sm:inline"><?= $_SESSION['error'] ?></span>
        <span class="absolute top-0 bottom-0 right-0 px-3 py-2" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-4 w-4 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
        </span>
    </div>
    <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-4 sm:gap-6">
        <!-- Profile Card - Top on mobile, left on desktop -->
        <div class="dark:bg-gray-800 p-4 sm:p-6 border border-gray-100 dark:border-gray-700 rounded-lg lg:col-span-1 order-1 lg:order-none">
            <div class="flex flex-col items-center">
                <!-- Profile Picture -->
                <div class="relative mb-3">
                    <?php
                    $profile_pic = !empty($vendor['profile_picture']) ? htmlspecialchars($vendor['profile_picture']) : 'assets/images/default-avatar.png';
                    $profile_path = '../uploads/' . $profile_pic;
                    
                    // Check if file exists
                    if (!empty($vendor['profile_picture']) && !file_exists($profile_path)) {
                        $profile_pic = 'assets/images/default-avatar.png';
                    }
                    ?>
                    <img src="../uploads/<?= $profile_pic ?>" 
                        alt="Profile Picture" 
                        class="w-20 h-20 sm:w-24 sm:h-24 md:w-32 md:h-32 rounded-full object-cover border-3 border-primary-100 dark:border-primary-800"
                        onerror="this.src='../assets/images/default-avatar.png'">
                    <div class="mt-2">
                        <a href="photo_edit.php" class="text-center bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white px-3 py-1.5 rounded text-xs sm:text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 transition inline-block">
                            <i class="fas fa-camera mr-1 text-xs"></i> Update Photo
                        </a>
                    </div>
                </div>
                
                <!-- Vendor Info -->
                <h2 class="text-lg sm:text-xl font-bold dark:text-white text-center"><?= htmlspecialchars($vendor['business_name']) ?></h2>
                <p class="text-gray-500 dark:text-gray-400 text-xs sm:text-sm mb-3">
                    <?= !empty($vendor['business_type']) ? htmlspecialchars($vendor['business_type']) : 'Not specified' ?>
                </p>
                
                <!-- Vendor Details - Compact -->
                <div class="w-full space-y-2 mt-3 text-xs sm:text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">Vendor ID:</span>
                        <span class="font-medium dark:text-white truncate" title="<?= htmlspecialchars($vendor['vendor_id']) ?>"><?= htmlspecialchars($vendor['vendor_id']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">Email:</span>
                        <span class="font-medium dark:text-white truncate" title="<?= htmlspecialchars($vendor['email']) ?>"><?= htmlspecialchars($vendor['email']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">Phone:</span>
                        <span class="font-medium dark:text-white truncate"><?= !empty($vendor['phone']) ? htmlspecialchars($vendor['phone']) : 'Not set' ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">Last Login:</span>
                        <span class="font-medium dark:text-white truncate"><?= format_date($vendor['last_login']) ?></span>
                    </div>
                    <?php if ($last_login): ?>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">Device:</span>
                        <span class="font-medium dark:text-white truncate" title="<?= htmlspecialchars($last_login['device_info']) ?>"><?= substr(htmlspecialchars($last_login['device_info']), 0, 15) . (strlen($last_login['device_info']) > 15 ? '...' : '') ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400 truncate mr-2">IP:</span>
                        <span class="font-medium dark:text-white truncate" title="<?= htmlspecialchars($last_login['ip_address']) ?>"><?= htmlspecialchars($last_login['ip_address']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Action Buttons - Stacked on mobile -->
                <div class="w-full space-y-2 mt-4">
                    <a href="profile_edit.php" class="block text-center bg-primary-500 dark:bg-primary-600 text-white px-4 py-2 rounded text-xs sm:text-sm font-medium hover:bg-primary-600 transition">
                        <i class="fas fa-user-edit mr-1 text-xs"></i> Edit Profile
                    </a>
                    
                    <a href="password_change.php" class="block text-center bg-blue-500 dark:bg-blue-600 text-white px-4 py-2 rounded text-xs sm:text-sm font-medium hover:bg-blue-600 transition">
                        <i class="fas fa-lock mr-1 text-xs"></i> Change Password
                    </a>
                </div>
            </div>
        </div>

        <!-- Main Content Area -->
        <div class="dark:bg-gray-800 p-4 sm:p-6 border border-gray-100 dark:border-gray-700 rounded-lg lg:col-span-2 order-2">
            <!-- Business Information Section -->
            <div class="mb-6 sm:mb-8">
                <h3 class="text-base sm:text-lg font-semibold mb-3 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Business Info</h3>
                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Business Name</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate" title="<?= htmlspecialchars($vendor['business_name']) ?>">
                            <?= htmlspecialchars($vendor['business_name']) ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Business Type</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base">
                            <?= !empty($vendor['business_type']) ? htmlspecialchars($vendor['business_type']) : 'Not specified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Tax ID</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate">
                            <?= !empty($vendor['tax_id']) ? htmlspecialchars($vendor['tax_id']) : 'Not specified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Registered</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base">
                            <?= format_date($vendor['created_at']) ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Contact Information Section -->
            <div class="mb-6 sm:mb-8">
                <h3 class="text-base sm:text-lg font-semibold mb-3 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Contact Info</h3>
                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Contact Person</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate">
                            <?= !empty($vendor['contact_person']) ? htmlspecialchars($vendor['contact_person']) : 'Not specified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Email</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate" title="<?= htmlspecialchars($vendor['email']) ?>">
                            <?= htmlspecialchars($vendor['email']) ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Phone</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base">
                            <?= !empty($vendor['phone']) ? htmlspecialchars($vendor['phone']) : 'Not specified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Address</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate" title="<?= !empty($vendor['address']) ? htmlspecialchars($vendor['address']) : 'Not specified' ?>">
                            <?= !empty($vendor['address']) ? htmlspecialchars($vendor['address']) : 'Not specified' ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <!-- Account Status Section -->
            <div class="mb-0">
                <h3 class="text-base sm:text-lg font-semibold mb-3 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Account Status</h3>
                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Verification</label>
                        <p class="font-medium text-sm sm:text-base <?= $vendor['is_verified'] ? 'text-green-500' : 'text-yellow-500' ?>">
                            <?= $vendor['is_verified'] ? '✓ Verified' : '⚠ Not Verified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Approval</label>
                        <p class="font-medium text-sm sm:text-base <?= $vendor['is_approved'] ? 'text-green-500' : 'text-yellow-500' ?>">
                            <?= $vendor['is_approved'] ? '✓ Approved' : '⏳ Pending' ?>
                        </p>
                    </div>
                    <?php if ($vendor['is_approved'] && !empty($vendor['approved_by'])): ?>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Approved By</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base truncate" title="<?= htmlspecialchars($vendor['approved_by_name']) ?>">
                            <?= htmlspecialchars($vendor['approved_by_name']) ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-xs sm:text-sm text-gray-500 dark:text-gray-400 mb-1">Approval Date</label>
                        <p class="font-medium dark:text-white text-sm sm:text-base">
                            <?= format_date($vendor['approval_date']) ?>
                        </p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>

<?php include 'include/footer.php'; ?>