<?php
// vendor/vendor-transactions-live.php
include 'include/header.php';

if (!isset($_SESSION['vendor_id']) || empty($_SESSION['vendor_id'])) {
    header("Location: vendor-login.php");
    exit();
}

$vendor_id = $_SESSION['vendor_id'];

// Get all transactions
require_once '../config/conn.php';

// Get filter parameters
$filter_type = $_GET['type'] ?? 'all';
$filter_status = $_GET['status'] ?? 'all';
$filter_date = $_GET['date'] ?? '';

// Build query
$query = "SELECT * FROM vendor_wallet_transactions WHERE vendor_id = ?";
$params = [$vendor_id];
$types = "s";

if ($filter_type !== 'all') {
    $query .= " AND transaction_type = ?";
    $params[] = $filter_type;
    $types .= "s";
}

if ($filter_status !== 'all') {
    $query .= " AND status = ?";
    $params[] = $filter_status;
    $types .= "s";
}

if (!empty($filter_date)) {
    $query .= " AND DATE(created_at) = ?";
    $params[] = $filter_date;
    $types .= "s";
}

$query .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($query);
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$transactions = [];
while ($row = $result->fetch_assoc()) {
    $transactions[] = $row;
}

// Get summary stats
$stats_query = "SELECT 
    COUNT(*) as total_transactions,
    SUM(CASE WHEN transaction_type = 'credit' THEN amount ELSE 0 END) as total_credits,
    SUM(CASE WHEN transaction_type = 'debit' THEN amount ELSE 0 END) as total_debits
    FROM vendor_wallet_transactions 
    WHERE vendor_id = ?";
$stats_stmt = $conn->prepare($stats_query);
$stats_stmt->bind_param("s", $vendor_id);
$stats_stmt->execute();
$stats_result = $stats_stmt->get_result();
$stats = $stats_result->fetch_assoc();

mysqli_close($conn);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction History - Real-time</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50 dark:bg-gray-900">
    
    <div class="container mx-auto px-4 py-8">
        <!-- Page Header -->
        <div class="mb-8">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-3xl font-bold text-gray-900 dark:text-white">Transaction History</h1>
                    <p class="text-gray-600 dark:text-gray-400">Real-time updates of all transactions</p>
                </div>
                <div class="flex items-center space-x-2">
                    <div id="connection-status" class="flex items-center px-3 py-1 rounded-full bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200 text-sm">
                        <i class="fas fa-circle text-xs mr-2 animate-pulse"></i>
                        <span>Live Updates Active</span>
                    </div>
                    <a href="vendor-dashboard.php" class="bg-yellow-500 hover:bg-yellow-600 text-white py-2 px-4 rounded-lg transition duration-200">
                        <i class="fas fa-arrow-left mr-2"></i>Back to Dashboard
                    </a>
                </div>
            </div>
        </div>
        
        <!-- Stats Cards -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow p-6">
                <div class="flex items-center">
                    <div class="p-3 rounded-lg bg-blue-100 dark:bg-blue-900/30 mr-4">
                        <i class="fas fa-exchange-alt text-blue-600 dark:text-blue-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600 dark:text-gray-400">Total Transactions</p>
                        <p id="total-transactions" class="text-2xl font-bold text-gray-900 dark:text-white">
                            <?= $stats['total_transactions'] ?? 0 ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow p-6">
                <div class="flex items-center">
                    <div class="p-3 rounded-lg bg-green-100 dark:bg-green-900/30 mr-4">
                        <i class="fas fa-arrow-down text-green-600 dark:text-green-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600 dark:text-gray-400">Total Credits</p>
                        <p id="total-credits" class="text-2xl font-bold text-green-600 dark:text-green-400">
                            ₦<?= number_format($stats['total_credits'] ?? 0, 2) ?>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow p-6">
                <div class="flex items-center">
                    <div class="p-3 rounded-lg bg-red-100 dark:bg-red-900/30 mr-4">
                        <i class="fas fa-arrow-up text-red-600 dark:text-red-400 text-xl"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600 dark:text-gray-400">Total Debits</p>
                        <p id="total-debits" class="text-2xl font-bold text-red-600 dark:text-red-400">
                            ₦<?= number_format($stats['total_debits'] ?? 0, 2) ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Filters -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-lg p-6 mb-8">
            <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Transaction Type</label>
                    <select name="type" class="w-full border border-gray-300 dark:border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-yellow-500 dark:bg-gray-700 dark:text-white">
                        <option value="all" <?= $filter_type === 'all' ? 'selected' : '' ?>>All Types</option>
                        <option value="credit" <?= $filter_type === 'credit' ? 'selected' : '' ?>>Credits (Income)</option>
                        <option value="debit" <?= $filter_type === 'debit' ? 'selected' : '' ?>>Debits (Withdrawals)</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Status</label>
                    <select name="status" class="w-full border border-gray-300 dark:border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-yellow-500 dark:bg-gray-700 dark:text-white">
                        <option value="all" <?= $filter_status === 'all' ? 'selected' : '' ?>>All Status</option>
                        <option value="completed" <?= $filter_status === 'completed' ? 'selected' : '' ?>>Completed</option>
                        <option value="pending" <?= $filter_status === 'pending' ? 'selected' : '' ?>>Pending</option>
                        <option value="failed" <?= $filter_status === 'failed' ? 'selected' : '' ?>>Failed</option>
                    </select>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Date</label>
                    <input type="date" name="date" value="<?= htmlspecialchars($filter_date) ?>"
                           class="w-full border border-gray-300 dark:border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-yellow-500 dark:bg-gray-700 dark:text-white">
                </div>
                
                <div class="flex items-end">
                    <button type="submit" class="w-full bg-yellow-500 hover:bg-yellow-600 text-white py-2 px-4 rounded-lg transition duration-200">
                        <i class="fas fa-filter mr-2"></i>Apply Filters
                    </button>
                    <a href="vendor-transactions-live.php" class="ml-2 bg-gray-200 dark:bg-gray-700 hover:bg-gray-300 dark:hover:bg-gray-600 text-gray-800 dark:text-gray-200 py-2 px-4 rounded-lg transition duration-200">
                        <i class="fas fa-redo"></i>
                    </a>
                </div>
            </form>
        </div>
        
        <!-- Transactions Table -->
        <div class="bg-white dark:bg-gray-800 rounded-xl shadow-lg overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                <h2 class="text-lg font-semibold text-gray-900 dark:text-white">
                    All Transactions <span class="text-sm text-gray-500">(Live Updates)</span>
                </h2>
            </div>
            
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                    <thead class="bg-gray-50 dark:bg-gray-700/50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Date & Time</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Type</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Description</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Reference</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Amount</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Status</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-400 uppercase">Balance After</th>
                        </tr>
                    </thead>
                    <tbody id="transactions-body" class="divide-y divide-gray-200 dark:divide-gray-700">
                        <?php if (empty($transactions)): ?>
                            <tr>
                                <td colspan="7" class="px-6 py-12 text-center text-gray-500 dark:text-gray-400">
                                    <i class="fas fa-exchange-alt text-4xl mb-3 opacity-50"></i>
                                    <p class="text-lg">No transactions found</p>
                                    <p class="text-sm mt-1">New transactions will appear here automatically</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($transactions as $txn): ?>
                            <tr id="txn-<?= $txn['id'] ?>" class="hover:bg-gray-50 dark:hover:bg-gray-700/50">
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                    <?= date('M j, Y', strtotime($txn['created_at'])) ?><br>
                                    <span class="text-gray-500 text-xs"><?= date('g:i A', strtotime($txn['created_at'])) ?></span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                        <?= $txn['transaction_type'] === 'credit' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 
                                           'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200' ?>">
                                        <i class="fas fa-<?= $txn['transaction_type'] === 'credit' ? 'arrow-down' : 'arrow-up' ?> mr-1 text-xs"></i>
                                        <?= ucfirst($txn['transaction_type']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-900 dark:text-white">
                                    <?= htmlspecialchars($txn['description']) ?>
                                    <?php if (!empty($txn['notes'])): ?>
                                        <p class="text-xs text-gray-500 mt-1"><?= htmlspecialchars($txn['notes']) ?></p>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300 font-mono">
                                    <?= $txn['reference'] ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium 
                                    <?= $txn['transaction_type'] === 'credit' ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400' ?>">
                                    <?= $txn['transaction_type'] === 'credit' ? '+' : '-' ?>₦<?= number_format($txn['amount'], 2) ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <span class="px-2.5 py-1 text-xs font-medium rounded-full 
                                        <?= $txn['status'] === 'completed' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 
                                           ($txn['status'] === 'failed' ? 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200' : 
                                           'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200') ?>">
                                        <?= ucfirst($txn['status']) ?>
                                    </span>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                                    ₦<?= number_format($txn['balance_after'] ?? 0, 2) ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <div class="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                <div class="flex justify-between items-center">
                    <div class="text-sm text-gray-500 dark:text-gray-400">
                        Showing <span id="transaction-count"><?= count($transactions) ?></span> transactions
                    </div>
                    <div class="text-sm text-gray-500 dark:text-gray-400 flex items-center">
                        <i class="fas fa-sync-alt animate-spin mr-2 text-yellow-500"></i>
                        <span id="last-update-time">Auto-updating...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Toast Notifications -->
    <div id="toast-container" class="fixed bottom-4 right-4 z-50 space-y-3"></div>
    
    <script>
    // Real-time updates for transactions page
    let eventSource = null;
    let lastUpdateTime = Date.now();
    let transactionCount = <?= count($transactions) ?>;
    let reconnectAttempts = 0;
    const maxReconnectAttempts = 10;
    
    // Function to get the correct base URL for SSE
    function getBaseUrl() {
        // For live server - use absolute URL
        const protocol = window.location.protocol;
        const host = window.location.host;
        
        // Get current path and go up one level if in vendor directory
        const currentPath = window.location.pathname;
        const pathParts = currentPath.split('/').filter(part => part);
        
        // Remove the current file from path
        if (pathParts.length > 0 && pathParts[pathParts.length - 1].includes('.php')) {
            pathParts.pop();
        }
        
        // Construct base path
        const basePath = pathParts.length > 0 ? '/' + pathParts.join('/') : '';
        
        return `${protocol}//${host}${basePath}`;
    }
    
    // Start real-time updates
    function startTransactionUpdates() {
        const vendorId = '<?= $vendor_id ?>';
        
        if (eventSource) {
            eventSource.close();
        }
        
        // Get the correct URL for both localhost and live server
        const baseUrl = getBaseUrl();
        const sseUrl = `${baseUrl}/api/sse-vendor-updates.php?vendor_id=${vendorId}`;
        
        console.log('Connecting to SSE:', sseUrl);
        
        eventSource = new EventSource(sseUrl);
        
        eventSource.onopen = function() {
            console.log('SSE Connection opened successfully');
            reconnectAttempts = 0;
            updateConnectionStatus('Connected', 'green');
        };
        
        eventSource.onmessage = function(event) {
            try {
                // Skip heartbeats
                if (event.data.trim() === ': heartbeat' || event.data.startsWith(':')) {
                    return;
                }
                
                const data = JSON.parse(event.data);
                lastUpdateTime = Date.now();
                
                // Update last update time
                const now = new Date();
                document.getElementById('last-update-time').textContent = 
                    `Last updated: ${now.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit', second:'2-digit'})}`;
                
                // Handle new transactions
                if (data.new_transactions && data.new_transactions.length > 0) {
                    data.new_transactions.forEach(transaction => {
                        handleNewTransaction(transaction);
                    });
                }
                
            } catch (error) {
                console.error('Error parsing SSE data:', error);
            }
        };
        
        eventSource.onerror = function(error) {
            console.error('SSE Error occurred:', error);
            updateConnectionStatus('Disconnected - Reconnecting...', 'red');
            
            // Close current connection
            if (eventSource) {
                eventSource.close();
            }
            
            // Attempt to reconnect
            if (reconnectAttempts < maxReconnectAttempts) {
                reconnectAttempts++;
                const delay = Math.min(3000 * reconnectAttempts, 15000); // Exponential backoff max 15 seconds
                
                console.log(`Reconnecting in ${delay/1000} seconds... Attempt ${reconnectAttempts}/${maxReconnectAttempts}`);
                
                setTimeout(() => {
                    startTransactionUpdates();
                }, delay);
            } else {
                updateConnectionStatus('Connection Failed - Refresh page', 'red');
                console.error('Maximum reconnection attempts reached');
                
                // Show error message to user
                showToast('Connection lost. Please refresh the page to reconnect.', 'error');
            }
        };
    }
    
    // Handle new transaction in table
    function handleNewTransaction(transaction) {
        // Check if transaction already exists
        const existingRow = document.getElementById(`txn-${transaction.id}`);
        if (existingRow) {
            return; // Already exists
        }
        
        // Update transaction count
        transactionCount++;
        document.getElementById('transaction-count').textContent = transactionCount;
        
        // Update stats
        updateTransactionStats(transaction);
        
        // Create new table row
        const tbody = document.getElementById('transactions-body');
        if (!tbody) return;
        
        // Remove placeholder if exists
        const placeholder = tbody.querySelector('.text-center');
        if (placeholder) {
            placeholder.remove();
        }
        
        // Format date/time
        const date = new Date(transaction.created_at);
        const formattedDate = date.toLocaleDateString('en-US', {month: 'short', day: 'numeric', year: 'numeric'});
        const formattedTime = date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
        
        // Create row
        const row = document.createElement('tr');
        row.id = `txn-${transaction.id}`;
        row.className = 'hover:bg-gray-50 dark:hover:bg-gray-700/50 animate-pulse bg-green-50/30 dark:bg-green-900/10';
        setTimeout(() => {
            row.classList.remove('animate-pulse');
            row.classList.remove('bg-green-50/30', 'dark:bg-green-900/10');
        }, 2000);
        
        row.innerHTML = `
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                ${formattedDate}<br>
                <span class="text-gray-500 text-xs">${formattedTime}</span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                    ${transaction.type === 'credit' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 
                       'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'}">
                    <i class="fas fa-${transaction.type === 'credit' ? 'arrow-down' : 'arrow-up'} mr-1 text-xs"></i>
                    ${transaction.type.charAt(0).toUpperCase() + transaction.type.slice(1)}
                </span>
            </td>
            <td class="px-6 py-4 text-sm text-gray-900 dark:text-white">
                ${transaction.description || 'New transaction'}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300 font-mono">
                ${transaction.reference || 'N/A'}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium 
                ${transaction.type === 'credit' ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}">
                ${transaction.type === 'credit' ? '+' : '-'}₦${parseFloat(transaction.amount).toFixed(2)}
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="px-2.5 py-1 text-xs font-medium rounded-full 
                    ${transaction.status === 'completed' ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' : 
                       (transaction.status === 'failed' ? 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200' : 
                       'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200')}">
                    ${transaction.status.charAt(0).toUpperCase() + transaction.status.slice(1)}
                </span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-700 dark:text-gray-300">
                ₦${(transaction.balance_after || 0).toFixed(2)}
            </td>
        `;
        
        // Insert at the top
        tbody.insertBefore(row, tbody.firstChild);
        
        // Show notification
        showTransactionNotification(transaction);
    }
    
    // Update transaction stats
    function updateTransactionStats(transaction) {
        if (transaction.type === 'credit') {
            const creditsElem = document.getElementById('total-credits');
            const current = parseFloat(creditsElem.textContent.replace('₦', '').replace(/,/g, ''));
            const newAmount = current + parseFloat(transaction.amount);
            creditsElem.textContent = `₦${newAmount.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
        } else if (transaction.type === 'debit') {
            const debitsElem = document.getElementById('total-debits');
            const current = parseFloat(debitsElem.textContent.replace('₦', '').replace(/,/g, ''));
            const newAmount = current + parseFloat(transaction.amount);
            debitsElem.textContent = `₦${newAmount.toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
        }
        
        // Update total count
        const totalElem = document.getElementById('total-transactions');
        const currentTotal = parseInt(totalElem.textContent);
        totalElem.textContent = currentTotal + 1;
    }
    
    // Show transaction notification
    function showTransactionNotification(transaction) {
        const container = document.getElementById('toast-container');
        
        const toast = document.createElement('div');
        toast.className = 'bg-white dark:bg-gray-800 shadow-xl rounded-lg p-4 border-l-4 border-green-500 transform transition-all duration-300 translate-x-full w-80';
        
        toast.innerHTML = `
            <div class="flex items-start">
                <div class="flex-shrink-0 pt-0.5">
                    <i class="fas fa-${transaction.type === 'credit' ? 'money-bill-wave text-green-500' : 'arrow-up text-red-500'} text-lg"></i>
                </div>
                <div class="ml-3 flex-1">
                    <p class="font-medium text-gray-900 dark:text-white">
                        New ${transaction.type === 'credit' ? 'Payment' : 'Withdrawal'}!
                    </p>
                    <p class="text-sm text-gray-600 dark:text-gray-400 mt-1 truncate">
                        ${transaction.description || 'Transaction completed'}
                    </p>
                    <div class="flex justify-between items-center mt-2">
                        <p class="text-sm font-bold ${transaction.type === 'credit' ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'}">
                            ${transaction.type === 'credit' ? '+' : '-'}₦${parseFloat(transaction.amount).toFixed(2)}
                        </p>
                        <p class="text-xs text-gray-500">Just now</p>
                    </div>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        container.appendChild(toast);
        
        // Animate in
        setTimeout(() => {
            toast.classList.remove('translate-x-full');
        }, 10);
        
        // Auto remove
        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.remove();
                }
            }, 300);
        }, 5000);
    }
    
    // Show error toast
    function showToast(message, type = 'error') {
        const container = document.getElementById('toast-container');
        const toast = document.createElement('div');
        
        const bgColor = type === 'error' ? 'border-red-500' : 'border-yellow-500';
        const icon = type === 'error' ? 'exclamation-circle' : 'exclamation-triangle';
        const iconColor = type === 'error' ? 'text-red-500' : 'text-yellow-500';
        
        toast.className = `bg-white dark:bg-gray-800 shadow-xl rounded-lg p-4 border-l-4 ${bgColor} transform transition-all duration-300 translate-x-full w-80`;
        
        toast.innerHTML = `
            <div class="flex items-start">
                <div class="flex-shrink-0 pt-0.5">
                    <i class="fas fa-${icon} ${iconColor} text-lg"></i>
                </div>
                <div class="ml-3 flex-1">
                    <p class="font-medium text-gray-900 dark:text-white">
                        ${type === 'error' ? 'Connection Error' : 'Warning'}
                    </p>
                    <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        ${message}
                    </p>
                </div>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-4 text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        container.appendChild(toast);
        
        setTimeout(() => {
            toast.classList.remove('translate-x-full');
        }, 10);
        
        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.remove();
                }
            }, 300);
        }, 7000);
    }
    
    // Update connection status
    function updateConnectionStatus(message, color) {
        const statusElem = document.getElementById('connection-status');
        const colors = {
            green: 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200',
            yellow: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200',
            red: 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'
        };
        
        statusElem.className = `flex items-center px-3 py-1 rounded-full text-sm ${colors[color] || colors.green}`;
        statusElem.innerHTML = `<i class="fas fa-circle text-xs mr-2 ${color === 'green' ? 'animate-pulse' : ''}"></i><span>${message}</span>`;
    }
    
    // Initialize on page load
    document.addEventListener('DOMContentLoaded', function() {
        // Start SSE connection
        startTransactionUpdates();
        
        // Monitor connection status
        setInterval(() => {
            if (Date.now() - lastUpdateTime > 30000) { // 30 seconds
                updateConnectionStatus('Connection slow - Monitoring...', 'yellow');
            }
        }, 10000);
        
        // Clean up on page unload
        window.addEventListener('beforeunload', () => {
            if (eventSource) {
                eventSource.close();
            }
        });
        
        // Reconnect button (optional)
        document.getElementById('connection-status').addEventListener('click', function() {
            if (eventSource && eventSource.readyState !== EventSource.OPEN) {
                startTransactionUpdates();
            }
        });
    });
    </script>
    
    <?php include 'include/footer.php'; ?>
</body>
</html>