<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
require '../config/conn.php';

if (isset($_SESSION['vendor_logged_in']) && $_SESSION['vendor_logged_in'] === true) {
    header("Location: vendor/index.php");
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Campus Services - Vendor Login</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Poppins', sans-serif; }
    </style>
</head>
<body class="bg-gray-100 flex items-center justify-center p-4">
    <div class="w-full max-w-sm">
        <!-- Logo and Header -->
        <div class="text-center mb-6">
            <div class="w-16 h-16 mx-auto mb-3 flex items-center justify-center">
                <img src="../img/schools.png" alt="School Logo" class="max-w-full max-h-full">
            </div>
            <h1 class="text-xl font-bold text-gray-800">Campus Services</h1>
            <p class="text-gray-500 text-sm mt-1">Vendor Login</p>
        </div>

        <!-- Messages -->
        <div id="messageContainer" class="mb-4"></div>

        <?php if (isset($_SESSION['verification_success'])): ?>
            <div class="bg-green-50 border border-green-200 p-3 mb-4 text-green-700 text-sm rounded-lg">
                <div class="flex items-center">
                    <i class="fas fa-check-circle mr-2"></i>
                    <?php echo htmlspecialchars($_SESSION['verification_success']); ?>
                </div>
                <button onclick="location.href='vendor_login.php'" 
                        class="mt-2 bg-green-600 text-white px-3 py-1.5 text-xs rounded font-medium hover:bg-green-700">
                    Proceed to Login
                </button>
            </div>
            <?php unset($_SESSION['verification_success']); ?>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['verification_error'])): ?>
            <div class="bg-red-50 border border-red-200 p-3 mb-4 text-red-700 text-sm rounded-lg">
                <?php echo htmlspecialchars($_SESSION['verification_error']); ?>
                <?php if (isset($_SESSION['pending_verification_email'])): ?>
                    <div class="mt-2">
                        <button type="button" onclick="resendVerification('<?php echo $_SESSION['pending_verification_email']; ?>')" 
                                class="text-[#fbbb01] underline hover:text-[#e0a800] text-xs">
                            Resend verification code
                        </button>
                    </div>
                <?php endif; ?>
            </div>
            <?php 
            unset($_SESSION['verification_error']);
            unset($_SESSION['pending_verification_email']);
            ?>
        <?php endif; ?>

        <!-- Login Form -->
        <div class="bg-white p-6">
            <form id="loginForm" method="POST">
                <!-- Email -->
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-medium mb-1">Email</label>
                    <div class="relative">
                        <i class="fas fa-envelope absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm"></i>
                        <input type="email" name="email" 
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                               class="w-full pl-9 pr-3 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                               placeholder="vendor@business.com"
                               required>
                    </div>
                    <small class="text-red-500 text-xs mt-1 block" id="email_error"></small>
                </div>
                
                <!-- Password -->
                <div class="mb-5">
                    <label class="block text-gray-700 text-sm font-medium mb-1">Password</label>
                    <div class="relative">
                        <i class="fas fa-lock absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm"></i>
                        <input type="password" id="password" name="password" 
                               class="w-full pl-9 pr-9 py-2 text-sm border border-gray-300 rounded-lg focus:outline-none focus:ring-1 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                               placeholder="••••••••"
                               required>
                        <i class="fas fa-eye absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 cursor-pointer text-sm" 
                           onclick="togglePassword('password', this)"></i>
                    </div>
                    <small class="text-red-500 text-xs mt-1 block" id="password_error"></small>
                </div>
                
                <!-- Remember me & Forgot password -->
                <div class="flex justify-between items-center mb-5 text-xs">
                    <div class="flex items-center">
                        <input type="checkbox" id="rememberMe" name="remember_me" value="1" class="mr-1.5 w-3.5 h-3.5">
                        <label for="rememberMe" class="text-gray-600">Remember me</label>
                    </div>
                    <a href="vendor_forgot_password.php" class="text-[#fbbb01] hover:text-[#e0a800] hover:underline">Forgot password?</a>
                </div>
                
                <!-- Submit Button -->
                <button type="submit" id="submitBtn" 
                        class="w-full py-2.5 bg-[#fbbb01] text-white text-sm font-medium rounded-lg hover:bg-[#e0a800] transition disabled:opacity-70 disabled:cursor-not-allowed">
                    <span id="btnText">Login</span>
                    <span id="loadingSpinner" class="hidden ml-2">
                        <i class="fas fa-spinner fa-spin"></i>
                    </span>
                </button>
                
                <!-- Register Link -->
                <div class="text-center mt-4">
                    <p class="text-gray-500 text-xs">
                        Don't have an account? 
                        <a href="vendor_registration.php" class="text-[#fbbb01] hover:text-[#e0a800] font-medium">Register</a>
                    </p>
                </div>
            </form>
        </div>
    </div>

    <script>
    function togglePassword(inputId, icon) {
        const input = document.getElementById(inputId);
        if (input.type === 'password') {
            input.type = 'text';
            icon.classList.replace('fa-eye', 'fa-eye-slash');
        } else {
            input.type = 'password';
            icon.classList.replace('fa-eye-slash', 'fa-eye');
        }
    }

    async function resendVerification(email) {
        if (!email) return;
        
        try {
            const formData = new FormData();
            formData.append('email', email);
            formData.append('resend_verification', '1');
            
            showMessage('Sending verification code...', 'info');
            
            const response = await fetch('vendor_resend_verification.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success) {
                showMessage('Verification code sent!', 'success');
            } else {
                showMessage(data.message || 'Failed to send code.', 'error');
            }
        } catch (error) {
            showMessage('Network error. Try again.', 'error');
        }
    }

    function showMessage(message, type) {
        const container = document.getElementById('messageContainer');
        const colors = {
            success: 'bg-green-50 text-green-700 border border-green-200',
            error: 'bg-red-50 text-red-700 border border-red-200',
            info: 'bg-blue-50 text-blue-700 border border-blue-200'
        };
        
        const div = document.createElement('div');
        div.className = `p-3 rounded-lg text-sm mb-3 ${colors[type] || colors.info}`;
        div.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
                <span>${message}</span>
            </div>
        `;
        
        container.prepend(div);
        setTimeout(() => div.remove(), 4000);
    }

    document.getElementById('loginForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const btn = document.getElementById('submitBtn');
        const btnText = document.getElementById('btnText');
        const spinner = document.getElementById('loadingSpinner');
        
        btn.disabled = true;
        btnText.textContent = 'Logging in...';
        spinner.classList.remove('hidden');
        
        // Clear messages
        document.getElementById('messageContainer').innerHTML = '';
        document.getElementById('email_error').textContent = '';
        document.getElementById('password_error').textContent = '';
        
        try {
            const formData = new FormData(this);
            const response = await fetch('process_vendor_login.php', {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            btn.disabled = false;
            btnText.textContent = 'Login';
            spinner.classList.add('hidden');
            
            if (data.success) {
                showMessage(data.message, 'success');
                
                if (data.redirect) {
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 800);
                } else if (data.verification_required) {
                    setTimeout(() => {
                        window.location.href = 'vendor_verify_code.php';
                    }, 800);
                }
            } else {
                if (data.message) showMessage(data.message, 'error');
                
                if (data.errors) {
                    if (data.errors.email) {
                        document.getElementById('email_error').textContent = data.errors.email;
                    }
                    if (data.errors.password) {
                        document.getElementById('password_error').textContent = data.errors.password;
                    }
                }
                
                if (data.resend_verification && data.email) {
                    const div = document.createElement('div');
                    div.className = 'bg-yellow-50 border border-yellow-200 p-3 mt-3 text-yellow-700 text-sm rounded-lg';
                    div.innerHTML = `
                        <p class="mb-1">Account not verified.</p>
                        <button type="button" onclick="resendVerification('${data.email}')" 
                                class="text-[#fbbb01] underline hover:text-[#e0a800] text-xs font-medium">
                            Resend verification code
                        </button>
                    `;
                    document.getElementById('messageContainer').appendChild(div);
                }
                
                if (data.approval_pending) {
                    const div = document.createElement('div');
                    div.className = 'bg-blue-50 border border-blue-200 p-3 mt-3 text-blue-700 text-sm rounded-lg';
                    div.innerHTML = `
                        <p class="mb-1">${data.message || 'Your account is pending approval.'}</p>
                        <p class="text-xs">Please wait for administrator approval.</p>
                    `;
                    document.getElementById('messageContainer').appendChild(div);
                }
            }
        } catch (error) {
            btn.disabled = false;
            btnText.textContent = 'Login';
            spinner.classList.add('hidden');
            showMessage('Network error. Try again.', 'error');
        }
    });

    document.addEventListener('DOMContentLoaded', function() {
        document.querySelector('input[name="email"]').focus();
    });
    </script>
</body>
</html>