<?php
include 'include/header.php';
require_once '../config/conn.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'vendor') {
    header("Location: login.php");
    exit();
}

$vendor_id = $_SESSION['user_id'];

// Check if user is a vendor
$vendor_stmt = $conn->prepare("SELECT * FROM vendors WHERE user_id = ?");
$vendor_stmt->bind_param("i", $vendor_id);
$vendor_stmt->execute();
$vendor_result = $vendor_stmt->get_result();
$vendor = $vendor_result->fetch_assoc();

if (!$vendor) {
    $_SESSION['error'] = "Vendor account not found. Please complete vendor registration.";
    header("Location: vendor_register.php");
    exit();
}

// Get vendor's bank accounts
$account_stmt = $conn->prepare("
    SELECT * FROM vendor_bank_accounts
    WHERE vendor_id = ?
    ORDER BY is_default DESC, created_at DESC
");
$account_stmt->bind_param("i", $vendor['id']);
$account_stmt->execute();
$account_result = $account_stmt->get_result();
$bank_accounts = $account_result->fetch_all(MYSQLI_ASSOC);

// Check Paystack configuration
$paystack_configured = !empty(getenv('PAYSTACK_SECRET_KEY')) || file_exists('../config/paystack.php');
?>

<div class="container mx-auto px-4 py-8" style="margin-bottom:80px">
    <!-- Header -->
    <div class="flex justify-between items-center mb-8">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">Bank Accounts</h1>
            <p class="text-gray-600 mt-1">Manage your bank accounts for receiving payments</p>
        </div>
        <button onclick="openAddAccountModal()"
                class="bg-primary-500 hover:bg-primary-600 text-white px-4 py-2 rounded-lg text-sm font-medium flex items-center">
            <i class="fas fa-plus mr-2"></i> Add Bank Account
        </button>
    </div>

    <!-- Messages Container -->
    <div id="messageContainer" class="mb-6"></div>

    <!-- Paystack Warning -->
    <?php if (!$paystack_configured): ?>
        <div class="bg-red-50 border-l-4 border-red-400 p-4 mb-6 rounded">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-exclamation-triangle text-red-400"></i>
                </div>
                <div class="ml-3">
                    <p class="text-sm text-red-700">
                        <strong>Paystack not configured:</strong> Bank account verification is not available. Please contact administrator.
                    </p>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <!-- Bank Accounts Grid -->
    <?php if (empty($bank_accounts)): ?>
        <div class="text-center py-12 bg-gray-50 rounded-lg">
            <i class="fas fa-university text-4xl text-gray-400 mb-4"></i>
            <h3 class="text-lg font-medium text-gray-900 mb-2">No bank accounts added</h3>
            <p class="text-gray-500 mb-6">Add your first bank account to start receiving payments</p>
            <button onclick="openAddAccountModal()"
                    class="bg-primary-500 hover:bg-primary-600 text-white px-4 py-2 rounded-lg text-sm font-medium">
                <i class="fas fa-plus mr-1"></i> Add Bank Account
            </button>
        </div>
    <?php else: ?>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <?php foreach ($bank_accounts as $account): ?>
                <div class="bg-white rounded-lg border border-gray-200 p-6 hover:shadow-md transition-shadow">
                    <div class="flex justify-between items-start mb-4">
                        <div class="flex items-center">
                            <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center mr-3">
                                <i class="fas fa-university text-blue-500"></i>
                            </div>
                            <div>
                                <h3 class="font-semibold text-gray-800"><?= htmlspecialchars($account['bank_name']) ?></h3>
                                <p class="text-sm text-gray-500"><?= htmlspecialchars($account['account_number']) ?></p>
                            </div>
                        </div>
                        <?php if ($account['is_default']): ?>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                Default
                            </span>
                        <?php endif; ?>
                    </div>
                    
                    <div class="space-y-2 mb-6">
                        <div>
                            <p class="text-sm text-gray-500">Account Name</p>
                            <p class="font-medium"><?= htmlspecialchars($account['account_name']) ?></p>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Status</p>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                <?= $account['is_verified'] == 1 ? 'bg-green-100 text-green-800' : 
                                   ($account['is_verified'] == 0 ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800') ?>">
                                <i class="fas fa-<?= $account['is_verified'] == 1 ? 'check-circle' : 
                                                   ($account['is_verified'] == 0 ? 'clock' : 'times-circle') ?> mr-1"></i>
                                <?= $account['is_verified'] == 1 ? 'Verified' : 
                                   ($account['is_verified'] == 0 ? 'Pending' : 'Failed') ?>
                            </span>
                        </div>
                        <div>
                            <p class="text-sm text-gray-500">Added On</p>
                            <p class="text-sm"><?= date('M d, Y', strtotime($account['created_at'])) ?></p>
                        </div>
                    </div>
                    
                    <div class="flex space-x-3 pt-4 border-t border-gray-100">
                        <?php if (!$account['is_default']): ?>
                            <button onclick="setDefaultAccount(<?= $account['id'] ?>)"
                                    class="text-primary-600 hover:text-primary-700 text-sm font-medium">
                                Set as Default
                            </button>
                        <?php endif; ?>
                        <button onclick="deleteBankAccount(<?= $account['id'] ?>)"
                                class="text-red-600 hover:text-red-700 text-sm font-medium ml-auto">
                            Delete
                        </button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <!-- Information Panel -->
    <div class="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h3 class="text-lg font-semibold text-blue-800 mb-4 flex items-center">
            <i class="fas fa-info-circle mr-2"></i> Important Information
        </h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div class="flex items-start">
                <i class="fas fa-check-circle text-green-500 mt-1 mr-3"></i>
                <div>
                    <h4 class="font-medium text-gray-800">Verified Accounts</h4>
                    <p class="text-sm text-gray-600 mt-1">Only verified accounts can receive payments</p>
                </div>
            </div>
            <div class="flex items-start">
                <i class="fas fa-exclamation-triangle text-yellow-500 mt-1 mr-3"></i>
                <div>
                    <h4 class="font-medium text-gray-800">Default Account</h4>
                    <p class="text-sm text-gray-600 mt-1">Payments will be sent to your default account</p>
                </div>
            </div>
            <div class="flex items-start">
                <i class="fas fa-shield-alt text-blue-500 mt-1 mr-3"></i>
                <div>
                    <h4 class="font-medium text-gray-800">Secure Verification</h4>
                    <p class="text-sm text-gray-600 mt-1">Accounts are verified via Paystack for security</p>
                </div>
            </div>
            <div class="flex items-start">
                <i class="fas fa-clock text-purple-500 mt-1 mr-3"></i>
                <div>
                    <h4 class="font-medium text-gray-800">Processing Time</h4>
                    <p class="text-sm text-gray-600 mt-1">Payments are processed within 24-48 hours</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Bank Account Modal -->
<div class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center hidden" id="addAccountModal">
    <div class="bg-white p-6 rounded-lg w-full max-w-md mx-4 relative max-h-[90vh] overflow-y-auto">
        <span class="absolute top-4 right-4 text-gray-500 cursor-pointer text-2xl" 
              onclick="closeAddAccountModal()">&times;</span>
        
        <h3 class="text-xl font-bold mb-4">Add Bank Account</h3>
        
        <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-4 rounded">
            <p class="text-sm text-yellow-700">
                <i class="fas fa-exclamation-triangle mr-1"></i> 
                <strong>Important:</strong> Ensure account name matches your business account exactly.
            </p>
        </div>
        
        <!-- Modal Messages -->
        <div id="modalMessageContainer" class="mb-4"></div>
        
        <form id="addAccountForm" method="POST">
            <input type="hidden" name="vendor_id" value="<?= $vendor['id'] ?>">
            
            <div class="mb-4">
                <label class="block text-gray-700 mb-2 font-medium">Bank</label>
                <select name="bank_code" id="bankCode" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500" required>
                    <option value="">Select Bank</option>
                </select>
                <small class="text-red-500 text-sm mt-1 hidden" id="bank_code_error"></small>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 mb-2 font-medium">Account Number</label>
                <input type="text" name="account_number" id="accountNumber" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-primary-500" 
                       required maxlength="10" placeholder="10-digit account number"
                       oninput="validateAccountNumber(this.value)">
                <small class="text-red-500 text-sm mt-1 hidden" id="account_number_error"></small>
                <p class="text-xs text-gray-500 mt-1">Enter your 10-digit NUBAN account number</p>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 mb-2 font-medium">Account Name</label>
                <input type="text" name="account_name" id="accountNameInput" 
                       class="w-full px-3 py-2 border border-gray-300 rounded-lg bg-gray-50" 
                       readonly placeholder="Will be fetched automatically">
                <small class="text-red-500 text-sm mt-1 hidden" id="account_name_error"></small>
                <p class="text-xs text-gray-500 mt-1">Account name fetched from bank via Paystack</p>
            </div>
            
            <div class="mb-6">
                <div class="flex items-start">
                    <input type="checkbox" id="confirmName" name="confirm_name" class="mt-1 mr-2" required>
                    <label for="confirmName" class="text-sm text-gray-700">
                        I confirm that the account name displayed above matches my business bank account exactly
                    </label>
                </div>
                <small class="text-red-500 text-sm mt-1 hidden" id="confirm_name_error"></small>
            </div>
            
            <div class="mb-6">
                <div class="flex items-start">
                    <input type="checkbox" id="setAsDefault" name="set_as_default" class="mt-1 mr-2" checked>
                    <label for="setAsDefault" class="text-sm text-gray-700">
                        Set as default account for receiving payments
                    </label>
                </div>
            </div>
            
            <div class="flex justify-end space-x-3 pt-4 border-t border-gray-200">
                <button type="button" 
                        onclick="closeAddAccountModal()"
                        class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition">
                    Cancel
                </button>
                <button type="submit" name="add_account" id="submitBtn"
                        class="px-4 py-2 bg-primary-500 text-white rounded-lg hover:bg-primary-600 transition flex items-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed">
                    <span id="btnText">Add Account</span>
                    <div id="btnSpinner" class="hidden w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                </button>
            </div>
        </form>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
// Modal functions
function openAddAccountModal() {
    document.getElementById('addAccountModal').classList.remove('hidden');
    document.getElementById('modalMessageContainer').innerHTML = '';
    resetForm();
    fetchBanksFromPaystack();
}

function closeAddAccountModal() {
    document.getElementById('addAccountModal').classList.add('hidden');
    resetForm();
}

function resetForm() {
    document.getElementById('addAccountForm').reset();
    document.getElementById('accountNameInput').value = '';
    document.getElementById('accountNameInput').classList.remove('bg-gray-50', 'bg-green-100', 'bg-red-100');
    document.getElementById('accountNameInput').classList.add('bg-gray-50');
    document.getElementById('confirmName').checked = false;
    hideAllErrors();
}

function hideAllErrors() {
    $('.text-red-500').addClass('hidden').text('');
}

// Fetch banks from Paystack
function fetchBanksFromPaystack() {
    $('#bankCode').html('<option value="">Loading banks...</option>');
    $('#bankCode').prop('disabled', true);
    
    $.ajax({
        url: 'ajax_fetch_banks.php',
        type: 'GET',
        success: function(response) {
            $('#bankCode').prop('disabled', false);
            
            if (response.success && response.banks.length > 0) {
                let options = '<option value="">Select Bank</option>';
                response.banks.forEach(function(bank) {
                    options += `<option value="${bank.code}">${bank.name}</option>`;
                });
                $('#bankCode').html(options);
            } else {
                $('#bankCode').html('<option value="">Failed to load banks</option>');
                showModalMessage(response.message || 'Failed to load banks', 'error');
            }
        },
        error: function() {
            $('#bankCode').prop('disabled', false);
            $('#bankCode').html('<option value="">Failed to load banks</option>');
            showModalMessage('Network error. Please try again.', 'error');
        }
    });
}

// Account number validation and name lookup
let accountValidationTimeout;

function validateAccountNumber(accountNumber) {
    const accountNumberError = document.getElementById('account_number_error');
    const accountNameInput = document.getElementById('accountNameInput');
    const confirmCheckbox = document.getElementById('confirmName');
    
    clearTimeout(accountValidationTimeout);
    
    // Basic validation
    if (!/^\d{10}$/.test(accountNumber)) {
        accountNumberError.textContent = 'Please enter a valid 10-digit account number';
        accountNumberError.classList.remove('hidden');
        accountNameInput.value = '';
        accountNameInput.classList.remove('bg-green-100', 'bg-red-100');
        accountNameInput.classList.add('bg-gray-50');
        confirmCheckbox.checked = false;
        confirmCheckbox.disabled = true;
        return;
    } else {
        accountNumberError.classList.add('hidden');
    }
    
    accountValidationTimeout = setTimeout(() => {
        lookupAccountName(accountNumber);
    }, 800);
}

function lookupAccountName(accountNumber) {
    const bankCode = document.getElementById('bankCode').value;
    const accountNameInput = document.getElementById('accountNameInput');
    const accountNameError = document.getElementById('account_name_error');
    const confirmCheckbox = document.getElementById('confirmName');
    
    if (!bankCode) {
        accountNameError.textContent = 'Please select a bank first';
        accountNameError.classList.remove('hidden');
        accountNameInput.value = '';
        confirmCheckbox.checked = false;
        confirmCheckbox.disabled = true;
        return;
    }
    
    accountNameError.classList.add('hidden');
    accountNameInput.value = 'Fetching account name...';
    accountNameInput.classList.remove('bg-gray-50', 'bg-green-100', 'bg-red-100');
    confirmCheckbox.checked = false;
    confirmCheckbox.disabled = true;
    
    $.ajax({
        url: 'ajax_lookup_account_name.php',
        type: 'POST',
        data: {
            account_number: accountNumber,
            bank_code: bankCode
        },
        success: function(response) {
            if (response.success) {
                accountNameInput.value = response.account_name;
                accountNameInput.classList.remove('bg-gray-50', 'bg-red-100');
                accountNameInput.classList.add('bg-green-100');
                confirmCheckbox.disabled = false;
            } else {
                accountNameInput.value = 'Account not found';
                accountNameInput.classList.remove('bg-gray-50', 'bg-green-100');
                accountNameInput.classList.add('bg-red-100');
                confirmCheckbox.checked = false;
                confirmCheckbox.disabled = true;
                
                if (response.message) {
                    accountNameError.textContent = response.message;
                    accountNameError.classList.remove('hidden');
                }
            }
        },
        error: function() {
            accountNameInput.value = 'Verification failed';
            accountNameInput.classList.remove('bg-gray-50', 'bg-green-100');
            accountNameInput.classList.add('bg-red-100');
            confirmCheckbox.checked = false;
            confirmCheckbox.disabled = true;
            accountNameError.textContent = 'Failed to verify account. Please check manually.';
            accountNameError.classList.remove('hidden');
        }
    });
}

// Add bank account form submission
$(document).ready(function() {
    $('#addAccountForm').on('submit', function(e) {
        e.preventDefault();
        
        const formData = $(this).serialize();
        const submitBtn = $('#submitBtn');
        const btnText = $('#btnText');
        const btnSpinner = $('#btnSpinner');
        const modalMessageContainer = $('#modalMessageContainer');
        
        hideAllErrors();
        modalMessageContainer.html('');
        
        const accountName = $('#accountNameInput').val();
        const isConfirmed = $('#confirmName').is(':checked');
        
        if (!accountName || accountName === 'Fetching account name...' || 
            accountName === 'Account not found' || accountName === 'Verification failed') {
            showModalMessage('Please verify your account number first', 'error');
            return;
        }
        
        if (!isConfirmed) {
            $('#confirm_name_error').text('You must confirm the account name matches').removeClass('hidden');
            showModalMessage('Please confirm that the account name matches your bank account', 'error');
            return;
        }
        
        submitBtn.prop('disabled', true);
        btnText.text('Adding...');
        btnSpinner.removeClass('hidden');
        
        $.ajax({
            url: 'ajax_add_vendor_bank_account.php',
            type: 'POST',
            data: formData,
            success: function(response) {
                submitBtn.prop('disabled', false);
                btnText.text('Add Account');
                btnSpinner.addClass('hidden');
                
                if (response.success) {
                    showMessage(response.message, 'success');
                    closeAddAccountModal();
                    
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showModalMessage(response.message, 'error');
                    
                    if (response.errors) {
                        for (const field in response.errors) {
                            $(`#${field}_error`).text(response.errors[field]).removeClass('hidden');
                        }
                    }
                }
            },
            error: function(xhr, status, error) {
                submitBtn.prop('disabled', false);
                btnText.text('Add Account');
                btnSpinner.addClass('hidden');
                
                console.error('AJAX Error:', error);
                showModalMessage('An error occurred. Please try again.', 'error');
            }
        });
    });
});

// Set default account
function setDefaultAccount(accountId) {
    if (!confirm('Set this as your default account for receiving payments?')) {
        return;
    }
    
    $.ajax({
        url: 'ajax_set_default_account.php',
        type: 'POST',
        data: { account_id: accountId, type: 'vendor' },
        success: function(response) {
            if (response.success) {
                showMessage(response.message, 'success');
                setTimeout(() => {
                    location.reload();
                }, 1000);
            } else {
                showMessage(response.message, 'error');
            }
        },
        error: function() {
            showMessage('An error occurred. Please try again.', 'error');
        }
    });
}

// Delete bank account
function deleteBankAccount(accountId) {
    if (!confirm('Are you sure you want to delete this bank account?')) {
        return;
    }
    
    $.ajax({
        url: 'ajax_delete_vendor_bank_account.php',
        type: 'POST',
        data: { account_id: accountId },
        success: function(response) {
            if (response.success) {
                showMessage(response.message, 'success');
                setTimeout(() => {
                    location.reload();
                }, 1500);
            } else {
                showMessage(response.message, 'error');
            }
        },
        error: function() {
            showMessage('An error occurred. Please try again.', 'error');
        }
    });
}

// Message functions
function showMessage(message, type) {
    const messageContainer = document.getElementById('messageContainer');
    showMessageInContainer(message, type, messageContainer);
}

function showModalMessage(message, type) {
    const messageContainer = document.getElementById('modalMessageContainer');
    showMessageInContainer(message, type, messageContainer);
}

function showMessageInContainer(message, type, container) {
    const messageClass = type === 'success' 
        ? 'bg-green-100 border-green-400 text-green-700' 
        : 'bg-red-100 border-red-400 text-red-700';
    
    const messageHTML = `
        <div class="mb-4 border px-4 py-3 rounded relative ${messageClass}" role="alert">
            <span class="block sm:inline">${message}</span>
            <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.remove()">
                <svg class="fill-current h-6 w-6 ${type === 'success' ? 'text-green-500' : 'text-red-500'}" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
                    <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
                </svg>
            </span>
        </div>
    `;
    
    container.innerHTML = messageHTML;
    
    if (type === 'success') {
        setTimeout(() => {
            const alert = container.querySelector('.bg-green-100');
            if (alert) {
                alert.remove();
            }
        }, 5000);
    }
}

window.addEventListener('click', function(event) {
    const modal = document.getElementById('addAccountModal');
    if (event.target === modal) {
        closeAddAccountModal();
    }
});
</script>

<?php include 'include/footer.php'; ?>