<?php
// verify.php - Updated with correct column names
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

session_start();

// Check if verification parameters exist
if (!isset($_GET['code']) || !isset($_GET['email'])) {
    $error = "Invalid verification link. Missing parameters.";
    $success = false;
} else {
    $verification_code = trim($_GET['code']);
    $email = trim($_GET['email']);
    
    try {
        // Database connection
        require '../config/conn.php';
        
        if (!isset($conn) || !$conn) {
            throw new Exception("Database connection failed.");
        }
        
        // Set timezone
        $conn->query("SET time_zone = '+01:00'");
        
        // Check if vendor exists with verification code
        $query = "SELECT id, vendor_id, business_name, contact_person, 
                         verification_code_expiry, verification_code, 
                         email_verified_at, account_status, is_verified, created_at 
                  FROM vendors 
                  WHERE verification_code = ? AND email = ?";
        $stmt = $conn->prepare($query);
        
        if (!$stmt) {
            throw new Exception("Database query preparation failed: " . $conn->error);
        }
        
        $stmt->bind_param("ss", $verification_code, $email);
        
        if (!$stmt->execute()) {
            throw new Exception("Query execution failed: " . $stmt->error);
        }
        
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            // Check if email exists but with different verification code
            $check_email_query = "SELECT id, verification_code, email_verified_at, is_verified FROM vendors WHERE email = ?";
            $check_stmt = $conn->prepare($check_email_query);
            $check_stmt->bind_param("s", $email);
            $check_stmt->execute();
            $email_result = $check_stmt->get_result();
            
            if ($email_result->num_rows > 0) {
                $vendor_data = $email_result->fetch_assoc();
                // Check if already verified
                if ($vendor_data['email_verified_at'] !== null || $vendor_data['is_verified'] == 1) {
                    $error = "This email has already been verified. Please login.";
                    $already_verified = true;
                } else {
                    $error = "Invalid verification link. Please use the latest verification email sent to you.";
                    $resend_available = true;
                }
            } else {
                $error = "Invalid verification link. No vendor account found with this email.";
            }
            
            $check_stmt->close();
            $success = false;
        } else {
            $vendor = $result->fetch_assoc();
            
            // Check if already verified
            if ($vendor['email_verified_at'] !== null || $vendor['is_verified'] == 1) {
                $error = "This email has already been verified. You can now login.";
                $already_verified = true;
                $success = false;
            } else {
                // Check if verification code has expired
                $current_time = time();
                $expiry_time = strtotime($vendor['verification_code_expiry']);
                
                if ($current_time > $expiry_time) {
                    $error = "Verification link has expired. Please request a new verification email.";
                    $expired = true;
                    $resend_available = true;
                    $success = false;
                } else {
                    // Verify the email
                    $verify_time = date('Y-m-d H:i:s');
                    $update_query = "UPDATE vendors 
                                    SET email_verified_at = ?, 
                                        verification_code = NULL, 
                                        verification_code_expiry = NULL,
                                        account_status = 'pending_approval',
                                        is_verified = 1,
                                        status = 'pending',
                                        updated_at = ?
                                    WHERE id = ?";
                    
                    $update_stmt = $conn->prepare($update_query);
                    
                    if (!$update_stmt) {
                        throw new Exception("Failed to prepare update statement: " . $conn->error);
                    }
                    
                    $update_stmt->bind_param("ssi", $verify_time, $verify_time, $vendor['id']);
                    
                    if ($update_stmt->execute()) {
                        $success = true;
                        $message = "Email verification successful! Your vendor account is now under review by our procurement team. You'll be notified via email once your account is approved.";
                        
                        // Store vendor info in session
                        $_SESSION['vendor_id'] = $vendor['vendor_id'];
                        $_SESSION['vendor_name'] = $vendor['business_name'];
                        $_SESSION['vendor_email'] = $email;
                        $_SESSION['vendor_verified'] = true;
                        $_SESSION['vendor_pending'] = true;
                    } else {
                        throw new Exception("Failed to update verification status: " . $update_stmt->error);
                    }
                    
                    $update_stmt->close();
                }
            }
        }
        
        $stmt->close();
        $conn->close();
        
    } catch (Exception $e) {
        $error = "An error occurred during verification: " . $e->getMessage();
        $success = false;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Verification - Aletheia University</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { 
            font-family: 'Poppins', sans-serif;
            background-color: #f8fafc;
        }
    </style>
</head>
<body class="min-h-screen flex items-center justify-center p-4">
    <div class="w-full max-w-md">
        <!-- University Header -->
        <div class="text-center mb-8">
            <div class="w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                <div class="w-16 h-16 bg-[#fbbb01] rounded-full flex items-center justify-center">
                    <i class="fas fa-university text-white text-2xl"></i>
                </div>
            </div>
            <h1 class="text-2xl font-bold text-gray-800">ALETHEIA UNIVERSITY</h1>
            <p class="text-gray-500 text-sm mt-1">Campus Services - Vendor Email Verification</p>
        </div>

        <!-- Verification Result Card -->
        <div class="bg-white rounded-xl shadow-lg p-6">
            <?php if (isset($success) && $success): ?>
                <!-- Success Message -->
                <div class="text-center">
                    <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-check-circle text-green-500 text-3xl"></i>
                    </div>
                    <h2 class="text-xl font-bold text-gray-800 mb-3">Email Verified Successfully!</h2>
                    <p class="text-gray-600 mb-4"><?php echo htmlspecialchars($message); ?></p>
                    
                    <?php if (isset($vendor)): ?>
                        <div class="bg-gray-50 p-4 rounded-lg mb-4">
                            <p class="text-sm text-gray-500 mb-1">Vendor ID</p>
                            <p class="font-medium text-gray-800"><?php echo htmlspecialchars($vendor['vendor_id']); ?></p>
                            <p class="text-sm text-gray-500 mb-1 mt-3">Business Name</p>
                            <p class="font-medium text-gray-800"><?php echo htmlspecialchars($vendor['business_name']); ?></p>
                            <p class="text-sm text-gray-500 mb-1 mt-3">Contact Person</p>
                            <p class="font-medium text-gray-800"><?php echo htmlspecialchars($vendor['contact_person']); ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <div class="space-y-3">
                        <a href="vendor-login.php" 
                           class="block w-full py-3 bg-[#fbbb01] text-white font-medium rounded-lg hover:bg-[#e0a800] transition text-center">
                            Go to Login
                        </a>
                        <a href="../index.php" 
                           class="block w-full py-2.5 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition text-center text-sm">
                            Return to Home
                        </a>
                    </div>
                </div>
                
            <?php elseif (isset($already_verified)): ?>
                <!-- Already Verified -->
                <div class="text-center">
                    <div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-info-circle text-blue-500 text-3xl"></i>
                    </div>
                    <h2 class="text-xl font-bold text-gray-800 mb-3">Already Verified</h2>
                    <p class="text-gray-600 mb-4"><?php echo htmlspecialchars($error); ?></p>
                    
                    <div class="space-y-3">
                        <a href="vendor-login.php" 
                           class="block w-full py-3 bg-[#fbbb01] text-white font-medium rounded-lg hover:bg-[#e0a800] transition text-center">
                            Go to Login
                        </a>
                    </div>
                </div>
                
            <?php elseif (isset($expired) || isset($resend_available)): ?>
                <!-- Expired or Invalid Link -->
                <div class="text-center">
                    <div class="w-16 h-16 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-exclamation-triangle text-yellow-500 text-3xl"></i>
                    </div>
                    <h2 class="text-xl font-bold text-gray-800 mb-3">
                        <?php echo isset($expired) ? 'Verification Link Expired' : 'Invalid Verification Link'; ?>
                    </h2>
                    <p class="text-gray-600 mb-4"><?php echo htmlspecialchars($error); ?></p>
                    
                    <?php if (isset($resend_available) && isset($email)): ?>
                        <form method="POST" action="resend-verification.php" class="mb-4">
                            <input type="hidden" name="email" value="<?php echo htmlspecialchars($email); ?>">
                            <button type="submit" 
                                    class="w-full py-3 bg-[#fbbb01] text-white font-medium rounded-lg hover:bg-[#e0a800] transition flex items-center justify-center">
                                <i class="fas fa-paper-plane mr-2"></i> Send New Verification Email
                            </button>
                        </form>
                    <?php endif; ?>
                    
                    <div class="space-y-3">
                        <a href="vendor-login.php" 
                           class="block w-full py-2.5 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition text-center text-sm">
                            Return to Login
                        </a>
                        <a href="vendor_registration.php" 
                           class="block w-full py-2.5 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition text-center text-sm">
                            Register New Account
                        </a>
                    </div>
                </div>
                
            <?php else: ?>
                <!-- Error -->
                <div class="text-center">
                    <div class="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-times-circle text-red-500 text-3xl"></i>
                    </div>
                    <h2 class="text-xl font-bold text-gray-800 mb-3">Verification Failed</h2>
                    <p class="text-gray-600 mb-4"><?php echo htmlspecialchars($error ?? 'Unknown error occurred.'); ?></p>
                    
                    <div class="space-y-3">
                        <a href="vendor-login.php" 
                           class="block w-full py-3 bg-[#fbbb01] text-white font-medium rounded-lg hover:bg-[#e0a800] transition text-center">
                            Go to Login
                        </a>
                        <a href="vendor_registration.php" 
                           class="block w-full py-2.5 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition text-center text-sm">
                            Register New Account
                        </a>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Support Info -->
            <div class="mt-6 pt-4 border-t border-gray-100 text-center">
                <p class="text-xs text-gray-500">
                    Need help? Contact: 
                    <a href="mailto:procurement@aletheiauniversity.edu.ng" class="text-[#fbbb01] hover:text-[#e0a800]">
                        procurement@aletheiauniversity.edu.ng
                    </a>
                </p>
            </div>
        </div>
    </div>

    <script>
    // Auto-redirect to login after 8 seconds if verified
    <?php if (isset($success) && $success): ?>
        setTimeout(function() {
            window.location.href = 'vendor-login.php';
        }, 8000);
    <?php endif; ?>
    
    // Auto-redirect to login after 5 seconds if already verified
    <?php if (isset($already_verified)): ?>
        setTimeout(function() {
            window.location.href = 'vendor-login.php';
        }, 5000);
    <?php endif; ?>
    </script>
</body>
</html>