<?php
// ajax_add_vendor_bank_account.php
header('Content-Type: application/json');
session_start();

if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'vendor') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

require_once '../config/conn.php';

// Get POST data
$vendor_id = $_POST['vendor_id'] ?? '';
$account_number = $_POST['account_number'] ?? '';
$account_name = $_POST['account_name'] ?? '';
$bank_code = $_POST['bank_code'] ?? '';
$confirm_name = isset($_POST['confirm_name']) ? 1 : 0;
$set_as_default = isset($_POST['set_as_default']) ? 1 : 0;

// Validation
$errors = [];

if (empty($vendor_id)) $errors['vendor_id'] = 'Vendor ID is required';
if (empty($account_number)) $errors['account_number'] = 'Account number is required';
if (empty($account_name)) $errors['account_name'] = 'Account name is required';
if (empty($bank_code)) $errors['bank_code'] = 'Bank is required';
if (!$confirm_name) $errors['confirm_name'] = 'You must confirm the account name';

// Validate account number format
if (!empty($account_number) && !preg_match('/^\d{10}$/', $account_number)) {
    $errors['account_number'] = 'Invalid account number format';
}

// Check for duplicate account number for this vendor
if (empty($errors)) {
    $check_stmt = $conn->prepare("
        SELECT id FROM vendor_bank_accounts 
        WHERE vendor_id = ? AND account_number = ?
    ");
    $check_stmt->bind_param("is", $vendor_id, $account_number);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        $errors['account_number'] = 'This account number is already added';
    }
}

if (!empty($errors)) {
    echo json_encode(['success' => false, 'message' => 'Please fix the errors', 'errors' => $errors]);
    exit();
}

// Get bank name from Paystack
$bank_name = 'Unknown Bank';
$paystack_secret_key = 'sk_test_YOUR_SECRET_KEY'; // Replace with your key

$bank_ch = curl_init();
curl_setopt($bank_ch, CURLOPT_URL, "https://api.paystack.co/bank/" . urlencode($bank_code));
curl_setopt($bank_ch, CURLOPT_RETURNTRANSFER, 1);
curl_setopt($bank_ch, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer " . $paystack_secret_key,
    "Content-Type: application/json"
]);

$bank_response = curl_exec($bank_ch);
if ($bank_response !== false) {
    $bank_result = json_decode($bank_response, true);
    if (isset($bank_result['status']) && $bank_result['status'] === true) {
        $bank_name = $bank_result['data']['name'];
    }
}
curl_close($bank_ch);

// If setting as default, unset other defaults
if ($set_as_default) {
    $reset_stmt = $conn->prepare("
        UPDATE vendor_bank_accounts 
        SET is_default = 0 
        WHERE vendor_id = ?
    ");
    $reset_stmt->bind_param("i", $vendor_id);
    $reset_stmt->execute();
}

// Insert bank account
try {
    $conn->begin_transaction();
    
    $stmt = $conn->prepare("
        INSERT INTO vendor_bank_accounts 
        (vendor_id, account_number, account_name, bank_code, bank_name, is_verified, is_default, created_at)
        VALUES (?, ?, ?, ?, ?, 1, ?, NOW())
    ");
    
    $stmt->bind_param("issssi", $vendor_id, $account_number, $account_name, $bank_code, $bank_name, $set_as_default);
    
    if ($stmt->execute()) {
        $conn->commit();
        
        // Log activity
        $log_stmt = $conn->prepare("
            INSERT INTO vendor_activity_log 
            (vendor_id, activity_type, description, created_at)
            VALUES (?, 'bank_account_added', ?, NOW())
        ");
        $activity_desc = "Added bank account: " . $bank_name . " - " . $account_number;
        $log_stmt->bind_param("is", $vendor_id, $activity_desc);
        $log_stmt->execute();
        
        echo json_encode([
            'success' => true,
            'message' => 'Bank account added successfully!',
            'is_default' => $set_as_default
        ]);
    } else {
        $conn->rollback();
        echo json_encode(['success' => false, 'message' => 'Failed to save bank account']);
    }
    
    $stmt->close();
} catch (Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}