<?php
// edit-product.php - Edit Product Page
include 'include/header.php';

// Verify session
if (!isset($_SESSION['vendor_id'])) {
    header("Location: login.php");
    exit();
}

require_once '../config/conn.php';

$vendor_id = $_SESSION['vendor_id'];
$product_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$message = '';
$error = '';

// Fetch product details
$stmt = $conn->prepare("SELECT * FROM products WHERE id = ? AND vendor_id = ?");
$stmt->bind_param("ii", $product_id, $vendor_id);
$stmt->execute();
$result = $stmt->get_result();
$product = $result->fetch_assoc();
$stmt->close();

if (!$product) {
    $_SESSION['error'] = "Product not found";
    header("Location: product.php");
    exit();
}

// Handle product update
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['update_product'])) {
    // Get form data
    $name = trim($_POST['name']);
    $description = trim($_POST['description']);
    $price = floatval($_POST['price']);
    $stock_status = $_POST['stock_status']; // 'in_stock' or 'out_of_stock'
    
    // Validate inputs
    if (empty($name)) {
        $error = "Product name is required.";
    } elseif ($price <= 0) {
        $error = "Price must be greater than 0.";
    } elseif (!in_array($stock_status, ['in_stock', 'out_of_stock'])) {
        $error = "Invalid stock status selected.";
    } else {
        // Handle image upload
        $image_path = $product['image_path']; // Keep existing image by default
        
        if (isset($_FILES['product_image']) && $_FILES['product_image']['error'] === UPLOAD_ERR_OK) {
            $upload_dir = '../uploads/products/';
            
            // Create upload directory if it doesn't exist
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            // Delete old image if exists
            if (!empty($product['image_path']) && file_exists('../' . $product['image_path'])) {
                unlink('../' . $product['image_path']);
            }
            
            // Generate unique filename
            $file_extension = strtolower(pathinfo($_FILES['product_image']['name'], PATHINFO_EXTENSION));
            $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
            
            if (in_array($file_extension, $allowed_extensions)) {
                $unique_name = uniqid() . '_' . time() . '.' . $file_extension;
                $target_file = $upload_dir . $unique_name;
                
                // Validate image size (max 5MB)
                if ($_FILES['product_image']['size'] <= 5 * 1024 * 1024) {
                    if (move_uploaded_file($_FILES['product_image']['tmp_name'], $target_file)) {
                        $image_path = 'uploads/products/' . $unique_name;
                    } else {
                        $error = "Failed to upload image. Please try again.";
                    }
                } else {
                    $error = "Image size must be less than 5MB.";
                }
            } else {
                $error = "Only JPG, JPEG, PNG, GIF, and WebP files are allowed.";
            }
        } elseif ($_FILES['product_image']['error'] !== UPLOAD_ERR_NO_FILE) {
            // Handle other upload errors
            switch ($_FILES['product_image']['error']) {
                case UPLOAD_ERR_INI_SIZE:
                case UPLOAD_ERR_FORM_SIZE:
                    $error = "Image file is too large.";
                    break;
                case UPLOAD_ERR_PARTIAL:
                    $error = "Image was only partially uploaded.";
                    break;
                case UPLOAD_ERR_NO_TMP_DIR:
                    $error = "Missing temporary folder.";
                    break;
                case UPLOAD_ERR_CANT_WRITE:
                    $error = "Failed to write image to disk.";
                    break;
                case UPLOAD_ERR_EXTENSION:
                    $error = "Image upload stopped by extension.";
                    break;
                default:
                    $error = "Unknown upload error.";
            }
        }
        
        // Handle image removal
        if (isset($_POST['remove_image']) && $_POST['remove_image'] == '1') {
            if (!empty($product['image_path']) && file_exists('../' . $product['image_path'])) {
                unlink('../' . $product['image_path']);
            }
            $image_path = '';
        }
        
        if (empty($error)) {
            // Update product in database
            $stmt = $conn->prepare("UPDATE products SET name = ?, description = ?, price = ?, stock_status = ?, image_path = ?, updated_at = NOW() WHERE id = ? AND vendor_id = ?");
            
            if ($stmt) {
                $stmt->bind_param("ssdssii", $name, $description, $price, $stock_status, $image_path, $product_id, $vendor_id);
                
                if ($stmt->execute()) {
                    $message = "Product updated successfully!";
                    // Refresh product data
                    $refresh_stmt = $conn->prepare("SELECT * FROM products WHERE id = ? AND vendor_id = ?");
                    $refresh_stmt->bind_param("ii", $product_id, $vendor_id);
                    $refresh_stmt->execute();
                    $refresh_result = $refresh_stmt->get_result();
                    $product = $refresh_result->fetch_assoc();
                    $refresh_stmt->close();
                } else {
                    $error = "Error updating product: " . $stmt->error;
                }
                $stmt->close();
            } else {
                $error = "Database error: " . $conn->error;
            }
        }
    }
}
?>

<main class="main-content flex-1 pb-8">
    <div class="max-w-4xl mx-auto py-6 sm:px-6 lg:px-8">
        <!-- Success/Error Messages -->
        <?php if ($message): ?>
        <div class="mb-4 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
            <span class="block sm:inline"><?= htmlspecialchars($message) ?></span>
            <button type="button" class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
                <svg class="fill-current h-6 w-6 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
                    <title>Close</title>
                    <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
                </svg>
            </button>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="mb-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
            <span class="block sm:inline"><?= htmlspecialchars($error) ?></span>
            <button type="button" class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
                <svg class="fill-current h-6 w-6 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20">
                    <title>Close</title>
                    <path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/>
                </svg>
            </button>
        </div>
        <?php endif; ?>

        <div class="bg-white dark:bg-gray-800 shadow rounded-lg">
            <div class="px-4 py-5 sm:px-6 border-b border-gray-200 dark:border-gray-700">
                <div class="flex items-center justify-between">
                    <div>
                        <h3 class="text-lg leading-6 font-medium text-gray-900 dark:text-white">Edit Product</h3>
                        <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Update your product details.</p>
                    </div>
                    <a href="product.php" class="bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-4 rounded-md text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                        Back to Products
                    </a>
                </div>
            </div>
            
            <div class="px-4 py-5 sm:p-6">
                <form method="POST" action="" enctype="multipart/form-data" id="editProductForm">
                    <div class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <!-- Left Column: Form Fields -->
                            <div class="space-y-4">
                                <div>
                                    <label for="name" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                        Product Name <span class="text-red-500">*</span>
                                    </label>
                                    <input type="text" name="name" id="name" required 
                                           value="<?= htmlspecialchars($product['name']) ?>"
                                           class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                </div>

                                <div>
                                    <label for="description" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Description</label>
                                    <textarea name="description" id="description" rows="4"
                                              class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white"><?= htmlspecialchars($product['description']) ?></textarea>
                                </div>

                                <div class="grid grid-cols-2 gap-4">
                                    <div>
                                        <label for="price" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Price (₦) <span class="text-red-500">*</span>
                                        </label>
                                        <input type="number" name="price" id="price" required step="0.01" min="0.01"
                                               value="<?= htmlspecialchars($product['price']) ?>"
                                               class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                    </div>

                                    <div>
                                        <label for="stock_status" class="block text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Stock Status <span class="text-red-500">*</span>
                                        </label>
                                        <select name="stock_status" id="stock_status" required
                                                class="mt-1 block w-full border border-gray-300 dark:border-gray-600 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary-500 focus:border-primary-500 dark:bg-gray-700 dark:text-white">
                                            <option value="in_stock" <?= ($product['stock_status'] == 'in_stock') ? 'selected' : '' ?>>In Stock</option>
                                            <option value="out_of_stock" <?= ($product['stock_status'] == 'out_of_stock') ? 'selected' : '' ?>>Out of Stock</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Right Column: Image Upload -->
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Current Image</label>
                                    <div class="mt-1">
                                        <?php if (!empty($product['image_path'])): ?>
                                            <div class="relative">
                                                <img id="currentImage" src="../<?= htmlspecialchars($product['image_path']) ?>" 
                                                     alt="<?= htmlspecialchars($product['name']) ?>"
                                                     class="w-full h-48 object-cover rounded-lg shadow-md">
                                                <button type="button" id="removeImageBtn" 
                                                        class="absolute top-2 right-2 bg-red-600 text-white p-1 rounded-full hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500">
                                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                                                    </svg>
                                                </button>
                                            </div>
                                            <input type="hidden" name="remove_image" id="removeImageInput" value="0">
                                        <?php else: ?>
                                            <div class="h-48 w-full bg-gray-200 dark:bg-gray-700 rounded-lg flex items-center justify-center">
                                                <svg class="h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                                </svg>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>

                                <div>
                                    <label for="product_image" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Update Image</label>
                                    <div class="mt-1 flex items-center">
                                        <input type="file" name="product_image" id="product_image" accept="image/*"
                                               class="block w-full text-sm text-gray-500 dark:text-gray-400
                                                      file:mr-4 file:py-2 file:px-4
                                                      file:rounded-md file:border-0
                                                      file:text-sm file:font-semibold
                                                      file:bg-primary-50 file:text-primary-700 dark:file:bg-primary-900 dark:file:text-primary-200
                                                      hover:file:bg-primary-100 dark:hover:file:bg-primary-800">
                                    </div>
                                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">Max size: 5MB. Allowed: JPG, PNG, GIF, WebP</p>
                                    <div id="imagePreview" class="mt-2 hidden">
                                        <img id="previewImage" class="w-full h-48 object-cover rounded-lg shadow-md">
                                    </div>
                                </div>

                                <!-- Product Info -->
                                <div class="pt-4 border-t border-gray-200 dark:border-gray-700">
                                    <h4 class="text-sm font-medium text-gray-700 dark:text-gray-300">Product Information</h4>
                                    <dl class="mt-2 space-y-2">
                                        <div class="flex justify-between">
                                            <dt class="text-sm text-gray-500 dark:text-gray-400">Created</dt>
                                            <dd class="text-sm text-gray-900 dark:text-white"><?= date('M d, Y', strtotime($product['created_at'])) ?></dd>
                                        </div>
                                        <div class="flex justify-between">
                                            <dt class="text-sm text-gray-500 dark:text-gray-400">Last Updated</dt>
                                            <dd class="text-sm text-gray-900 dark:text-white"><?= date('M d, Y', strtotime($product['updated_at'])) ?></dd>
                                        </div>
                                        <div class="flex justify-between">
                                            <dt class="text-sm text-gray-500 dark:text-gray-400">Product ID</dt>
                                            <dd class="text-sm text-gray-900 dark:text-white">#<?= $product['id'] ?></dd>
                                        </div>
                                    </dl>
                                </div>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="flex flex-col sm:flex-row sm:justify-between gap-4 pt-6 border-t border-gray-200 dark:border-gray-700">
                            <div class="flex gap-3">
                                <a href="product.php" class="bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 py-2 px-4 rounded-md text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500">
                                    Cancel
                                </a>
                                <a href="product.php?delete_id=<?= $product['id'] ?>" 
                                   class="bg-red-600 text-white py-2 px-4 rounded-md text-sm font-medium hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                   onclick="return confirm('Are you sure you want to delete this product? This action cannot be undone.')">
                                    Delete Product
                                </a>
                            </div>
                            <button type="submit" name="update_product" class="bg-primary-600 border border-transparent rounded-md shadow-sm py-2 px-4 inline-flex justify-center text-sm font-medium text-white hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 transition duration-150 ease-in-out">
                                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                                Update Product
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</main>

<script>
// Image preview functionality
document.getElementById('product_image').addEventListener('change', function(e) {
    const previewContainer = document.getElementById('imagePreview');
    const previewImage = document.getElementById('previewImage');
    const currentImage = document.getElementById('currentImage');
    
    if (this.files && this.files[0]) {
        const file = this.files[0];
        
        // Validate file size (5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('Image size must be less than 5MB.');
            this.value = '';
            previewContainer.classList.add('hidden');
            return;
        }
        
        // Validate file type
        const validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!validTypes.includes(file.type)) {
            alert('Only JPG, JPEG, PNG, GIF, and WebP files are allowed.');
            this.value = '';
            previewContainer.classList.add('hidden');
            return;
        }
        
        const reader = new FileReader();
        
        reader.onload = function(e) {
            previewImage.src = e.target.result;
            previewContainer.classList.remove('hidden');
            
            // Hide current image if exists
            if (currentImage) {
                currentImage.style.display = 'none';
            }
        }
        
        reader.onerror = function() {
            alert('Error reading image file.');
            previewContainer.classList.add('hidden');
        }
        
        reader.readAsDataURL(file);
    } else {
        previewContainer.classList.add('hidden');
        // Show current image again
        if (currentImage) {
            currentImage.style.display = 'block';
        }
    }
});

// Remove image functionality
const removeImageBtn = document.getElementById('removeImageBtn');
const removeImageInput = document.getElementById('removeImageInput');
const currentImage = document.getElementById('currentImage');

if (removeImageBtn && removeImageInput && currentImage) {
    removeImageBtn.addEventListener('click', function() {
        if (confirm('Are you sure you want to remove this image?')) {
            currentImage.style.display = 'none';
            removeImageInput.value = '1';
            removeImageBtn.style.display = 'none';
        }
    });
}

// Form validation
document.getElementById('editProductForm').addEventListener('submit', function(e) {
    const name = document.getElementById('name').value.trim();
    const price = document.getElementById('price').value;
    const stockStatus = document.getElementById('stock_status').value;
    
    // Clear previous error messages
    const errorElements = document.querySelectorAll('.error-message');
    errorElements.forEach(el => el.remove());
    
    let hasError = false;
    
    // Validate name
    if (!name) {
        showError('name', 'Product name is required.');
        hasError = true;
    }
    
    // Validate price
    if (!price || parseFloat(price) <= 0) {
        showError('price', 'Price must be greater than 0.');
        hasError = true;
    }
    
    // Validate stock status
    if (!stockStatus) {
        showError('stock_status', 'Stock status is required.');
        hasError = true;
    }
    
    if (hasError) {
        e.preventDefault();
        return false;
    }
    
    return true;
});

function showError(fieldId, message) {
    const field = document.getElementById(fieldId);
    const errorElement = document.createElement('p');
    errorElement.className = 'mt-1 text-sm text-red-600 dark:text-red-400 error-message';
    errorElement.textContent = message;
    
    // Insert error message after the field
    field.parentNode.appendChild(errorElement);
    
    // Highlight the field
    field.classList.add('border-red-500', 'dark:border-red-400');
    field.classList.remove('border-gray-300', 'dark:border-gray-600');
    
    // Remove error highlighting when field is focused
    field.addEventListener('input', function() {
        this.classList.remove('border-red-500', 'dark:border-red-400');
        this.classList.add('border-gray-300', 'dark:border-gray-600');
        const errorMsg = this.parentNode.querySelector('.error-message');
        if (errorMsg) {
            errorMsg.remove();
        }
    }, { once: true });
}

// Price formatting
document.getElementById('price').addEventListener('blur', function() {
    const value = parseFloat(this.value);
    if (!isNaN(value) && value > 0) {
        this.value = value.toFixed(2);
    }
});

// Confirm before leaving page if form has changes
let formChanged = false;
const form = document.getElementById('editProductForm');
const formInputs = form.querySelectorAll('input, textarea, select');

formInputs.forEach(input => {
    const initialValue = input.value;
    input.addEventListener('input', () => {
        formChanged = input.value !== initialValue;
    });
});

window.addEventListener('beforeunload', function(e) {
    if (formChanged) {
        e.preventDefault();
        e.returnValue = '';
    }
});

// Reset form changed flag on submit
form.addEventListener('submit', function() {
    formChanged = false;
});
</script>

<?php include 'include/footer.php'; ?>