<?php
// process_vendor_login.php - FIXED VERSION
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require '../config/conn.php';
require '../vendor/autoload.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// Set JSON header
header('Content-Type: application/json');
// Also set MySQL timezone if connection exists
if (isset($conn) && $conn) {
    $conn->query("SET time_zone = '+01:00'"); // For Africa/Lagos (UTC+1)
}

// Initialize response array
$response = [
    'success' => false,
    'message' => '',
    'errors' => [],
    'redirect' => '',
    'verification_required' => false,
    'approval_pending' => false,
    'resend_verification' => false
];

try {
    // Check if form is submitted
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method');
    }

    // Get and validate input
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $remember_me = isset($_POST['remember_me']) ? 1 : 0;

    // Validate email
    if (empty($email)) {
        $response['errors']['email'] = 'Email is required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $response['errors']['email'] = 'Please enter a valid email address';
    }

    // Validate password
    if (empty($password)) {
        $response['errors']['password'] = 'Password is required';
    }

    // If there are validation errors, return them
    if (!empty($response['errors'])) {
        $response['message'] = 'Please fix the errors below';
        echo json_encode($response);
        exit;
    }

    // Check if vendor exists - INCLUDING is_approved column
    $stmt = $conn->prepare("SELECT id, vendor_id, business_name, email, password, 
                                   email_verified_at, is_verified, account_status, 
                                   status, is_approved, is_approved as approved_status 
                            FROM vendors WHERE email = ?");
    if (!$stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }
    
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        $response['errors']['email'] = 'No account found with this email address';
        $response['message'] = 'Invalid email or password';
        echo json_encode($response);
        exit;
    }

    $vendor = $result->fetch_assoc();

    // Verify password
    if (!password_verify($password, $vendor['password'])) {
        $response['errors']['password'] = 'Invalid password';
        $response['message'] = 'Invalid email or password';
        echo json_encode($response);
        exit;
    }

    // Check if email is verified (using email_verified_at column)
    if ($vendor['email_verified_at'] === null) {
        $response['message'] = 'Your account is not verified. Please check your email for verification link.';
        $response['resend_verification'] = true;
        $response['email'] = $email;
        
        // Store email in session for resend functionality
        $_SESSION['pending_verification_email'] = $email;
        echo json_encode($response);
        exit;
    }

    // DEBUG: Log the vendor data for troubleshooting
    error_log("Vendor Login Attempt - Email: $email");
    error_log("Vendor Data: " . print_r($vendor, true));

    // CHECK APPROVAL STATUS - FIXED LOGIC
    // First check is_approved column (which seems to be the main approval column)
    $is_approved = $vendor['is_approved'];
    
    if ($is_approved == 0) {
        // Account is pending approval
        $response['message'] = 'Your account is pending approval by the procurement department. You will be notified via email once approved.';
        $response['approval_pending'] = true;
        echo json_encode($response);
        exit;
    } elseif ($is_approved == -1) {
        // Account is rejected/disapproved
        $response['message'] = 'Your account has been disapproved. Please contact the procurement department for more information.';
        echo json_encode($response);
        exit;
    } elseif ($is_approved != 1) {
        // Account has invalid approval status
        $response['message'] = 'Your account has an invalid approval status. Please contact support.';
        echo json_encode($response);
        exit;
    }

    // If is_approved = 1, then check other status fields
    $account_status = strtolower($vendor['account_status'] ?? 'pending_approval');
    $status = strtolower($vendor['status'] ?? 'pending');

    // Check if account is active - with more flexible logic
    $is_active = false;
    
    // Check various combinations that indicate active status
    if ($account_status === 'active' || $status === 'active') {
        $is_active = true;
    }
    
    // Also check if not in any restricted state
    $restricted_states = ['rejected', 'suspended', 'inactive', 'pending', 'pending_approval'];
    if (in_array($account_status, $restricted_states) || in_array($status, $restricted_states)) {
        $is_active = false;
    }

    if (!$is_active) {
        // Determine the specific reason
        if ($account_status === 'rejected' || $status === 'rejected') {
            $response['message'] = 'Your account has been rejected. Please contact the procurement department for more information.';
        } elseif ($account_status === 'suspended' || $status === 'suspended') {
            $response['message'] = 'Your account has been suspended. Please contact the procurement department.';
        } elseif ($account_status === 'inactive' || $status === 'inactive') {
            $response['message'] = 'Your account is inactive. Please contact support to reactivate.';
        } elseif ($account_status === 'pending_approval' || $status === 'pending') {
            $response['message'] = 'Your account is pending approval by the procurement department. You will be notified via email once approved.';
            $response['approval_pending'] = true;
        } else {
            $response['message'] = 'Your account is not active. Current status: ' . 
                                  ucfirst($account_status) . ' / ' . ucfirst($status) . 
                                  '. Please contact support.';
        }
        echo json_encode($response);
        exit;
    }

    // Generate a 6-digit verification code for 2FA
    $verification_code = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);
    $expires_at = date('Y-m-d H:i:s', strtotime('+10 minutes'));
    
    // Check if vendor verification table exists, if not create it
    
    
    // Store the code in the vendor-specific verification table
    $stmt = $conn->prepare("INSERT INTO vendor_verification_codes (vendor_id, code, expires_at) VALUES (?, ?, ?)");
    if (!$stmt) {
        throw new Exception('Database error: ' . $conn->error);
    }
    
    if (!$stmt->bind_param("iss", $vendor['id'], $verification_code, $expires_at)) {
        throw new Exception('Failed to bind parameters');
    }
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to execute query: ' . $stmt->error);
    }

    // Send the verification code via email
    $mail = new PHPMailer(true);
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host       = 'smtp.gmail.com';
        $mail->SMTPAuth   = true;
        $mail->Username   = 'olorodemicheal2274@gmail.com';
        $mail->Password   = 'farq ptyc vjtl eses';
        $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
        $mail->Port       = 465;
        $mail->SMTPDebug = 0;
        
        // Recipients
        $mail->setFrom('noreply@aletheiauniversity.edu.ng', 'Aletheia University Vendor Portal');
        $mail->addAddress($email, $vendor['business_name']);
        
        // Content - Updated with Aletheia University branding and no header background
        $mail->isHTML(true);
        $mail->Subject = 'Your Vendor Login Verification Code - Aletheia University';
        
        $mail->Body = "
        <div style='max-width: 600px; margin: 0 auto; border: 1px solid #e0e0e0; border-radius: 8px; overflow: hidden; font-family: Arial, sans-serif;'>
            <!-- University Header - No Background Color -->
            <div style='padding: 25px; text-align: center;'>
                <h1 style='color: #fbbb01; margin: 0; font-size: 28px; font-weight: bold;'>ALETHEIA UNIVERSITY</h1>
                <p style='color: #666; margin: 5px 0 0; font-size: 16px;'>Campus Services - Vendor Portal</p>
            </div>
            
            <div style='padding: 25px;'>
                <h2 style='color: #fbbb01; margin-top: 0;'>Login Verification Code</h2>
                
                <p style='font-size: 16px;'>Hello <strong>{$vendor['business_name']}</strong>,</p>
                
                <p style='font-size: 16px;'>You have requested to login to your vendor account. Use the verification code below:</p>
                
                <div style='text-align: center; margin: 30px 0;'>
                    <div style='display: inline-block; padding: 20px 40px; background-color: #fef8e6; border: 3px dashed #fbbb01; border-radius: 8px; font-size: 32px; font-weight: bold; letter-spacing: 8px; color: #333;'>
                        $verification_code
                    </div>
                </div>
                
                <div style='background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin: 20px 0;'>
                    <p style='font-size: 14px; color: #666; margin: 0;'>
                        <strong>Important:</strong> This verification code will expire in 10 minutes.
                    </p>
                    <p style='font-size: 13px; color: #888; margin: 5px 0 0;'>
                        Do not share this code with anyone. Aletheia University will never ask for your verification code.
                    </p>
                </div>
                
                <p style='font-size: 14px; color: #666;'>
                    If you did not request this code, please ignore this email or contact our support team immediately.
                </p>
            </div>
            
            <div style='background-color: #f5f5f5; padding: 20px; text-align: center; font-size: 12px; color: #666; border-top: 1px solid #e0e0e0;'>
                <p style='margin: 0 0 5px;'>
                    <strong>Aletheia University - Campus Services</strong><br>
                    Vendor Portal System
                </p>
                <p style='margin: 5px 0;'>
                    Email: procurement@aletheiauniversity.edu.ng<br>
                    Hours: Monday-Friday, 9:00 AM - 5:00 PM
                </p>
                <p style='margin: 10px 0 0; font-size: 11px; color: #999;'>
                    © " . date('Y') . " Aletheia University. All rights reserved.
                </p>
            </div>
        </div>
        ";
        
        $mail->AltBody = "ALETHEIA UNIVERSITY - CAMPUS SERVICES\n" .
            "Vendor Portal Login Verification\n\n" .
            "Hello {$vendor['business_name']},\n\n" .
            "You have requested to login to your vendor account. Use the verification code below:\n\n" .
            "Verification Code: $verification_code\n\n" .
            "This code will expire in 10 minutes. Do not share this code with anyone.\n\n" .
            "If you did not request this code, please ignore this email or contact our support team immediately.\n\n" .
            "Best regards,\n" .
            "Aletheia University Vendor Portal Team\n" .
            "procurement@aletheiauniversity.edu.ng\n\n" .
            "© " . date('Y') . " Aletheia University. All rights reserved.";
        
        $mail->send();
        
        // Store vendor ID in session for verification
        $_SESSION['2fa_vendor_id'] = $vendor['id'];
        $_SESSION['2fa_remember_me'] = $remember_me;
        $_SESSION['2fa_email'] = $email;
        $_SESSION['2fa_created_at'] = time();
        $_SESSION['2fa_table'] = 'vendor_verification_codes';
        
        $response['success'] = true;
        $response['message'] = 'Verification code sent to your email!';
        $response['verification_required'] = true;
        $response['redirect'] = 'vendor_verify_code.php';
        
    } catch (Exception $e) {
        error_log("Email error: " . $e->getMessage());
        $response['message'] = 'Failed to send verification code. Please try again.';
        $response['errors']['email'] = 'Email service temporarily unavailable';
    }

} catch (Exception $e) {
    error_log("Login error: " . $e->getMessage());
    $response['message'] = 'An error occurred during login. Please try again.';
}

// Ensure we output valid JSON
echo json_encode($response);
exit;
?>