<?php
// process_vendor_verification.php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require '../config/conn.php';

// Set JSON header
header('Content-Type: application/json');

// Check if 2FA session exists
if (!isset($_SESSION['2fa_vendor_id'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Session expired. Please login again.',
        'redirect' => 'vendor-login.php'
    ]);
    exit();
}

// Check if verification code is provided
if (!isset($_POST['verification_code']) || empty($_POST['verification_code'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Please enter verification code.',
        'errors' => ['code' => 'Verification code is required']
    ]);
    exit();
}

$verification_code = trim($_POST['verification_code']);
$vendor_id = $_SESSION['2fa_vendor_id'];
$email = $_SESSION['2fa_email'] ?? '';

// Validate code format
if (!preg_match('/^\d{6}$/', $verification_code)) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid code format. Please enter 6 digits.',
        'errors' => ['code' => 'Invalid format'],
        'clear_inputs' => true
    ]);
    exit();
}



// Check for valid, unused code
$query = "SELECT id, expires_at, used 
          FROM vendor_verification_codes 
          WHERE vendor_id = ? AND code = ? AND used = 0
          ORDER BY created_at DESC LIMIT 1";
$stmt = $conn->prepare($query);
$stmt->bind_param("is", $vendor_id, $verification_code);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // Invalid code - also check if any code exists for this vendor
    $check_any_code = $conn->prepare("SELECT id FROM vendor_verification_codes WHERE vendor_id = ? AND used = 0 LIMIT 1");
    $check_any_code->bind_param("i", $vendor_id);
    $check_any_code->execute();
    $any_code_result = $check_any_code->get_result();
    
    if ($any_code_result->num_rows === 0) {
        // No codes exist - maybe expired or never generated
        echo json_encode([
            'success' => false,
            'message' => 'No verification code found. Please request a new one.',
            'redirect' => 'resend_vendor_verification.php?action=resend',
            'clear_inputs' => true
        ]);
    } else {
        // Code exists but doesn't match
        echo json_encode([
            'success' => false,
            'message' => 'Invalid verification code. Please check and try again.',
            'errors' => ['code' => 'Invalid code'],
            'clear_inputs' => true
        ]);
    }
    exit();
}

$code_data = $result->fetch_assoc();

// Check if code has expired
$current_time = time();
$expiry_time = strtotime($code_data['expires_at']);

if ($current_time > $expiry_time) {
    // Mark as expired
    $update_stmt = $conn->prepare("UPDATE vendor_verification_codes SET used = 1 WHERE id = ?");
    $update_stmt->bind_param("i", $code_data['id']);
    $update_stmt->execute();
    
    echo json_encode([
        'success' => false,
        'message' => 'Verification code has expired. Please request a new one.',
        'redirect' => 'resend_vendor_verification.php?action=resend',
        'clear_inputs' => true
    ]);
    exit();
}

// Code is valid - complete login
$used_at = date('Y-m-d H:i:s');

// Mark code as used
$update_stmt = $conn->prepare("UPDATE vendor_verification_codes SET used = 1, used_at = ? WHERE id = ?");
$update_stmt->bind_param("si", $used_at, $code_data['id']);
$update_stmt->execute();

// Get vendor details
$vendor_query = $conn->prepare("SELECT id, vendor_id, business_name, email, contact_person, account_status, status FROM vendors WHERE id = ?");
$vendor_query->bind_param("i", $vendor_id);
$vendor_query->execute();
$vendor_result = $vendor_query->get_result();

if ($vendor_result->num_rows === 0) {
    echo json_encode([
        'success' => false,
        'message' => 'Vendor account not found.',
        'redirect' => 'vendor-login.php'
    ]);
    exit();
}

$vendor = $vendor_result->fetch_assoc();

// Check if vendor account is active
if ($vendor['account_status'] !== 'active' || $vendor['status'] !== 'active') {
    echo json_encode([
        'success' => false,
        'message' => 'Your account is not active. Please contact support.',
        'redirect' => 'vendor-login.php'
    ]);
    exit();
}

// Set session variables for logged in vendor
$_SESSION['vendor_logged_in'] = true;
$_SESSION['vendor_id'] = $vendor['id'];
$_SESSION['vendor_uid'] = $vendor['vendor_id'];
$_SESSION['vendor_name'] = $vendor['business_name'];
$_SESSION['vendor_email'] = $vendor['email'];
$_SESSION['vendor_contact'] = $vendor['contact_person'];
$_SESSION['last_activity'] = time();

// Update last login
$update_login = $conn->prepare("UPDATE vendors SET last_login = NOW() WHERE id = ?");
$update_login->bind_param("i", $vendor_id);
$update_login->execute();

// Clear 2FA session
unset($_SESSION['2fa_vendor_id']);
unset($_SESSION['2fa_email']);
unset($_SESSION['2fa_created_at']);
unset($_SESSION['2fa_remember_me']);
unset($_SESSION['2fa_table']);

// Log successful verification
error_log("Vendor verification successful - Vendor ID: {$vendor['id']}, Email: {$vendor['email']}");

echo json_encode([
    'success' => true,
    'message' => 'Verification successful! Logging you in...',
    'redirect' => 'index.php'
]);
exit();
?>