<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
require '../config/conn.php';

// Check if token and email are provided
$token = $_GET['token'] ?? '';
$email = $_GET['email'] ?? '';

if (empty($token) || empty($email)) {
    header("Location: vendor-login.php?error=invalid_reset_link");
    exit();
}

// Validate token
$stmt = $conn->prepare("
    SELECT pt.id, pt.vendor_id, pt.expires_at, v.business_name, v.email 
    FROM password_reset_tokens pt 
    JOIN vendors v ON pt.vendor_id = v.id 
    WHERE pt.token = ? 
    AND v.email = ? 
    AND pt.used = 0 
    AND pt.expires_at > NOW()
    LIMIT 1
");

$stmt->bind_param("ss", $token, $email);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header("Location: vendor-login.php?error=invalid_or_expired_token");
    exit();
}

$data = $result->fetch_assoc();
$token_id = $data['id'];
$vendor_id = $data['vendor_id'];
$business_name = $data['business_name'];

$stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ALETHEIA UNIVERSITY - Reset Password</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Poppins', sans-serif; }
        .password-requirements {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 10px;
        }
        .requirement {
            font-size: 13px;
            margin-bottom: 5px;
        }
        .requirement.valid {
            color: #10b981;
        }
        .requirement.invalid {
            color: #ef4444;
        }
    </style>
</head>
<body class="bg-gray-100 flex items-center justify-center p-5">
    <div class="w-full max-w-md">
        <!-- Header -->
        <div class="text-center mb-8">
            <div class="w-16 h-16 mx-auto mb-3 flex items-center justify-center bg-white rounded-lg shadow-sm">
                <i class="fas fa-university text-2xl text-[#fbbb01]"></i>
            </div>
            <h1 class="text-2xl font-bold text-gray-800">ALETHEIA UNIVERSITY</h1>
            <p class="text-gray-600 text-sm mt-1">Campus Services Vendor Portal</p>
            <div class="mt-4">
                <h2 class="text-xl font-semibold text-gray-800">Reset Your Password</h2>
                <p class="text-gray-500 text-sm mt-1">for <?php echo htmlspecialchars($business_name); ?></p>
            </div>
        </div>

        <!-- Messages -->
        <div id="messageContainer" class="mb-4"></div>

        <!-- Reset Password Form -->
        <div class="bg-white p-6 rounded-lg shadow-sm">
            <form id="resetForm" method="POST">
                <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">
                <input type="hidden" name="email" value="<?php echo htmlspecialchars($email); ?>">
                
                <div class="space-y-4">
                    <div>
                        <label for="new_password" class="block text-sm font-medium text-gray-700 mb-1">
                            New Password
                        </label>
                        <div class="relative">
                            <input 
                                type="password" 
                                id="new_password" 
                                name="new_password"
                                class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                                placeholder="Enter new password"
                                required>
                            <button type="button" class="absolute right-3 top-3 text-gray-400" onclick="togglePassword('new_password')">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <small class="text-red-500 text-sm mt-1 block" id="password_error"></small>
                        
                        <!-- Password Requirements -->
                        <div class="password-requirements mt-2">
                            <p class="text-sm font-medium text-gray-700 mb-2">Password must contain:</p>
                            <div class="requirement invalid" id="req-length">• At least 8 characters</div>
                            <div class="requirement invalid" id="req-uppercase">• At least one uppercase letter</div>
                            <div class="requirement invalid" id="req-lowercase">• At least one lowercase letter</div>
                            <div class="requirement invalid" id="req-number">• At least one number</div>
                            <div class="requirement invalid" id="req-special">• At least one special character</div>
                        </div>
                    </div>
                    
                    <div>
                        <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-1">
                            Confirm New Password
                        </label>
                        <div class="relative">
                            <input 
                                type="password" 
                                id="confirm_password" 
                                name="confirm_password"
                                class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[#fbbb01] focus:border-[#fbbb01]"
                                placeholder="Confirm new password"
                                required>
                            <button type="button" class="absolute right-3 top-3 text-gray-400" onclick="togglePassword('confirm_password')">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                        <small class="text-red-500 text-sm mt-1 block" id="confirm_error"></small>
                    </div>
                    
                    <button 
                        type="submit" 
                        id="resetBtn"
                        class="w-full py-3 bg-[#fbbb01] text-white rounded-lg font-medium hover:bg-[#e0a800] transition disabled:opacity-70 disabled:cursor-not-allowed relative">
                        <span id="btnText">Reset Password</span>
                        <span id="loadingSpinner" class="hidden absolute inset-0 flex items-center justify-center">
                            <i class="fas fa-spinner fa-spin"></i>
                        </span>
                    </button>
                    
                    <div class="text-center mt-4">
                        <a href="vendor_login.php" class="text-[#fbbb01] hover:text-[#e0a800] text-sm font-medium">
                            <i class="fas fa-arrow-left mr-1"></i>Back to Login
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const newPassword = document.getElementById('new_password');
        const confirmPassword = document.getElementById('confirm_password');
        const resetForm = document.getElementById('resetForm');
        const resetBtn = document.getElementById('resetBtn');
        const btnText = document.getElementById('btnText');
        const spinner = document.getElementById('loadingSpinner');
        
        // Password validation
        newPassword.addEventListener('input', validatePassword);
        confirmPassword.addEventListener('input', validateConfirmPassword);
        
        resetForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            if (!validatePassword() || !validateConfirmPassword()) {
                return;
            }
            
            resetBtn.disabled = true;
            btnText.textContent = 'Processing...';
            spinner.classList.remove('hidden');
            
            const formData = new FormData(resetForm);
            
            try {
                const response = await fetch('process_vendor_password_reset.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                resetBtn.disabled = false;
                btnText.textContent = 'Reset Password';
                spinner.classList.add('hidden');
                
                if (data.success) {
                    showMessage(data.message, 'success');
                    setTimeout(() => {
                        window.location.href = 'vendor_login.php';
                    }, 3000);
                } else {
                    showMessage(data.message, 'error');
                }
            } catch (error) {
                console.error('Error:', error);
                resetBtn.disabled = false;
                btnText.textContent = 'Reset Password';
                spinner.classList.add('hidden');
                showMessage('Network error. Please try again.', 'error');
            }
        });
        
        function validatePassword() {
            const password = newPassword.value;
            let isValid = true;
            
            // Length requirement
            const lengthReq = document.getElementById('req-length');
            if (password.length >= 8) {
                lengthReq.classList.remove('invalid');
                lengthReq.classList.add('valid');
            } else {
                lengthReq.classList.remove('valid');
                lengthReq.classList.add('invalid');
                isValid = false;
            }
            
            // Uppercase requirement
            const upperReq = document.getElementById('req-uppercase');
            if (/[A-Z]/.test(password)) {
                upperReq.classList.remove('invalid');
                upperReq.classList.add('valid');
            } else {
                upperReq.classList.remove('valid');
                upperReq.classList.add('invalid');
                isValid = false;
            }
            
            // Lowercase requirement
            const lowerReq = document.getElementById('req-lowercase');
            if (/[a-z]/.test(password)) {
                lowerReq.classList.remove('invalid');
                lowerReq.classList.add('valid');
            } else {
                lowerReq.classList.remove('valid');
                lowerReq.classList.add('invalid');
                isValid = false;
            }
            
            // Number requirement
            const numberReq = document.getElementById('req-number');
            if (/[0-9]/.test(password)) {
                numberReq.classList.remove('invalid');
                numberReq.classList.add('valid');
            } else {
                numberReq.classList.remove('valid');
                numberReq.classList.add('invalid');
                isValid = false;
            }
            
            // Special character requirement
            const specialReq = document.getElementById('req-special');
            if (/[!@#$%^&*(),.?":{}|<>]/.test(password)) {
                specialReq.classList.remove('invalid');
                specialReq.classList.add('valid');
            } else {
                specialReq.classList.remove('valid');
                specialReq.classList.add('invalid');
                isValid = false;
            }
            
            return isValid;
        }
        
        function validateConfirmPassword() {
            const password = newPassword.value;
            const confirm = confirmPassword.value;
            
            if (confirm !== password) {
                document.getElementById('confirm_error').textContent = 'Passwords do not match';
                confirmPassword.classList.add('border-red-300');
                return false;
            } else {
                document.getElementById('confirm_error').textContent = '';
                confirmPassword.classList.remove('border-red-300');
                return true;
            }
        }
        
        function showMessage(message, type) {
            const messageContainer = document.getElementById('messageContainer');
            const div = document.createElement('div');
            div.className = `p-4 rounded-lg border mb-3 ${
                type === 'success' 
                ? 'bg-green-50 border-green-200 text-green-700' 
                : 'bg-red-50 border-red-200 text-red-700'
            }`;
            div.innerHTML = `
                <div class="flex items-center">
                    <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
                    <span>${message}</span>
                </div>
            `;
            messageContainer.appendChild(div);
        }
        
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
    });
    </script>
</body>
</html>