<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);
require '../config/conn.php';

// Set PHP timezone - CRITICAL FIX
date_default_timezone_set('Africa/Lagos'); // Match your server's timezone

// Redirect if no 2FA session
if (!isset($_SESSION['2fa_vendor_id'])) {
    header("Location: vendor-login.php");
    exit();
}

// Check if verification code is expired
if (isset($_SESSION['2fa_created_at'])) {
    $expiry_time = strtotime('+10 minutes', $_SESSION['2fa_created_at']);
    if (time() > $expiry_time) {
        session_destroy();
        header("Location: vendor-login.php?error=code_expired");
        exit();
    }
}

// Store creation time if not set
if (!isset($_SESSION['2fa_created_at'])) {
    $_SESSION['2fa_created_at'] = time();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Campus Services - Verification</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        body { font-family: 'Poppins', sans-serif; }
        .code-input {
            width: 45px;
            height: 55px;
            text-align: center;
            font-size: 22px;
            font-weight: bold;
            border: 2px solid #d1d5db;
            border-radius: 8px;
            margin: 0 5px;
            background-color: white;
            color: #1f2937;
            transition: all 0.2s ease;
        }
        .code-input:focus {
            border-color: #fbbb01;
            box-shadow: 0 0 0 2px rgba(251, 187, 1, 0.2);
            outline: none;
        }
        .code-input.filled {
            border-color: #fbbb01;
            background-color: #fefce8;
        }
        .countdown-timer {
            font-family: 'Courier New', monospace;
            font-weight: bold;
            color: #1f2937;
            font-size: 16px;
        }
    </style>
</head>
<body class="bg-gray-100 flex items-center justify-center p-4">
    <div class="w-full max-w-sm">
        <!-- Logo and Header -->
        <div class="text-center mb-6">
            <div class="w-16 h-16 mx-auto mb-3 flex items-center justify-center">
                <img src="../img/schools.png" alt="School Logo" class="max-w-full max-h-full">
            </div>
            <h1 class="text-xl font-bold text-gray-800">Campus Services</h1>
            <p class="text-gray-500 text-sm mt-1">Vendor Verification</p>
            <p class="text-xs text-gray-400 mt-2">
                <i class="fas fa-envelope mr-1"></i>
                <?php echo htmlspecialchars($_SESSION['2fa_email'] ?? ''); ?>
            </p>
        </div>

        <!-- Messages -->
        <div id="messageContainer" class="mb-4"></div>

        <!-- Verification Form -->
        <div class="bg-white p-6">
            <form id="verifyForm" method="POST">
                <!-- Code Input -->
                <div class="mb-5">
                    <label class="block text-gray-700 text-sm font-medium mb-3 text-center">Enter verification code</label>
                    <div class="flex justify-center mb-4" id="codeInputs">
                        <input type="text" maxlength="1" class="code-input" data-index="1" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                        <input type="text" maxlength="1" class="code-input" data-index="2" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                        <input type="text" maxlength="1" class="code-input" data-index="3" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                        <input type="text" maxlength="1" class="code-input" data-index="4" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                        <input type="text" maxlength="1" class="code-input" data-index="5" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                        <input type="text" maxlength="1" class="code-input" data-index="6" oninput="moveToNext(this)" onkeydown="handleBackspace(event, this)">
                    </div>
                    <input type="hidden" id="verificationCode" name="verification_code">
                    <small class="text-red-500 text-xs mt-1 block text-center" id="code_error"></small>
                </div>
                
                <!-- Countdown Timer -->
                <div class="text-center mb-5 text-xs text-gray-600">
                    <p>
                        <i class="fas fa-clock mr-1"></i>
                        Code expires in: <span id="countdown" class="countdown-timer font-mono">10:00</span>
                    </p>
                </div>
                
                <!-- Verify Button -->
                <button type="submit" id="submitBtn" 
                        class="w-full py-2.5 bg-[#fbbb01] text-white text-sm font-medium rounded-lg hover:bg-[#e0a800] transition disabled:opacity-70 disabled:cursor-not-allowed">
                    <span id="btnText">Verify & Login</span>
                    <span id="loadingSpinner" class="hidden ml-2">
                        <i class="fas fa-spinner fa-spin"></i>
                    </span>
                </button>
                
                <!-- Resend Code -->
                <div class="text-center mt-4 pt-4 border-t border-gray-200">
                    <p class="text-gray-500 text-xs mb-2">Didn't receive the code?</p>
                    <button type="button" id="resendBtn" 
                            class="text-[#fbbb01] hover:text-[#e0a800] text-xs font-medium disabled:opacity-50 disabled:cursor-not-allowed">
                        <i class="fas fa-redo-alt mr-1"></i>
                        <span id="resendText">Resend verification code</span>
                        <span id="resendTimer" class="hidden ml-1">(60s)</span>
                    </button>
                </div>
                
                <!-- Back to Login -->
                <div class="text-center mt-4">
                    <a href="vendor_login.php" class="text-gray-500 text-xs hover:text-gray-700">
                        <i class="fas fa-arrow-left mr-1"></i>Back to Login
                    </a>
                </div>
            </form>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const codeInputs = document.querySelectorAll('.code-input');
        const hiddenInput = document.getElementById('verificationCode');
        const verifyForm = document.getElementById('verifyForm');
        const submitBtn = document.getElementById('submitBtn');
        const btnText = document.getElementById('btnText');
        const spinner = document.getElementById('loadingSpinner');
        const resendBtn = document.getElementById('resendBtn');
        const resendText = document.getElementById('resendText');
        const resendTimer = document.getElementById('resendTimer');
        const messageContainer = document.getElementById('messageContainer');
        const countdownElement = document.getElementById('countdown');
        
        let countdown = 600; // 10 minutes in seconds
        let resendCountdown = 60; // 60 seconds wait for resend
        let canResend = false;

        // Start countdown timer
        const countdownInterval = setInterval(updateCountdown, 1000);

        function updateCountdown() {
            if (countdown <= 0) {
                clearInterval(countdownInterval);
                showMessage('Verification code has expired. Please request a new one.', 'error');
                submitBtn.disabled = true;
                codeInputs.forEach(input => input.disabled = true);
                resendBtn.disabled = false;
                canResend = true;
                return;
            }

            const minutes = Math.floor(countdown / 60);
            const seconds = countdown % 60;
            countdownElement.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            
            // Change color when less than 1 minute
            if (countdown <= 60) {
                countdownElement.style.color = '#ef4444';
            }
            
            countdown--;
        }

        // Update hidden input when code inputs change
        codeInputs.forEach(input => {
            input.addEventListener('input', function() {
                updateHiddenInput();
                // Update visual state
                if (this.value) {
                    this.classList.add('filled');
                } else {
                    this.classList.remove('filled');
                }
            });
        });

        function updateHiddenInput() {
            const code = Array.from(codeInputs).map(input => input.value).join('');
            hiddenInput.value = code;

            // Auto-submit when all digits are entered
            if (code.length === 6) {
                setTimeout(() => {
                    verifyForm.dispatchEvent(new Event('submit'));
                }, 100);
            }
        }

        // Form submission - FIXED VERSION
        verifyForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const code = hiddenInput.value;
            if (code.length !== 6) {
                showMessage('Please enter the complete 6-digit code.', 'error');
                return;
            }

            submitBtn.disabled = true;
            btnText.textContent = 'Verifying...';
            spinner.classList.remove('hidden');
            
            // Clear messages
            messageContainer.innerHTML = '';
            document.getElementById('code_error').textContent = '';
            
            try {
                // Use FormData for proper form submission
                const formData = new FormData();
                formData.append('verification_code', code);
                
                // IMPORTANT: Add credentials for session cookies
                const response = await fetch('process_vendor_verification.php', {
                    method: 'POST',
                    body: formData,
                    credentials: 'same-origin' // This preserves session
                });
                
                const data = await response.json();
                
                submitBtn.disabled = false;
                btnText.textContent = 'Verify & Login';
                spinner.classList.add('hidden');
                
                if (data.success) {
                    showMessage(data.message, 'success');
                    
                    if (data.redirect) {
                        setTimeout(() => {
                            window.location.href = data.redirect;
                        }, 800);
                    }
                } else {
                    showMessage(data.message, 'error');
                    
                    if (data.errors && data.errors.code) {
                        document.getElementById('code_error').textContent = data.errors.code;
                    }
                    
                    // Clear inputs on error
                    if (data.clear_inputs) {
                        codeInputs.forEach(input => {
                            input.value = '';
                            input.classList.remove('filled');
                        });
                        hiddenInput.value = '';
                        codeInputs[0].focus();
                    }
                }
            } catch (error) {
                console.error('Error:', error);
                submitBtn.disabled = false;
                btnText.textContent = 'Verify & Login';
                spinner.classList.add('hidden');
                showMessage('Network error. Try again.', 'error');
            }
        });

        // Resend code functionality - FIXED VERSION
        resendBtn.addEventListener('click', async function() {
            if (resendBtn.disabled) return;
            
            resendBtn.disabled = true;
            resendText.textContent = 'Sending...';
            
            try {
                // Use proper FormData
                const formData = new FormData();
                formData.append('action', 'resend_code');
                
                const response = await fetch('resend_vendor_verification.php', {
                    method: 'POST',
                    body: formData,
                    credentials: 'same-origin' // Important for session
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage('New verification code sent! Please check your email.', 'success');
                    
                    // Reset countdown
                    countdown = 600;
                    updateCountdown();
                    
                    // Start resend cooldown
                    resendCountdown = 60;
                    resendText.classList.add('hidden');
                    resendTimer.classList.remove('hidden');
                    resendBtn.disabled = true;
                    
                    const resendInterval = setInterval(() => {
                        resendCountdown--;
                        resendTimer.textContent = `(${resendCountdown}s)`;
                        
                        if (resendCountdown <= 0) {
                            clearInterval(resendInterval);
                            resendText.textContent = 'Resend verification code';
                            resendText.classList.remove('hidden');
                            resendTimer.classList.add('hidden');
                            resendBtn.disabled = false;
                        }
                    }, 1000);
                    
                    // Clear inputs
                    codeInputs.forEach(input => {
                        input.value = '';
                        input.classList.remove('filled');
                    });
                    hiddenInput.value = '';
                    codeInputs[0].focus();
                    
                } else {
                    showMessage(data.message || 'Failed to resend code.', 'error');
                    resendBtn.disabled = false;
                    resendText.textContent = 'Resend verification code';
                }
            } catch (error) {
                console.error('Error:', error);
                showMessage('Failed to resend code. Please try again.', 'error');
                resendBtn.disabled = false;
                resendText.textContent = 'Resend verification code';
            }
        });

        function showMessage(message, type) {
            const colors = {
                success: 'bg-green-50 text-green-700 border border-green-200',
                error: 'bg-red-50 text-red-700 border border-red-200',
                info: 'bg-blue-50 text-blue-700 border border-blue-200'
            };
            
            const div = document.createElement('div');
            div.className = `p-3 rounded-lg text-sm mb-3 ${colors[type] || colors.info}`;
            div.innerHTML = `
                <div class="flex items-center">
                    <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-2"></i>
                    <span>${message}</span>
                </div>
            `;
            
            messageContainer.prepend(div);
            setTimeout(() => div.remove(), 4000);
        }

        // Focus on first input
        setTimeout(() => {
            codeInputs[0].focus();
        }, 300);
    });

    function moveToNext(input) {
        const index = parseInt(input.getAttribute('data-index'));
        const value = input.value;
        
        // Only allow numbers
        if (value && !/^\d$/.test(value)) {
            input.value = '';
            return;
        }
        
        if (input.value && index < 6) {
            document.querySelector(`.code-input[data-index="${index + 1}"]`).focus();
        }
    }

    function handleBackspace(event, input) {
        const index = parseInt(input.getAttribute('data-index'));
        if (event.key === 'Backspace') {
            if (!input.value && index > 1) {
                event.preventDefault();
                document.querySelector(`.code-input[data-index="${index - 1}"]`).focus();
            } else if (input.value) {
                input.value = '';
                input.classList.remove('filled');
                updateHiddenInput();
            }
        }
        
        // Handle left/right arrow keys
        if (event.key === 'ArrowLeft' && index > 1) {
            event.preventDefault();
            document.querySelector(`.code-input[data-index="${index - 1}"]`).focus();
        }
        if (event.key === 'ArrowRight' && index < 6) {
            event.preventDefault();
            document.querySelector(`.code-input[data-index="${index + 1}"]`).focus();
        }
    }
    </script>
</body>
</html>